package org.apache.james.userrepository;

import org.apache.commons.lang.StringUtils;
import org.apache.james.security.DigestUtil;

import java.util.StringTokenizer;
import java.security.NoSuchAlgorithmException;

/** Useful class to to handle cleartext and hashed passwords.
 *
 * @author Alexander Zhukov <zhukov@ukrpost.net>
 */
public class Password {
    public Password(String pass) {
        this(pass, null, true);
    }

    public String getClearText() {
        return clearText;
    }

    public void setClearText(String clearText) {
        this.clearText = clearText;
    }

    private String clearText = null;

    public Password(String hashed, String algorithm, boolean detectAlgorithm) {
        this.hashed = stripAlgorithm(hashed);
        this.algorithm = algorithm;
        if ( algorithm == null || detectAlgorithm )
            this.algorithm = detectAlgorithm(hashed);

        if ( "clear".equalsIgnoreCase(this.algorithm) )
            setClearText(hashed);

    }

    private static final String stripAlgorithm(String pass) {
        if ( pass == null )
            return null;
        if ( pass.indexOf('}') == -1 )
            return pass;

        return pass.substring(pass.indexOf('}')+1);
    }
    private static final String detectAlgorithm(String pass) {
        if ( pass == null )
            return null;
        if ( pass.indexOf('}') == -1 )
            return "clear";

        return pass.substring(1, pass.indexOf('}'));
    }

    public String getHashed() {
        try {
            return getHashed(getAlgorithm());
        } catch (NoSuchAlgorithmException e) {
            return hashed;
        }
    }

    public String getHashed(String alg) throws NoSuchAlgorithmException {
        if ( getClearText() != null )
            return DigestUtil.digestString(getClearText(), alg);
        if ( alg.equalsIgnoreCase(getAlgorithm()) )
            return hashed;
        throw new NoSuchAlgorithmException(alg);
    }

    public void setHashed(String hashed) {
        this.hashed = hashed;
    }

    public String getAlgorithm() {
        return algorithm;
    }

    public void setAlgorithm(String algorithm) {
        this.algorithm = algorithm;
    }

    private String hashed = null;
    private String algorithm = null;

    public boolean equals(Object o) {
        if ( o instanceof String ) {
            if ( getClearText() != null )
                return StringUtils.equals(getClearText(), (String)o);
            try {
                String hashedCandidate = DigestUtil.digestString((String)o, getAlgorithm());
                return getHashed().equals(hashedCandidate);
            } catch (NoSuchAlgorithmException e) {
                e.printStackTrace(System.out);
                return false;
            }
        }
        if ( !(o instanceof Password) )
            return false;
        Password p = (Password) o;
        if ( StringUtils.equalsIgnoreCase(p.getAlgorithm(), getAlgorithm())
            || getHashed() == null )
            return false;

        return getHashed().equals(p.getHashed());
    }

}
