#!/usr/bin/bash

# Provides a tool to "apply" a review from Github.

# Change this if you want to apply patches from  other organisations than default "apache"
REPO="apache"

JSONURL=$(dirname ${0})/jsonurl.py
JSONARRAY=$(dirname ${0})/jsonarray.py

function usage {
cat <<EOF
Generic tool to apply Github pull request in SVN.
It can be used to integrate Github in the workflow of your Apache project.

Usage: $0 [-h] [-f] [-n] <Project> <Number of the pull request>
  -h   Print this help message and exit
  -f   Force to apply patchs, even if the pull request is closed, or not mergeable.
  -n   Don't close pull request

Default Github targeted repository used is "apache"
EOF
}

CLOSE_PULL_REQUEST=true
FORCE=false

while getopts ":nhrg" opt; do
  case $opt in
    n)
      CLOSE_PULL_REQUEST=false
      ;;
    f)
      FORCE=false
      ;;
    h)
      usage
      exit 0
      ;;
    *)
      echo "Unknown option: -$OPTARG"
      usage
      exit 1
      ;;
  esac
done


shift $(($OPTIND - 1))

if test ${#} -ne 2; then
  usage
  exit 1
fi

read -p "Github login : " GITHUB_LOGIN
read -s -p "Github password : " GITHUB_PASSWORD
echo

export SYS_GITHUB_LOGIN=$GITHUB_LOGIN
export SYS_GITHUB_PASSWORD=$GITHUB_PASSWORD

PROJECT=${1}
REVIEW=${2}

GITHUB_BASE_URL="https://github.com/${REPO}/${PROJECT}"
GITHUB_URL="${GITHUB_BASE_URL}/pull"
PATCH_BASE="${GITHUB_BASE_URL}/commit"

API_URL_BASE="https://api.github.com/repos/${REPO}/${PROJECT}"
PULL_REQUEST_URL="${API_URL_BASE}/pulls/${REVIEW}"
COMMITS_URL="${PULL_REQUEST_URL}/commits"
COMMIT_BASE_URL="${API_URL_BASE}/commits"


# Ensure we can merge this PR ( there should be no conflicts )
MERGEABLE=$(${JSONURL} ${PULL_REQUEST_URL} ${GITHUB_LOGIN} ${GITHUB_PASSWORD} mergeable)
STATE=$(${JSONURL} ${PULL_REQUEST_URL} ${GITHUB_LOGIN} ${GITHUB_PASSWORD} state)
PERFORM_MERGE=false
if $FORCE;
then
    PERFORM_MERGE=true
fi
if [ "$MERGEABLE" == "True" ];
then
    if [ "$STATE" == "open" ];
    then
        PERFORM_MERGE=true
    fi
fi

if $PERFORM_MERGE ; 
then 
    for SHA in $(${JSONARRAY} ${COMMITS_URL} ${GITHUB_LOGIN} ${GITHUB_PASSWORD})
    do
        COMMIT_MESSAGE=$(${JSONURL} "${COMMIT_BASE_URL}/${SHA}" ${GITHUB_LOGIN} ${GITHUB_PASSWORD} commit message)
        curl -i "${PATCH_BASE}/${SHA}.patch" > ${SHA}.patch
        AUTHOR=$(head -n 100 ${SHA}.patch | grep "From: " | cut -d ' ' -f3-)
        echo
        echo "========================"
        echo "Managing ${SHA}"
        echo "${AUTHOR}"
        echo "${COMMIT_MESSAGE}"
        echo "========================"
        echo
        svn patch "${SHA}.patch"
        RETVAL=$?
        [ $RETVAL -eq 0 ] && echo "--------------Patch applied----------------"
        [ $RETVAL -ne 0 ] && echo "Patch can not be applied for ${SHA} : ${COMMIT_MESSAGE}" && exit 1;
        svn commit -m "${COMMIT_MESSAGE}. Contributed by ${AUTHOR}"
        RETVAL=$?
        [ $RETVAL -eq 0 ] && echo "--------------Changes commited on SVN applied----------------"
        [ $RETVAL -ne 0 ] && echo "Changes can not be commited on SVN for ${SHA} : ${COMMIT_MESSAGE}" && exit 1;
        svn update
        rm "${SHA}.patch"
    done
    if $CLOSE_PULL_REQUEST; then
        curl -XPOST -u "${GITHUB_LOGIN}:${GITHUB_PASSWORD}" ${PULL_REQUEST_URL} -d '{"state":"closed"}'
        echo "---------------------"
        echo "Closing pull request ${PROJECT} / ${REVIEW}"
        echo "---------------------"
    fi
    exit 0;
fi

echo "The PR ${REVIEW} for ${PROJECT} is not meageable."
exit 1;
