/*
 * $Header$
 * $Revision$
 * $Date$
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 * This source code was written by Richard Unger while in employ of
 * Camino IT Consulting GMBH, Vienna, Austria (Camino).
 * The source code was written with the specific intent of contributing
 * it to the Apache Software Foundation Jakarta-Slide open source project.
 * The copyright of the source code is Richard Unger's, the source is released
 * under the Apache Software License v1.1, and Camino can use the source code
 * under the terms of this license.
 * An agreement specifying these conditions between Richard Unger and Camino exists.
 *
 */




package org.apache.slide.security;



import java.security.Principal;

import java.util.HashSet;
import java.util.HashMap;




/**
 * This class represents a slide user, and encapsulates all interesting
 * information about such a user...
 * This class can be passed about and used to retrieve information about the user, as well
 * as store information for the user.
 * It is up to the programmer to ensure this object is 'fresh'. In other words, changes saved to the userdatabase
 * are not immediately visible in this object. The user should be reloaded freshly from the Userdatabase whenever
 * a change could have occured.
 * In the same way, changes to this object are not automatically saved to the userdatabase. It is up to the programmer
 * to call the userdatabase.saveUser() method to save any changes back to the database.
 *
 * Some rules about slide users:
 *
 * Usernames must be unique within a userdatabase
 * Usernames cannot be changed after saving in the database (although the user can be deleted and recreated 
 *  with a different name)
 * Usernames are alphanumeric only, case sensitive, and must begin with a letter.
 *
 *
 * @author Richard Unger, runger@cim.mcgill.ca
 * @version $Revision$ $Date$
 **/
public class SlideUser implements Principal {

    /**
     * The user's username
     * This property is not mutable once the user is created
     **/
    protected String username = null;
    /**
     * The user's password
     **/
    protected String password = null;
    /**
     * The user's full name, eg: "Richard Unger"
     **/
    protected String fullname = null;
    /**
     * Other string properties the user may have
     **/
    protected HashMap props = null;
    /**
     * Roles the user has
     **/
    protected HashSet roles = null;
    /**
     * SlideUserdatabase whence we were loaded from or were created in
     **/
    protected SlideUserDatabase myDB = null;



    /**
     * Constructor for creating new users in the userdatabase
     * @param home the userdatabase the user will belong to
     * @param name the name of the user
     **/
    SlideUser(SlideUserdatabase home, String name){
	myDB = home;
	username = name;
    }



    /**
     * Get the SlideUserDatabase
     * @return the SlideUserDatabase this user came from
     **/
    public SlideUserDatabase getSlideUserDatabase(){
	return myDB;
    }



    /**
     * Return the username
     * @return the username, as a string
     **/
    public String getName(){
	return username;
    }



    /**
     * Return the users password (Careful what you do with this information!)
     * @return the password, as a plaintext string
     **/
    public String getPassword(){
	return password;
    }



    /**
     * Set the users password
     * @param password the new password
     **/
    public void setPassword(String password){
	this.password = password;
    }



    /**
     * Get the users real name
     * @return the full user name, as a String, eg: Richard Unger
     **/
    public String getFullName(){
	return fullname;
    }



    /**
     * Set the users real name
     * @param fullname the full name of the user, eg Richard Unger
     **/
    public void setFullName(String fullname){
	this.fullname = fullname;
    }



    /**
     * Get a named user property...
     * @param propName the name of the property to get
     * @return the string value of the named property
     **/
    public String getProperty(String propName){
	if ("fullname".equals(propName)) return fullname;
	if ("password".equals(propName)) return password;
	return (String)props.get(propName);
    }


    /**
     * Set a named user property
     * @param propName the name of the property to set
     * @param propValue the value to set
     **/
    public void setProperty(String propName,String propValue){
	if ("fullname".equals(propName)) { fullname = propValue; return; }
	if ("password".equals(propName)) { password = propValue; return; }
	props.put(propName,propValue);
    }



    /**
     * Check whether this user is member of the specified group
     * @param theGroup the group to check for membership
     * @return true if the user is a member of the group
     **/
    public boolean isInGroup(SlideGroup theGroup){
	return theGroup.hasMember(this);
    }



    /**
     * Check whether this user has the specified role
     * @param theRole the role to check for
     * @return true if this user has the role
     **/
    public boolean hasRole(SlideRole theRole){
	if (!myDB.equals(theRole.getSlideUserDatabase()))
	    return false;
	return hasRole(theRole.getName());
    }


    /**
     * Check whether the user has the specified role
     * @param roleName the name of the role
     * @return true if this user has the role
     **/
    public boolean hasRole(String roleName){
	return roles.contains(roleName);
    }



    /**
     * Add a role to a user
     * @param theRole the role to add
     **/
    public void addRole(SlideRole theRole) throws UserManagementException {
	if (!myDB.equals(theRole.getSlideUserDatabase()))
	    throw new UserManagementException("The role you are adding is not from the same SlideUserDatabase!");
	if (hasRole(theRole.getName())
	    throw new UserManagementException("This user already has the role you are adding!");
	roles.add(theRole.getName());
    }



    /**
     * Remove a role from a user
     * @param theRole the role to remove
     **/
    public void removeRole(SlideRole theRole) throws UserManagementException {
	if (!myDB.equals(theRole.getSlideUserDatabase()))
	    throw new UserManagementException("The role you are removing is not from the same SlideUserDatabase!");
	if (!hasRole(theRole.getName())
	    throw new UserManagementException("This user does not have the role you are removing!");
	roles.remove(theRole.getName());
    }




	//TODO: Complete me!
    public SlideToken getSlideToken(){

    }




	//TODO: Complete me!
    public CredentialsToken getCredentialsToken(){

    }




    /**
     * Return a java.security.Pricipal object for this user...
     * @return the Principal object (this object itself)
     **/
    public Principal getPrincipal(){
	return this;
    }




    /**
     * Compare this SlideUser to another
     * Two SlideUsers are the same if they come from the same Userdatabase
     * and have the same username.
     * @param o the SlideUser to compare to
     * @return true if they refer to the same slideuser, false otherwise or if o is not an instance of a SlideUser
     **/
    public boolean equals(Object o){
	if (!(o instanceof SlideUser))
	    return false;
	SlideUser other = (SlideUser)o;
	if (myDB.equals(other.getUserDatabase()) && username.equals(other.getName()))
	    return true;
	return false;
    }


}
