/*
 * $Header$
 * $Revision$
 * $Date$
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */





package org.apache.slide.security;


/**
 * SlideUserDatabase
 * This interface specifies the functionality for a SlideUserDatabase
 * The SlideUserDatabase is a slide helper class similar to the contenthelper and securityhelper
 * classes in slide. The SlideUserDatabase is used for all operations involving slide users, roles and
 * groups. This interface specifies the functionality that all slide userdatabases must offer, so that slide
 * itself can access all the information it needs internally about a user.
 * A basic implementation of the SlideUserDatabase is provided with the SlideStoreUserdatabase 
 * (@see org.apache.slide.security.SlideStoreUserDatabase ). This Userdatabase stores the users in the slide namespace,
 * as slide structure.
 * This suggests two basic methods for implementing connections to other user management systems 
 * (eg LDAP, Active Directory, NIS, etc):
 * 1) Write a new class implementing SlideUserDatabase, for example LDAPSlideUserDatabase. Set the slide config to
 * use your new Userdatabase class. This allows you to replace all aspects of user storage and retrieval.
 * 2) Write a new class implementing SlideStore and use it as the userstore for your slide namespace. 
 * This allows the standard userdatabase to store its users to your store.
 *
 * Which is better? This depends on the task at hand! For a system like LDAP, which does not cleanly map to slide's
 * file-system-like view of things, it may be better to write a new Userdatabase. For other systems, which store their
 * users in the same way, it may be easier to simply write a store class, and allow slide to serialize the users.
 *
 * In either case, the goal here is to ensure that dealing with the slide users does not involve calls to retrieve
 * slide structure, content or revisionproperties, but instead this information can be obtained cleanly by making calls
 * to java functions.
 * 
 * The objects returned from the UserDatabase are not intended for storing permanently or caching. In general, the
 * objects should not be kept except for use in the current function call or calls. They should not be saved in the
 * session or the like, as they might be changed by another user in another session.
 * Changes made to the object should be saved back to the database using the saveUser() or analogous methods.
 * 
 * There is a wrapper class (@see wrappers.catalina.SlideUserDatabaseWrapper ) which wraps any SlideUserDatabase and
 * provides a tomcat compatible API (@see org.apache.catalina.UserDatabase ). This allows tomcat to authenticate against
 * the slide users, and allows user management using the tomcat web guis...
 * @author Richard Unger
 * @version $Revision$ $Date$
 **/
public interface SlideUserDatabase {


    /**
     * Retrieve the named user
     * @param name the name of the user
     * @return the user, as a SlideUser object, or null if not found
     **/
    public SlideUser findUser(String name);


    /**
     * Retrieve the named group
     * @param name the name of the group
     * @return the group, as a SlideGroup object, or null if not found
     **/
    public SlideGroup findGroup(String name);


    /**
     * Retrieve the named role
     * @param name the name of the role
     * @return the role, as a SlideRole object, or null if not found
     **/
    public SlideRole findRole(String name);


    /**
     * Create a new user for saving in this database
     * @param name the new name
     * @return a new empty user
     **/
    public void createUser(String name) throws UserManagementException;
    

    /**
     * Create a new group for saving in this database
     * @param name the new name
     * @return a new empty group
     **/
    public SlideGroup createGroup(String name) throws UserManagementException;


    /**
     * Create a new role for saving in this database
     * @param name the new name
     * @return a new empty role
     **/
    public SlideRole createRole(String name) throws UserManagementException;


    /**
     * Save updated information about a user
     * Note: it is not possible to change a user's username
     * @param theUser the user to save
     **/
    public void saveUser(SlideUser theUser) throws UserManagementException;


    /**
     * Save a group
     * Note: it is not possible to change a group's groupname
     * @param theGroup the group to save
     **/
    public void saveGroup(SlideGroup theGroup) throws UserManagementException;


    /**
     * Save a role
     * Note: it is not possible to change the role's name
     * @param theRole the role to save
     **/
    public void saveRole(SlideRole theRole) throws UserManagementException;


    /**
     * Delete an existing user
     * @param theUser the user to delete
     **/
    public void removeUser(SlideUser theUser) throws UserManagementException;


    /**
     * Delete an existing group
     * @param theGroup the group to delete
     **/
    public void removeGroup(SlideGroup theGroup) throws UserManagementException;


    /**
     * Delete an existing role
     * @param theRole the role to delete
     **/
    public void removeRole(SlideRole theRole) throws UserManagementException;


    /**
     * Get a list of user names from this database
     * @return an iterator over a set of strings, no special order
     **/
    public Iterator userNames();


    /**
     * Get a list of group names defined in this database
     * @return an iterator over a set of strings, no special order
     **/
    public Iterator groupNames();
    

    /**
     * Get a list of role names defined in this database
     * @return an iterator over a set of strings, no special order
     **/
    public Iterator roleNames();



}
