/*
 * $Header: $
 * $Revision: 1.0 $
 * $Date:  $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Slide", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.slide.event;

import org.apache.slide.util.conf.Configurable;
import org.apache.slide.util.conf.Configuration;
import org.apache.slide.util.conf.ConfigurationException;

import java.lang.reflect.Method;
import java.lang.reflect.Field;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Event dispatcher class
 *
 * @author <a href="mailto:dflorey@c1-fse.de">Daniel Florey</a>
 * @version $Revision: 1.0 $
 */
public class EventDispatcher implements Configurable {
    private final static Logger logger = Logger.getLogger(EventDispatcher.class.getName());

    private static EventDispatcher eventDispatcher = new EventDispatcher();

    private List eventListeners = new ArrayList();

    private EventDispatcher() {
    }

    public static EventDispatcher getInstance() {
        return eventDispatcher;
    }

    public void addEventListener(EventListener listener) {
            logger.log(Level.INFO, "Registering event listener: "+listener);
            eventListeners.add(listener);
    }

    public void fireVetoableEvent(VetoableEventMethod eventMethod, EventObject event) throws VetoException {
        for (Iterator i = eventListeners.iterator(); i.hasNext(); ) {
            EventListener listener = (EventListener)i.next();
            if ( listener instanceof GlobalListener ) {
                ((GlobalListener)listener).vetoableEventFired(eventMethod, event);
            } else {
                eventMethod.fireVetaoableEvent(listener, event);
            }
        }
    }

    public void fireEvent(EventMethod eventMethod, EventObject event) {
        for (Iterator i = eventListeners.iterator(); i.hasNext(); ) {
            EventListener listener = (EventListener)i.next();
            if ( listener instanceof GlobalListener ) {
                ((GlobalListener)listener).eventFired(eventMethod, event);
            } else {
                eventMethod.fireEvent(listener, event);
            }
        }
    }

    public void configure(Configuration config) throws ConfigurationException {
        Enumeration listenerConfigs = config.getConfigurations("listener");
        while (listenerConfigs.hasMoreElements()) {
            Configuration listenerConfig = (Configuration)listenerConfigs.nextElement();
            String classname = listenerConfig.getAttribute("classname");
            try {
                Class listenerClass = Class.forName(classname);
                EventListener eventListener = null;
                try {
                    Method getInstanceMethod = listenerClass.getMethod("getInstance", new Class[0]);
                    eventListener = (EventListener)getInstanceMethod.invoke(null, null);
                } catch ( NoSuchMethodException e) {
                    eventListener = (EventListener)listenerClass.newInstance();
                }
                if ( eventListener instanceof Configurable ) {
                    ((Configurable)eventListener).configure(listenerConfig.getConfiguration("configuration"));
                }
                addEventListener(eventListener);
            } catch (ClassCastException e) {
                throw new ConfigurationException("Event listener '"+classname+"' is not of type EventListener", config);
            } catch (Exception e) {
                throw new ConfigurationException("Event listener '"+classname+"' could not be loaded", config);
            }
        }
        Enumeration enableConfigs = config.getConfigurations("event");
        while (enableConfigs.hasMoreElements()) {
            Configuration enableConfig = (Configuration)enableConfigs.nextElement();
            String classname = enableConfig.getAttribute("classname");
            String method = enableConfig.getAttribute("method", null);
            boolean enable = enableConfig.getAttributeAsBoolean("enable", true);
            try {
                Class eventClass = Class.forName(classname);
                Field methodsField = eventClass.getField("methods");
                AbstractEventMethod[] methods = (AbstractEventMethod [])methodsField.get(null);
                for ( int i = 0; i < methods.length; i++ ) {
                    if ( method == null || methods[i].getName().equals(method) ) {
                        logger.log(Level.INFO, "Enabling method '"+methods[i].getId()+"': "+enable);
                        methods[i].setEnabled(enable);
                    }
                }
            } catch (NoSuchFieldException e) {
                throw new ConfigurationException("Event '"+classname+"' does not provide the required static member 'methods'", config);
            } catch (Exception e) {
                throw new ConfigurationException("Event '"+classname+"' could not be loaded", config);
            }
        }
    }
}