package fr.aliacom.lucene.store;

import java.io.IOException;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.naming.NamingException;

import org.apache.lucene.store.Directory;
import org.apache.lucene.store.InputStream;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.OutputStream;

/**
 * @author nicolasr
 * cr le 13 juin 2003
 * 
 * Toutes les SQLException sont catchs et relances en IOException
 * 
 * TODO: p un pb si on a des fichiers de mm noms ?
 */
public class SQLDirectory extends Directory implements Cloneable {

	/** nombre d'octets du fichier stocks dans chaque enregistrement */
	public static int SEGMENT_SIZE = 1024;
	private boolean closed;
	
	public SQLDirectory(boolean create) throws IOException {
		closed = false;
		if (create) {
			create();
		}
	}

	public SQLDirectory(Map parameters) throws IOException {
		this(false);
		parameters.clear();
	}

	protected Connection getConnection()
		throws NamingException, SQLException, IOException {
		return LuceneHelper.getConnection();
	}

	/**
	 * create == vider la table
	 */
	private void create() throws IOException {
		Connection conn = null;
		PreparedStatement pst = null;
		try {
			conn = getConnection();
			pst = conn.prepareStatement("delete from LUCENE");
			pst.executeUpdate();
		} catch (SQLException e) {
			throw new IOException(e.getMessage());
		} catch (NamingException e) {
			throw new IOException(e.getMessage());
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, null);
			} catch (SQLException e1) {
				throw new IOException(e1.getMessage());
			}
		}
	}

	/**
	 * @see org.apache.lucene.store.Directory#list()
	 */
	public String[] list() throws IOException {
//		if (closed) {
//			throw new IOException("The Directory is closed");
//		}
		List result = null;
		Connection conn = null;
		PreparedStatement pst = null;
		ResultSet rs = null;
		try {
			conn = getConnection();
			pst = conn.prepareStatement("select file_name from LUCENE");
			rs = pst.executeQuery();
			result = new ArrayList();
			while (rs.next()) {
				result.add(rs.getString(1));
			}
		} catch (SQLException e) {
			throw new IOException(e.getMessage());
		} catch (NamingException e) {
			throw new IOException(e.getMessage());
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, rs);
			} catch (SQLException e) {
				throw new IOException(e.getMessage());
			}
		}

		String[] retVal = new String[result.size()];
		System.arraycopy(result.toArray(), 0, retVal, 0, result.size());
		return retVal;
	}

	/**
	 * @see org.apache.lucene.store.Directory#fileExists(java.lang.String)
	 */
	public boolean fileExists(String arg0) throws IOException {
		boolean present = false;
		Connection conn = null;
		ResultSet rs = null;
		PreparedStatement pst = null;
		try {

			conn = getConnection();
			pst =
				conn.prepareStatement(
					"select 1 from LUCENE where file_name = ?");
			pst.setString(1, arg0);
			rs = pst.executeQuery();
			present = rs.next();
		} catch (SQLException e) {
			throw new IOException(e.getMessage());
		} catch (NamingException e) {
			throw new IOException(e.getMessage());
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, rs);
			} catch (SQLException e1) {
				throw new IOException(e1.getMessage());
			}

		}
		return present;
	}

	/**
	 * @see org.apache.lucene.store.Directory#fileModified(java.lang.String)
	 */
	public long fileModified(String arg0) throws IOException {
		Connection conn = null;
		PreparedStatement pst = null;
		ResultSet rs = null;
		try {
			conn = getConnection();
			pst =
				conn.prepareStatement(
					"select max(last_modified) from LUCENE where file_name = ?");
			pst.setString(1, arg0);
			rs = pst.executeQuery();
			rs.next();
			long retVal = rs.getLong(1);
			return retVal;
		} catch (SQLException se) {
			throw new IOException(se.getMessage());
		} catch (NamingException e) {
			throw new IOException(e.getMessage());
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, rs);
			} catch (SQLException e) {
				throw new IOException(e.getMessage());
			}
		}
	}

	/**
	 * @see org.apache.lucene.store.Directory#touchFile(java.lang.String)
	 */
	public void touchFile(String arg0) throws IOException {
		Connection conn = null;
		PreparedStatement pst = null;

		try {
			conn = getConnection();
			pst =
				conn.prepareStatement(
					"update LUCENE set last_modified = ? where file_name= ?");
			pst.setLong(1, System.currentTimeMillis());
			pst.setString(2, arg0);
			pst.executeUpdate();
		} catch (SQLException e) {
			throw new IOException(e.getMessage());
		} catch (NamingException e) {
			throw new IOException(e.getMessage());
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, null);
			} catch (SQLException e) {
				throw new IOException(e.getMessage());
			}
		}
	}

	/**
	 * @see org.apache.lucene.store.Directory#deleteFile(java.lang.String)
	 */
	public void deleteFile(String arg0) throws IOException {
		Connection conn = null;
		PreparedStatement pst = null;
		try {
			conn = getConnection();
			pst =
				conn.prepareStatement("delete from LUCENE where file_name = ?");
			pst.setString(1, arg0);
			pst.executeUpdate();
		} catch (SQLException e) {
			throw new IOException(e.getMessage());
		} catch (NamingException e) {
			throw new IOException(e.getMessage());
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, null);
			} catch (SQLException e) {
				throw new IOException(e.getMessage());
			}
		}
	}

	/**
	 * @see org.apache.lucene.store.Directory#renameFile(java.lang.String, java.lang.String)
	 */
	public void renameFile(String src, String dest) throws IOException {
		Connection conn = null;
		PreparedStatement pst = null;
		try {

			if (fileExists(dest)) {
				conn = getConnection();
				pst =
					conn.prepareStatement(
						"delete from LUCENE where file_name = ?");
				pst.setString(1, dest);
				pst.executeUpdate();
				pst.close();
			} else {
				conn = getConnection();
			}

			pst =
				conn.prepareStatement(
					"update LUCENE set file_name = ?, last_modified = ? "
						+ "WHERE file_name = ?");
			pst.setString(1, dest);
			pst.setLong(2, System.currentTimeMillis());
			pst.setString(3, src);
			pst.executeUpdate();
		} catch (SQLException e) {
			throw new IOException(e.getMessage());
		} catch (NamingException e) {
			throw new IOException(e.getMessage());
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, null);
			} catch (SQLException e) {
				throw new IOException(e.getMessage());
			}
		}
	}

	/**
	 * @see org.apache.lucene.store.Directory#fileLength(java.lang.String)
	 */
	public long fileLength(String arg0) throws IOException {
		long end;
		Connection conn = null;
		PreparedStatement pst = null;
		ResultSet rs = null;
		if (!fileExists(arg0)) {
			throw new IOException("Le fichier " + arg0 + " n'existe pas !");
		}
		try {
			conn = getConnection();
			pst =
				conn.prepareStatement(
					"SELECT end_byte, start_byte, data FROM lucene "
						+ "WHERE file_name = ? ORDER BY end_byte desc");
			pst.setString(1, arg0);
			rs = pst.executeQuery();
			rs.next();
			end = rs.getLong(1);
			Blob blob = rs.getBlob(3);

			if (blob == null) {
				return 0;
			} else {
				return end;
			}
		} catch (SQLException e) {
			throw new IOException(e.getMessage());
		} catch (NamingException e) {
			throw new IOException(e.getMessage());
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, rs);
			} catch (SQLException e) {
				throw new IOException(e.getMessage());
			}
		}
	}

	/**
	 * @see org.apache.lucene.store.Directory#createFile(java.lang.String)
	 */
	public OutputStream createFile(String name) throws IOException {
		Connection conn = null;
		PreparedStatement pst = null;
		try {
			if (fileExists(name)) {
				deleteFile(name);
			}
			conn = getConnection();
			pst =
				conn.prepareStatement(
					"insert into LUCENE (file_name, last_modified, start_byte, end_byte, data) values "
						+ ""
						+ "(?, ?, 1, 1, null)");
			pst.setString(1, name);
			pst.setLong(2, System.currentTimeMillis());

			pst.executeUpdate();

		} catch (SQLException e) {
			throw new IOException(e.getMessage());
		} catch (NamingException e) {
			throw new IOException(e.getMessage());
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, null);
			} catch (SQLException e) {
				throw new IOException(e.getMessage());
			}
		}

		return new SQLOutputStream(name);
	}

	/**
	 * @see org.apache.lucene.store.Directory#openFile(java.lang.String)
	 */
	public InputStream openFile(String name) throws IOException {
		return new SQLInputStream(name, fileLength(name));
	}

	/**
	 * @see org.apache.lucene.store.Directory#makeLock(java.lang.String)
	 */
	public Lock makeLock(String arg0) {
		return new SQLLock();
	}

	/**
	 * @see org.apache.lucene.store.Directory#close()
	 * do nothing pour le moment...
	 */
	public void close() throws IOException {
		closed = true;
	}

}
