package fr.aliacom.lucene.store;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.store.InputStream;

/**
 * @author nicolasr
 * cr le 13 juin 2003
 */
public class SQLInputStream extends InputStream {

	private static Log logger = LogFactory.getLog(SQLInputStream.class);

	private long pos; // position courante dans le buffer
	private long segOrder;
	// numro du segment de fichier dans lequel le prochain read aura lieu
	private long segOffset; // offset dans le segment courant 
	private String filename;
	private boolean closed;

	/**
	 * @param name
	 */
	public SQLInputStream(String name, long length) throws IOException {
		filename = name;
		pos = 0;
		segOffset = 0;
		segOrder = 0;
		closed = false;
		this.length = length;
	}

	/**
	 * @see org.apache.lucene.store.InputStream#readInternal(byte[], int, int)
	 */
	protected void readInternal(byte[] b, int offset, int len)
		throws IOException {

		if (logger.isDebugEnabled()) {
			logger.debug(
				"readInternal("
					+ filename
					+ ") : offset = "
					+ offset
					+ " len = "
					+ len);
		}

		if (pos != getFilePointer()) {
			pos = getFilePointer();
		}

		int totalBytesToRead = len;
		int nbBytesToRead = len;
		int nbBytesReaded = 0;
		boolean eof = false;
		byte[] lastBytesRead = null;

		while (totalBytesToRead > 0 && !eof) {
			segOrder = pos / SQLDirectory.SEGMENT_SIZE;
			segOffset = pos - (segOrder * SQLDirectory.SEGMENT_SIZE);

			nbBytesToRead =
				Math.min(
					totalBytesToRead,
					(int) (SQLDirectory.SEGMENT_SIZE * (segOrder + 1) - pos));

			if (logger.isDebugEnabled()) {
				logger.debug(
					"pos = "
						+ pos
						+ " segOrder = "
						+ segOrder
						+ " segOffset = "
						+ segOffset
						+ " nbBytesToRead = "
						+ nbBytesToRead
						+ " totalBytesToRead = "
						+ totalBytesToRead);
			}

			lastBytesRead = getData(nbBytesToRead);

			if (lastBytesRead == null || lastBytesRead.length == 0) {
				eof = true;
				break;
			}

			System.arraycopy(
				lastBytesRead,
				0,
				b,
				offset + nbBytesReaded,
				nbBytesToRead);

			totalBytesToRead -= nbBytesToRead;
			nbBytesReaded += nbBytesToRead;
			pos += nbBytesToRead;
			nbBytesToRead = totalBytesToRead;

			if (logger.isDebugEnabled()) {
				logger.debug(
					"readInternal() : totalBytesToRead = "
						+ totalBytesToRead
						+ " nbBytesReaded = "
						+ nbBytesReaded
						+ " nbBytesToRead = "
						+ nbBytesToRead);
			}
		}
	}

	private byte[] getData(int nbBytesToRead) throws IOException {
		if (closed) {
			throw new IOException("InputStream is closed");
		}
		Connection conn = null;
		PreparedStatement pst = null;
		ResultSet rs = null;
		try {
			long startB = segOrder * SQLDirectory.SEGMENT_SIZE + 1;
			long endB = (segOrder + 1) * SQLDirectory.SEGMENT_SIZE;

			conn = getConnection();
			pst =
				conn.prepareStatement(
					" SELECT data, start_byte, end_byte FROM lucene"
						+ " WHERE file_name = ? AND start_byte = ? AND end_byte <= ? "
						+ " ORDER BY start_byte");

			pst.setString(1, filename);
			pst.setLong(2, startB);
			pst.setLong(3, endB);

			rs = pst.executeQuery();
			byte[] bytesRead = null;
			byte[] retVal = null;
			if (rs.next()) {
				bytesRead = LuceneHelper.readBlob(rs, 1);
			}
			rs.close();

			retVal = new byte[nbBytesToRead];

			System.arraycopy(
				bytesRead,
				(int) segOffset,
				retVal,
				0,
				nbBytesToRead);

			return retVal;

		} catch (SQLException e) {
			throw asIOException(e);
		} catch (NamingException e) {
			throw asIOException(e);
		} catch (IOException e) {
			throw e;
		} finally {
			try {
				LuceneHelper.cleanup(conn, pst, rs);
			} catch (SQLException e1) {
				asIOException(e1);
			}
		}
	}

	private IOException asIOException(Exception e) {
		IOException ioe = new IOException(e.getMessage());
		ioe.initCause(e);
		return ioe;
	}

	/**
	* @see org.apache.lucene.store.InputStream#close()
	*/
	public void close() throws IOException {
		closed = true;
	}

	/**
	 * positionne le point de lecture courante  pos
	 * @see org.apache.lucene.store.InputStream#seekInternal(long)
	 */
	protected void seekInternal(long position) throws IOException {
		if (closed) {
			throw new IOException("InputStream is closed");
		}
	}

	protected Connection getConnection()
		throws NamingException, SQLException, IOException {
		if (closed) {
			throw new IOException("InputStream is closed");
		}
		return LuceneHelper.getConnection();
	}

}
