// Ben Clewett, April 2004
//
// Wrapper for SQLite to return a query as a table in memory.
//
// There is no warrenty, the code provided for illistation reasons only. Any user withing to make use of this does so at their own risk.


#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "sqlite_wrapper.h"

#define MAX_GET_LOCK_ATTEMPT 10

int sw_error = SQLITE_OK;
const char *sw_errmsg;

unsigned char data_empty[1] = { 0 };
char name_unknown[] = "?";

// Internal functions:
sw_result *empty_table();
sw_result *scalar_table(unsigned char *data);
sw_result *malloc_result();
sw_cell_data *malloc_cell();
sw_meta_data *malloc_meta_data(int count);
const unsigned char *sw_u_strdup(const unsigned char *data);
const unsigned char *sw_cell_print(sw_result *result, int col, int row);
void sw_divider(sw_result *result, FILE *f, int *col_size);
void sw_show_escape(FILE *f, char *field);

sw_result *sw_query(sqlite3 *handle, const char *query)
{
	static sw_result *result;
	static sw_cell_data *cell, *last_cell;
	static sqlite3_stmt *h_statement;
	static const char *tail;
	static const unsigned char *data;
	static int current_col, i, return_value;

	h_statement = NULL;

	if (!handle) {
		sw_error = SQLITE_ERROR;
		sw_errmsg = "Null Handle";
		return NULL;
	}

	if (!query) {
		sw_error = SQLITE_ERROR;
		sw_errmsg = "Null Query";
		return NULL;
	}

	if ((return_value = sqlite3_prepare(handle, query, -1, &h_statement, &tail)) != SQLITE_OK) {
		sw_error = return_value;
		sw_errmsg = sqlite3_errmsg(handle);
		return NULL;
	}

	result = malloc_result();
	if (!result) {
		sw_error = SQLITE_NOMEM;
		sw_errmsg = "Out of Memory";
		return NULL;
	}

	result->row_count = 0;
	last_cell = NULL;

	while (1) {

		int lock_count = 0;

		do {
			return_value = sqlite3_step(h_statement);
		} while ((return_value == SQLITE_BUSY || return_value == SQLITE_LOCKED) && lock_count < MAX_GET_LOCK_ATTEMPT);

		if (return_value != SQLITE_ROW) break;

		result->col_count = sqlite3_column_count(h_statement);

		if (result->row_count == 0) {
			result->meta_data = malloc_meta_data(result->col_count);
			if (!result->meta_data) {
				sw_free_result(result);
				sw_error = SQLITE_NOMEM;
				sw_errmsg = "Out of Memory";
				return NULL;
			}
			for (current_col = 0; current_col < result->col_count; current_col++) {
				result->meta_data[current_col].type = sqlite3_column_type(h_statement, current_col);
				result->meta_data[current_col].name = strdup(sqlite3_column_name(h_statement, current_col));
				if (!result->meta_data[current_col].name) {
					result->meta_data[current_col].name = name_unknown;
				}
			}

		}

		for (current_col = 0; current_col < result->col_count; current_col++) {

			cell = malloc_cell();
			if (!cell) {
				sw_free_result(result);
				sw_error = SQLITE_NOMEM;
				sw_errmsg = "Out of Memory";
				return NULL;
			}

			if (last_cell) last_cell->next = cell;
			last_cell = cell;

			if (!result->cell_list) result->cell_list = cell;

			data = sqlite3_column_text(h_statement, current_col);

			if (data) {
				cell->data = sw_u_strdup(data);
				if (!cell->data) {
					sw_free_result(result);
					sw_error = SQLITE_NOMEM;
					sw_errmsg = "Out of Memory";
					return NULL;
				}
			} else {
				cell->data = NULL;
			}
		}

		result->row_count++;
	}

	if (return_value == SQLITE_BUSY || return_value == SQLITE_LOCKED) {
		sw_free_result(result);
		sw_error = return_value;
		sw_errmsg = "Cannot obtain lock or database busy.";
		return NULL;
	}

	if (return_value != SQLITE_DONE) {
		sw_free_result(result);
		sw_error = return_value;
		sw_errmsg = sqlite3_errmsg(handle);
		return NULL;
	}
	if (!result->row_count) {
		sw_free_result(result);
		sqlite3_finalize(h_statement);
		return empty_table();
	}

	result->cell_array = (sw_cell_data **)malloc(sizeof(sw_cell_data *) * result->row_count * result->col_count);
	if (!result->cell_array) {
		sw_free_result(result);
		sw_error = SQLITE_NOMEM;
		sw_errmsg = "Out of Memory";
		return NULL;
	}

	cell = result->cell_list;
	for (i = 0; i < result->row_count * result->col_count; i++) {
		result->cell_array[i] = cell;
		cell = (sw_cell_data *)cell->next;
	}

	sqlite3_finalize(h_statement);

	sw_error = SQLITE_OK;
	sw_errmsg = "";

	return result;
}

sw_result *empty_table()
{
	static sw_result *result;

	result = malloc_result();
	if (!result) {
		sw_error = SQLITE_NOMEM;
		sw_errmsg = "Out of Memory";
		return NULL;
	}
	
	result->col_count = result->row_count = 0;
	result->cell_array = NULL;
	result->cell_list = NULL;
	result->meta_data = NULL;

	sw_error = SQLITE_OK;
	sw_errmsg = "";

	return result;

}


int sw_get_errno()
{
	return sw_error;
}

const char *sw_get_errmsg()
{
	return sw_errmsg;
}

void sw_free_result(sw_result *result)
{
	sw_cell_data *cell, *prev_cell;
	int i;

	if (!result) return;

	if (result->meta_data) {
		for (i = 0; i < result->col_count; i++) {
			if (result->meta_data[i].name != name_unknown) free(result->meta_data[i].name);
		}
	}

	cell = result->cell_list;
	while (cell) {
		if (cell->data != data_empty) free((char *)cell->data);
		prev_cell = cell;
		cell = (sw_cell_data *)cell->next;
		free(prev_cell);
	}

	free(result->cell_array);

	free(result->meta_data);

	free(result);

}

unsigned char sw_is_null(sw_result *result, unsigned int row, unsigned int col)
{
	static int i;

	if (!result) return 0;

	i = (row * result->col_count) + col;

	return (result->cell_array[i]->data) ? 0 : 1;

}

const unsigned char *sw_cell(sw_result *result, unsigned int row, unsigned int col)
{
	static int i;
	static const unsigned char *data;

	if (!result) return data_empty;

	i = (row * result->col_count) + col;
	data = result->cell_array[i]->data;

	return (data) ? data : data_empty;
}

int sw_cell_i(sw_result *result, unsigned int row, unsigned int col)
{
	static int i;
	i = atoi((const char *)sw_cell(result, row, col));
	return i;
}
double sw_cell_d(sw_result *result, unsigned int row, unsigned int col)
{
	static double d;
	d = atof((const char *)sw_cell(result, row, col));
	return d;
}

// Malloc functions used so that can NULL content
// therefore can FREE data without core dump.
sw_result *malloc_result()
{
	static sw_result *result;

	result = (sw_result *)malloc(sizeof(sw_result));
	
	if (!result) return NULL;

	result->cell_array = NULL;
	result->cell_list = NULL;
	result->meta_data = NULL;
	result->row_count = 0;
	result->col_count = 0;

	return result;
}

sw_cell_data *malloc_cell()
{
	static sw_cell_data *cell;

	cell = (sw_cell_data *)malloc(sizeof(sw_cell_data));
	
	if (!cell) return NULL;

	cell->data = NULL;
	cell->next = NULL;

	return cell;
}

sw_meta_data *malloc_meta_data(int count)
{
	sw_meta_data *data;
	int i;

	data = (sw_meta_data *)malloc(sizeof(sw_meta_data) * count);
	if (!data) return NULL;

	for (i = 0; i < count; i++) {
		data[i].type = 0;
		data[i].name = NULL;
	}

	return data;

}


unsigned int sw_col_count(sw_result *result)
{
	if (!result) return 0;

	return result->col_count;
}

unsigned int sw_row_count(sw_result *result)
{
	if (!result) return 0;

	return result->row_count;

}

const unsigned char *sw_u_strdup(const unsigned char *data)
{
	static unsigned char *copy_data;
	static int i, len;

	for (i = 0; data[i]; i++) ;
	len = i;

	copy_data = (unsigned char *)malloc(sizeof(unsigned char) * (len + 1));

	for (i = 0; i < len; i++) copy_data[i] = data[i];
	copy_data[len] = 0;

	return copy_data;
}

void sw_show_table(sw_result *result, FILE *f)
{
	static int col_size[1024], col, row, len;
	static char format[20];

	if (sw_row_count(result) == 0) {
		return;
	}

	// Get the colunm size
	for (col = 0; col < sw_col_count(result); col++) {
		col_size[col] = 0;
		for (row = 0; row < sw_row_count(result); row++) {
			len = strlen((char *)sw_cell_print(result, col, row));
			if (len > col_size[col]) col_size[col] = len;
		}
		len = strlen(result->meta_data[col].name);
		if (len > col_size[col]) col_size[col] = len;
	}

	sw_divider(result, f, col_size);

	// Header:
	fprintf(f, "|");
	for (col = 0; col < sw_col_count(result); col++) {
		sprintf(format, "%%-%ds|", col_size[col]);
		fprintf(f, format, result->meta_data[col].name);
	}
	fprintf(f, "\n");
	fflush(f);

	sw_divider(result, f, col_size);

	// Data:
	for (row = 0; row < sw_row_count(result); row++) {
		fprintf(f, "|");
		for (col = 0; col < sw_col_count(result); col++) {
			sprintf(format, "%%-%ds|", col_size[col]);
			fprintf(f, format, sw_cell_print(result, col, row));
		}
		fprintf(f, "\n");
		fflush(f);
	}

	sw_divider(result, f, col_size);

}

void sw_divider(sw_result *result, FILE *f, int *col_size)
{
	static int i, col;

	fprintf(f, "+");
	for (col = 0; col < sw_col_count(result); col++) {
		for (i = 0; i < col_size[col]; i++) fprintf(f, "-");
		fprintf(f, "+");
	}
	fprintf(f, "\n");
	fflush(f);
}

const unsigned char *sw_cell_print(sw_result *result, int col, int row)
{

	if (sw_is_null(result, row, col)) return (const unsigned char *)"NULL";

	return sw_cell(result, row, col);

}

void sw_show_table_cvs(sw_result *result, FILE *f)
{
	int col, row;

	// Header:
	for (col = 0; col < sw_col_count(result); col++) {
		fprintf(f, "%s", (col == 0) ? "" : ",");
		sw_show_escape(f, result->meta_data[col].name);
	}
	fprintf(f, "\n");
	fflush(f);

	// Data:
	for (row = 0; row < sw_row_count(result); row++) {
		for (col = 0; col < sw_col_count(result); col++) {
			fprintf(f, "%s", (col == 0) ? "" : ",");
			sw_show_escape(f, (char *)sw_cell(result, row, col));
		}
		fprintf(f, "\n");
		fflush(f);
	}

}

void sw_show_table_xml(sw_result *result, FILE *f, char *table_tag_name, char *row_tag_name)
{
	int col, row;

	// Header
	printf("<%s>\n", table_tag_name);

	// Data:
	for (row = 0; row < sw_row_count(result); row++) {

		fprintf(f, "  <%s", row_tag_name);
		for (col = 0; col < sw_col_count(result); col++) {
			fprintf(f, " %s=", result->meta_data[col].name);
			sw_show_escape(f, (char *)sw_cell(result, row, col));
		}
		fprintf(f, " />\n");
		fflush(f);
	}

	printf("</%s>\n", table_tag_name);
}

void sw_show_table_html(sw_result *result, FILE *f)
{
	int col, row;

	// Header
	printf("<TABLE BORDER=1 CELLPADDING=2 CELLSPACING=0>\n");
	fprintf(f, "  <TR>\n");
	for (col = 0; col < sw_col_count(result); col++) {
		fprintf(f, "    <TD ALIGN=CENTER><B><FONT FACE=\"Courier\">%s</FONT></B></TD>\n", result->meta_data[col].name);
	}
	fprintf(f, "  </TR>\n");
	fflush(f);

	// Data:
	for (row = 0; row < sw_row_count(result); row++) {

		fprintf(f, "<TR>\n");
		for (col = 0; col < sw_col_count(result); col++) {
			fprintf(f, "    <TD><FONT FACE=\"Courier\">%s</FONT></TD>\n", sw_cell_print(result, col, row));
		}
		fprintf(f, "</TR>\n");
		fflush(f);
	}

	fprintf(f, "</TABLE>\n");
}


void sw_show_escape(FILE *f, char *field)
{
	int i;

	fprintf(f, "\"");

	for (i = 0; field[i]; i++) {
		if (field[i] == '\"') fprintf(f, "\"\"");
		else if (field[i] >= ' ') fprintf(f, "%c", field[i]);
		else fprintf(f, ".");
	}

	fprintf(f, "\"");

}



