from __future__ import with_statement

import httplib
import time

import server
import squid

class TestCacheBehaviour(object):
    connection = None
        
    def setup_class(cls):
        cls.server = server.HTTPServerThread(('0.0.0.0', 8000),
                                             ('0.0.0.0', 8001))
        cls.squid = squid.SquidController()
        cls.server.start()
        cls.server.ready.wait()
        cls.squid.start()
    
    def setup_method(self, method):
        if self.connection is not None:
            self.connection.close()
        self.connection = httplib.HTTPConnection('harfy.jeamland.net', 3128)
        
    def teardown_class(cls):
        if cls.connection is not None:
            cls.connection.close()
        cls.squid.stop()
        cls.server.quit = True
        cls.server.join()
    
    def basic_request(self, method, url, body=None, headers={}):
        self.connection.request(method, url, body, headers)
        response = self.connection.getresponse()
        assert response.status == 200
        assert response.read() == server.normal_body
        return response

    def redir_request(self, method, url, body=None, headers={}):
        self.connection.request(method, url, body, headers)
        response = self.connection.getresponse()
        assert response.status == 302
        return response

    def put_request(self, method, url, body=None, headers={}):
        self.connection.request(method, url, body, headers)
        response = self.connection.getresponse()
        assert response.status == 200
        assert response.read() == ''
        return response
        
    def post_request(self, method, url, body=None, headers={}):
        self.connection.request(method, url, body, headers)
        response = self.connection.getresponse()
        assert response.status == 201
        assert response.read() == ''
        return response
        
    def charset_request(self, method, url, charset=None, body=None, headers={}):
        if charset is not None:
            assert charset.lower() in server.charset_bodies
            headers['Accept-charset'] = charset
            self.connection.request(method, url, body, headers)
        else:
            charset = 'us-ascii'
            self.connection.request(method, url)
        response = self.connection.getresponse()
        assert response.status == 200
        assert response.read() == server.charset_bodies[charset.lower()]
        return response
        
    def test_get_simple(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')

    def test_put_simple(self):
        self.put_request('PUT', 'http://harfy.jeamland.net:8000/', '',
                           {'Content-length': 0})

    def test_fnord_simple(self):
        self.basic_request('FNORD', 'http://harfy.jeamland.net:8000/')

    def test_get_is_cached(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache
    
    def test_put_evicts(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache
        
        self.put_request('PUT', 'http://harfy.jeamland.net:8000/', '',
                           {'Content-length': 0})
        
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache

    def test_fnord_not_cached(self):
        self.basic_request('FNORD', 'http://harfy.jeamland.net:8000/')
        
        response = self.basic_request('FNORD', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache

    def test_fnord_evicts(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')

        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.basic_request('FNORD', 'http://harfy.jeamland.net:8000/')

        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
    
    def test_put_evicts_loc_absolute(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.redir_request('GET', 'http://harfy.jeamland.net:8000/absloc')
        response = self.redir_request('GET', 'http://harfy.jeamland.net:8000/absloc')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.redir_request('PUT', 'http://harfy.jeamland.net:8000/absloc', '',
                           {'Content-length': 0})

        response = self.redir_request('GET', 'http://harfy.jeamland.net:8000/absloc')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
    
    def test_put_evicts_loc_relative(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.redir_request('GET', 'http://harfy.jeamland.net:8000/loc')
        response = self.redir_request('GET', 'http://harfy.jeamland.net:8000/loc')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.redir_request('PUT', 'http://harfy.jeamland.net:8000/loc', '',
                           {'Content-length': 0})

        response = self.redir_request('GET', 'http://harfy.jeamland.net:8000/loc')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
    
    def test_fnord_evicts_loc_relative(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.redir_request('GET', 'http://harfy.jeamland.net:8000/loc')
        response = self.redir_request('GET', 'http://harfy.jeamland.net:8000/loc')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.redir_request('FNORD', 'http://harfy.jeamland.net:8000/loc')

        response = self.redir_request('GET', 'http://harfy.jeamland.net:8000/loc')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
        
    def test_fnord_evicts_cloc_relative(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.basic_request('GET', 'http://harfy.jeamland.net:8000/cloc')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/cloc')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.basic_request('FNORD', 'http://harfy.jeamland.net:8000/cloc')

        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/cloc')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
        
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
    
    def test_fnord_evicts_loc_absolute(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.redir_request('GET', 'http://harfy.jeamland.net:8000/absloc')
        response = self.redir_request('GET', 'http://harfy.jeamland.net:8000/absloc')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.redir_request('FNORD', 'http://harfy.jeamland.net:8000/absloc')

        response = self.redir_request('GET', 'http://harfy.jeamland.net:8000/absloc')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache

    def test_fnord_evicts_cloc_absolute(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.basic_request('GET', 'http://harfy.jeamland.net:8000/abscloc')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/abscloc')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.basic_request('FNORD', 'http://harfy.jeamland.net:8000/abscloc')

        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/abscloc')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache

        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
    
    def test_get_variants_cache(self):
        self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                             'us-ascii')

        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'us-ascii')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache
        
        self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                             'utf8')

        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'utf8')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

    def test_get_variants_purge(self):
        self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                             'us-ascii')

        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'us-ascii')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                             'utf8')

        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'utf8')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.put_request('PURGE', 'http://harfy.jeamland.net:8000/vary')
        
        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'us-ascii')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
        
        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'utf8')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
        
    def fnord_variant_evict(self, charset):
        self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                             'us-ascii')

        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'us-ascii')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache
        
        self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                             'utf8')

        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'utf8')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.charset_request('FNORD', 'http://harfy.jeamland.net:8000/vary', charset)

        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'us-ascii')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
        
        response = self.charset_request('GET', 'http://harfy.jeamland.net:8000/vary',
                                        'utf8')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache

    def test_fnord_evicts_variants(self):
        for charset in (None, 'us-ascii', 'utf8'):
            yield self.fnord_variant_evict, charset

    def test_fnord_doesnt_evict_cloc_diff_hosts(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8001/')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8001/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.basic_request('GET', 'http://harfy.jeamland.net:8000/hostcloc')
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/hostcloc')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

        self.basic_request('FNORD', 'http://harfy.jeamland.net:8000/hostcloc')

        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/hostcloc')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache

        response = self.basic_request('GET', 'http://harfy.jeamland.net:8001/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache

    def test_post_evicts(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache
        
        self.post_request('POST', 'http://harfy.jeamland.net:8000/')
        
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache

    def test_post_evicts_loc(self):
        self.basic_request('GET', 'http://harfy.jeamland.net:8000/posted')
        
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/posted')
        xcache = response.getheader('X-Cache')
        assert 'HIT' in xcache
        
        self.post_request('POST', 'http://harfy.jeamland.net:8000/')
        
        response = self.basic_request('GET', 'http://harfy.jeamland.net:8000/posted')
        xcache = response.getheader('X-Cache')
        assert 'MISS' in xcache
