# This is the LDAP driver for squid-auth

# TODO: Enable some sort of connection-limiter that
#       keeps us from defining 100 LDAP realms and keeping
#       100 connections open (i.e., only keep the most
#       recently needed, or most often used connections)

# Change this to 0 to disable caching of
# LDAP connections
my $CACHE_ENABLED = 1;

# There should be no need to modify this script below this line,
# unless you're adding code modifications or reprogramming

use Net::LDAP;
use Net::LDAPS;

my %LDAP_CACHE = ();

sub __LDAP_get_object
{
    my $host = $_[0];
    my $port = $_[1];
    my $config = $_[2];

    my %return_hash = ();

    # Check the defined protocol, and default it if
    # not specified
    my $protocol = $config->{'protocol'};
    $protocol = "ldap" unless defined( $protocol );

    my $is_tls = 0;
    my $is_encrypted = 0;

    # Get the security information, and set any defaults
    my $client_cert = $config->{'cert'};
    my $client_key = $config->{'key'};
    my $ca_cert = $config->{'cacert'};
    my $ca_path = $config->{'capath'};
    my $verify = $config->{'verify'};
    $client_key = $client_cert unless defined( $client_key );

    debug( 1, "Connecting to the LDAP server" );

    if ( $protocol eq "ldap" )
    {
        # Default the port if necessary
        $port = 389 unless defined( $port );

        # Check to see if this is an auto-TLS,
        # or unencrypted connection
        if ( defined( $client_cert ) or defined( $ca_cert ) or defined( $ca_path ) or defined( $verify ) )
        {
            $is_tls = 1;
            $is_encrypted = 1;
        }
    }
    elsif ( $protocol eq "ldaps" )
    {
        # Default the port if necessary
        $port = 636 unless defined( $port );

        # This does specify encryption, but not TLS
        $is_tls = 0;
        $is_encrypted = 1;
    }
    else
    {
        # Unknown protocol specified
        $return_hash{'error'} = "Unsupported LDAP Protocol variant '$protocol'";
        return \%return_hash;
    }

    # This hash stores the connection information
    my %conn_param = ();
    my $conn_hash = \%conn_param;

    # This hash stores the security information
    my %sec_param = ();
    my $sec_hash = \%sec_param;

    # Check to see if encryption is requested, and validate
    # the supplied configuration
    if ( $is_encrypted )
    {
        if ( defined( $client_cert ) )
        {
            unless ( -f $client_cert )
            {
                $return_hash{'error'} = "Client certificate '$client_cert' not found";
                return \%return_hash;
            }
            $sec_hash->{clientcert} = $client_cert;

            unless ( -f $client_key )
            {
                $return_hash{'error'} = "Client certificate key '$client_key' not found";
                return \%return_hash;
            }
            $sec_hash->{clientkey} = $client_key;

            debug( 3, "Client Certificate = '$client_cert'" );
            debug( 3, "Client Key = '$client_key'" );
        }

        if ( defined( $verify ) )
        {
            $sec_hash->{verify} = $verify;
        }
        $verify = "none" unless defined( $verify );

        if ( defined( $ca_cert ) or defined( $ca_path ) )
        {
            if ( defined( $ca_path ) )
            {
                unless ( -d $ca_path )
                {
                    $return_hash{'error'} = "CA path '$ca_path' not found";
                    return \%return_hash;
                }
                $sec_hash->{capath} = $ca_path;
                debug( 3, "CA Path = '$ca_path'" );
            }
            else
            {
                unless ( -f $ca_cert )
                {
                    $return_hash{'error'} = "CA certificate '$ca_cert' not found";
                    return \%return_hash;
                }
                $sec_hash->{cafile} = $ca_cert;
                debug( 3, "CA Certificate = '$ca_cert'" );
            }
            $verify = "optional" unless ( $verify eq "required" );
        }

        $sec_hash->{verify} = $verify;
        debug( 3, "Verify Mode = $verify" );

        # For LDAPS, the connection and security hash are
        # one and the same, so we make $conn_hash point
        # to the same hash as $sec_hash, so when we fill
        # the rest of the connection info, it all ends up
        # in the same place
        $conn_hash = $sec_hash unless ( $is_tls );
    }
    $conn_hash->{port} = $port;
    $conn_hash->{version} = 3;

    my $ldap;

    my $label = "LDAP";
    $label = "LDAP(TLS)" if ( $is_tls );
    $label = "LDAPS" if ( $is_encrypted and ! $is_tls );
    debug( 2, "$label: ( $host, $port )" );

    if ( $is_tls or ! $is_encrypted )
    {
        # Unencrypted LDAP, or LDAP+TLS
        $ldap = new Net::LDAP( $host, %{$conn_hash} );
    }
    else
    {
        # Encrypted LDAP using SSL
        $ldap = new Net::LDAPS( $host, %{$conn_hash} );
    }

    unless ( defined( $ldap ) )
    {
        $return_hash{'error'} = "Unable to create new LDAP object";
        return \%return_hash;
    }

    if ( $is_tls )
    {
        debug( 3, "Changing connection to TLS mode" );
        my $tls = $ldap->start_tls( %{$sec_hash} );
        if ( $tls->code() )
        {
            my $err = $tls->error();
            $return_hash{'error'} = $err;
            $ldap->disconnect();
            return \%return_hash;
        }
    }

    if ( $is_encrypted )
    {
        my $srv_cert = $ldap->certificate();
        my $cipher = $ldap->cipher();
        my $subject = $srv_cert->subject_name();
        debug( 3, "Server Certificate: '$subject'" );
        debug( 3, "Server Cipher: '$cipher'" );
    }

    my $version = $ldap->version();
    debug( 2, "LDAP version: '$version'" );

    $return_hash{'ldap'} = $ldap;
    return \%return_hash;
}


# Perform the actual authentication by binding using a specific
# DN/Password combination, and an existing LDAP object
sub __LDAP_authenticate_user
{
    my $ldap = $_[0];
    my $dn = $_[1];
    my $passwd = $_[2];
    my $error;

    debug( 3, "BIND: ( $dn, <passwd> )" );

    my $bind = $ldap->bind( $dn, password => $passwd );
    if ( $bind->code() )
    {
        $error = $bind->error();
    }

    # If we're caching the object, we don't unbind.
    # This needs improving upon, since I don't particularly
    # like maintaining an open, authenticated LDAP connection
    # hanging around when not needed
    $ldap->unbind() unless ( $CACHE_ENABLED );

    return error( $error ) if defined( $error );
    return success( "Bound as '$dn'" );
}

sub LDAP_authenticate_user
{
    my $config = $_[0];
    my $mod_home = $_[1];
    my $uid = $_[2];
    my $realm = $_[3];
    my $passwd = $_[4];

    # First, make sure we have the config parameters we need
    my $host = $config->{'host'};
    defined( $host ) or return error( "No host specified in $realm\'s configuration" );

    my $dn = $config->{'dnstring'};
    defined( $dn ) or return error( "No 'dnstring' parameter in $realm\'s configuration" );

    # First, we get whatever is in the cache
    my $ldap = $LDAP_CACHE{$realm};
    unless ( defined( $ldap ) )
    {
        # Nothing is in the cache, so we create a
        # new LDAP object for this realm
        my $ldap_hash = __LDAP_get_object( $host, $config->{'port'}, $config );

        my $error = $ldap_hash->{'error'};
        if ( defined( $error ) )
        {
            return error( $error );
        }

        $ldap = $ldap_hash->{'ldap'};
        if ( $CACHE_ENABLED )
        {
            debug( 2, "Caching object for realm '$realm'" );
            $LDAP_CACHE{$realm} = $ldap;
        }
    }
    else
    {
        debug( 2, "Using cached LDAP object for realm '$realm'" );
    }

    # This should never happen, but it's always good practice to
    # catch bugs with ad-hoc tests.  This code can eventually go
    # away when this script is more matured
    unless ( defined( $ldap ) )
    {
        return error( "Unknown error - no LDAP object is available" );
    }

    # Now, perform the substitution on the DN string
    while ( $dn =~ /%{.*}/ )
    {
        $dn =~ s/%{uid}/${uid}/g;
        $dn =~ s/%{realm}/${realm}/g;
    }

    # Perform the actual authentication
    return __LDAP_authenticate_user( $ldap, $dn, $passwd );
}

1;
