/* Copyright 2007 Petr Masopust, Tim Fennell
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.stripes.integration.ejb3;

import net.sourceforge.stripes.controller.Interceptor;
import net.sourceforge.stripes.controller.ExecutionContext;
import net.sourceforge.stripes.controller.Intercepts;
import net.sourceforge.stripes.controller.LifecycleStage;
import net.sourceforge.stripes.action.Resolution;
import net.sourceforge.stripes.util.Log;


/**
 * <p>An {@link Interceptor} that uses JNDI to inject EJB beans into newly created
 * ActionBeans immediateley following ActionBeanResolution.  For more information on how the injection
 * is performed see {@link EJBHelper#injectBeans(Object,
 *  net.sourceforge.stripes.action.ActionBeanContext)}.</p>
 *
 * <p>To configure the SpringInterceptor for use you will need to add the following to your
 * web.xml (assuming no other interceptors are yet configured):</p>
 *
 * <pre>
 * &lt;init-param&gt;
 *     &lt;param-name&gt;Interceptor.Classes&lt;/param-name&gt;
 *     &lt;param-value&gt;
 *         net.sourceforge.stripes.integration.spring.EJBInterceptor,
 *         net.sourceforge.stripes.controller.BeforeAfterMethodInterceptor
 *     &lt;/param-value&gt;
 * &lt;/init-param&gt;
 * </pre>
 *
 * <p>If one or more interceptors are already configured in your web.xml simply separate the
 * fully qualified names of the interceptors with commas (additional whitespace is ok).</p>
 *
 * @see EJBBean
 * @author Petr Masopust, Tim Fennell
 * @since Stripes 1.4.3
 */
@Intercepts(LifecycleStage.ActionBeanResolution)
public class EJBInterceptor implements Interceptor {
    private static final Log log = Log.getInstance(EJBInterceptor.class);

    /**
     * Allows ActionBean resolution to proceed and then once the ActionBean has been
     * located invokes the {@link EJBHelper} to perform EJB based dependency injection.
     *
     * @param context the current execution context
     * @return the Resolution produced by calling context.proceed()
     * @throws Exception if the Spring binding process produced unrecoverable errors
     */
    public Resolution intercept(ExecutionContext context) throws Exception {
        Resolution resolution = context.proceed();
        log.debug("Running EJB dependency injection for instance of ",
                  context.getActionBean().getClass().getSimpleName());
        EJBHelper.injectBeans(context.getActionBean());
        return resolution;
    }
}
