/**
 * Copyright (C) 2007 Robert Brautigam.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.stripes.tag.layout;

import javax.servlet.jsp.JspException;
import javax.servlet.jsp.tagext.SimpleTagSupport;
import javax.servlet.jsp.tagext.JspTag;
import javax.servlet.jsp.tagext.Tag;
import javax.servlet.jsp.tagext.SimpleTag;
import net.sourceforge.stripes.tag.StripesTagSupport;
import java.io.IOException;

/**
 * This is a dynamic layout component, because it invokes it's
 * jsp body every time it is needed in the layout definition,
 * instead of evaluating once into a string.
 * @author Brautigam Robert
 * @version Revision: $Revision$
 */
public class LayoutDynamicComponentTag extends SimpleTagSupport
{
   private String name;

   public String getName()
   {
      return name;
   }
   public void setName(String name)
   {
      this.name=name;
   }
   
   /**
    * This method is called when this tag is displaying. If the parent
    * is a layout-definition then we only evaluate the body, if it was
    * not overidden. If the parent is a layout-render, we do not display.
    */
   public void doTag()
      throws JspException, IOException
   {
        JspTag stripesTag = getParentTag(LayoutRenderTag.class);
        if ( stripesTag == null )
           stripesTag = getParentTag(LayoutDefinitionTag.class);

        if ( stripesTag == null )
           return; // Not found
        if ( stripesTag instanceof LayoutRenderTag )
        {
           // We should render, instead we just register
           ((LayoutRenderTag) stripesTag).addComponent(name,this);
        } else {
           // We are in a definition tag, let it decide, whether
           // we display ourselves
           if ( ((LayoutDefinitionTag) stripesTag).permissionToRender(name) )
              render();
        }
   }

   /**
    * Render body.
    */
   public void render()
      throws JspException, IOException
   {
      getJspBody().invoke(getJspContext().getOut());
   }

   /**
    * Find good parent.
    */
   private JspTag getParentTag(Class tagType)
   {
      JspTag parent = getParent();
      while (parent != null)
      {
         if (tagType.isAssignableFrom(parent.getClass()))
            return parent;
         if ( parent instanceof Tag )
            parent = ((Tag)parent).getParent();
         else if ( parent instanceof SimpleTag )
            parent = ((SimpleTag)parent).getParent();
         else
            parent = null;
      }
      return parent;
   }

}


