#define _BSD_SOURCE             /* To get NI_MAXHOST and NI_MAXSERV
                                   definitions from <netdb.h> */
#define _XOPEN_SOURCE 700
#define _DEFAULT_SOURCE

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <math.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/epoll.h>
#include <sys/time.h>


/* socket related includes */
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>


#include "main.h"
#include "deps/picohttpparser.h"

#define SQUID_FIX

/* global variables */
char gbuf1[BUFSIZE];
char gbuf2[BUFSIZE];
char * gbuf = gbuf1;
char normalized_dn[DN_MAXSIZE];
char stunnel_id[16]; /* random ID per session */
pid_t pid; /* store own's pid */
client_state cs = MORE_REQUEST;
unsigned long long int cread = 0;
unsigned long long int sread = 0;
int debug_lvl = 0; /* set the debug printing level: 0 (normal), 1 (client header dump), 2 (client/server exchanges) */

/* interal functions (my code is not big enough to warrant split on multiples files)*/
void usage_and_exit(char * prog_name, int exit_val) {
    fprintf(stderr, "[%d] Usage %s: [-v] -c host -p port \n", pid, prog_name);
    fprintf(stderr, "[%d]  -v: verbose mode\n", pid);
    fprintf(stderr, "[%d]  -d: set debug level (-d: client header dump, -dd: client/server exchanges)\n", pid);
    fprintf(stderr, "[%d]  -c: host to connect to\n", pid);
    fprintf(stderr, "[%d]  -p: port to connect to\n", pid);
    fprintf(stderr, "[%d] \n", pid);
    fprintf(stderr, "[%d]  -h: print this message\n", pid);

    exit(exit_val);
}

void my_perror_exit(char * err, pid_t pid, int exit_status) {
    char err_buf[256];

    memset(err_buf, 0, sizeof(err_buf));

    int nwrites = snprintf(err_buf, sizeof(err_buf), "[%d] ", pid);
    strncpy(err_buf + nwrites, err, sizeof(err_buf) - nwrites);
    nwrites += strnlen(err, sizeof(err_buf) - nwrites);
    strerror_r(errno, err_buf + nwrites, sizeof(err_buf) - nwrites);
    fprintf(stderr, "%s [client data rcv: %llu; server data rcv: %llu]\n", err_buf, cread, sread);
    exit(exit_status);
}

void less_verbose() {
    if (freopen("/dev/null", "a", stderr) == NULL) {
        my_perror_exit("fopen: ", pid, errno);
    }
}

/* strcmp, case insensitive */
static int strincmp(const char * s1, const char * s2, size_t len) {
    for(int i=0; i < len; i++) {
        char c1 = '\0', c2 = '\0';
        /* end of the strings */
        if (s1[i] == s2[i] && s1[i] == '\0') break;

        c1 = tolower(s1[i]);
        c2 = tolower(s2[i]);
        if (c1 < c2)
            return -1;
        else if (c1 == c2)
            continue;
        else
            return 1;
    }

    return 0;
}

/* try to print HTTP stream to stderr using the following format per line:
   - X char of ascii string corresponding to the text
   - 2*X char of the corresponding hex string
*/
void stream_print(const char prefix[], const char s[], int len) {
  char ascii_s[80] = { 0 };
  char hex_s[2*80] = { 0 };
  int i = 0;

  while(i < len) {
    memset(ascii_s, 0, sizeof(ascii_s));
    memset(hex_s, 0, sizeof(hex_s));
    for (int j=0; j< sizeof(ascii_s); j++) {
      if (isgraph(s[i]) || s[i] == ' ')
        ascii_s[j] = s[i];
      else
        ascii_s[j]  = '.';

      /* it could probably be optimized a little bit */
      hex_s[2*j] = (s[i] & 0xf0) >> 4;
      if (0 <= hex_s[2*j] && hex_s[2*j] <= 9) hex_s[2*j]+='0';
      if (10 <= hex_s[2*j] && hex_s[2*j] <= 15) hex_s[2*j]+='A'-10;
      hex_s[2*j+1] = s[i] & 0x0f;
      if (0 <= hex_s[2*j+1] && hex_s[2*j+1] <= 9) hex_s[2*j+1]+='0';
      if (10 <= hex_s[2*j+1] && hex_s[2*j+1] <= 15) hex_s[2*j+1]+='A'-10;

      /* force line printing under the three following conditions
         - we have an actual end of line character
         - ascii_s is full and need to be printed
         - the end of s has been reached */
      if (s[i] == '\n' || j == (sizeof(ascii_s) - 1)|| i == (len-1)) { 
        fprintf(stderr, "%sa:%.*s | hex:%.*s\n", prefix, j+1, ascii_s, 2*j+2, hex_s);
        i++;
        break;
      } else
        i++;
    }
  }
}

/* Handle input from client
 * will block until we get a full request
 *
 * return 0 on success, 1 on FD closed, -1 on failure */
int handle_client_request(int clt, int srv, const char * dn) {
    const char *method, *path;
    int pret, minor_version, i;
    struct phr_header headers[100];
    size_t buflen = 0, prevbuflen = 0, method_len, path_len, num_headers;
    unsigned long int content_length = 0;
    ssize_t nread;
#ifdef SQUID_FIX
    struct phr_header * host = NULL;
#endif

    memset(gbuf, 0, BUFSIZE); /* not mandatory here */

    prevbuflen = 0;
    nread = 0;

    while (1) {
    read_header:

        /* read the request (or a fragment of it) */
        while ((nread = read(clt, gbuf + buflen, BUFSIZE - buflen)) == -1 && errno == EINTR)
            ;
        if (nread < 0)
            my_perror_exit("Read FD: ", pid, errno);
        if (nread == 0 && errno != EINTR) {
          fprintf(stderr, "[%d] Read 0 byte from client request: %s\n", pid, strerror(errno));
          /* until I figure out how a read 0 could happen under normal
          circomstance, I'll consider that this is an error case */
          return 1; 
        }
        cread += nread;
        fprintf(stderr, "[%d] Read %zd bytes from client\n", pid, nread);

        prevbuflen = buflen;
        buflen += nread;

    resume_parse:

        /* parse the request */
        num_headers = sizeof(headers) / sizeof(headers[0]);
        pret = phr_parse_request(gbuf, buflen, &method, &method_len, &path, &path_len,
                &minor_version, headers, &num_headers, prevbuflen);
        if (pret > 0)
            break; /* successfully parsed the request */
        else if (pret == -1) {
            fprintf(stderr, "[%d] Unable to parse header[%d]\n", pid, pid);
            stream_print("Data-> ", gbuf, buflen);
            return -1; /* parse error */
        } else if (pret == -2) {
          fprintf(stderr, "[%d] Partial header, looping to obtain more data\n", pid);
          goto read_header; /* go read more data in order to parse header */
        }
        /* request is incomplete, continue the loop */
        if (buflen == BUFSIZE) {
          fprintf(stderr, "[%d] Request is too long (max length: %d)\n", pid, BUFSIZE);
          return -1;
        }
    }
    fprintf(stderr, "[%d] Request is %d bytes long\n", pid, pret);
    fprintf(stderr, "[%d] Method is %.*s\n", pid, (int)method_len, method);
    fprintf(stderr, "[%d] Path is %.*s\n", pid, (int)path_len, path);
    fprintf(stderr, "[%d] HTTP version is 1.%d\n", pid, minor_version);
    if (num_headers) {
      fprintf(stderr, "[%d] Headers:\n", pid);
      for (i = 0; i != num_headers; ++i) {
        fprintf(stderr, "[%d] - %.*s: %.*s\n", pid, (int)headers[i].name_len, headers[i].name,
                (int)headers[i].value_len, headers[i].value);
      }
    } else {
      fprintf(stderr, "[%d] No headers associated with the request\n", pid);
    }

    /* check method type */
    if (strincmp(method, "CONNECT", fmin(sizeof("CONNECT"), method_len)) == 0) {
      cs = NO_MORE_REQUEST;
    }

#if 0
    if (strincmp(method, "POST", fmin(sizeof("POST"), method_len)) == 0) {
      cs = NO_MORE_REQUEST;
    }
#endif

    /* reset variables */
    content_length = 0;

    /* parse header: check for "termination" cases */
    for (i = 0; i != num_headers; ++i) {
        if (headers[i].name_len == 0) continue;

        if (strincmp(headers[i].name, "Connection", fmin(headers[i].name_len, sizeof("Connection"))) == 0 &&
                strincmp(headers[i].value, "close", fmin(headers[i].value_len, sizeof("close"))) == 0) {
                cs = NO_MORE_REQUEST;
        }


#if 0
        /* request has a body, for now, we consider we won't have a new request after that (could be buggy!) */
        if (strincmp(headers[i].name, "Content-Type", fmin(headers[i].name_len, sizeof("Content-Type"))) == 0) {
              cs = NO_MORE_REQUEST;
        }
#endif

        if (strincmp(headers[i].name, "Content-Length", fmin(headers[i].name_len, sizeof("Content-Length"))) == 0) {
             /* we need to perform many checks before we can trust this user output */
             for(int j=0; j<headers[i].value_len; ++j) {
               if (!isdigit(headers[i].value[j])) {
                 fprintf(stderr, "[%d] Content lenght provided by client contains non-digit char, exiting\n", pid);
                 exit(1);
               }
             }
             if (headers[i].value[headers[i].value_len] != '\r')
               fprintf(stderr, "[%d] Content lenght does not end by CRLF\n", pid);

             content_length = strtoul(headers[i].value, NULL, 10);
             if (content_length == ULONG_MAX && errno == ERANGE) {
               fprintf(stderr, "[%d] Content lenght provided by client can't be stored in an unsigned long, exiting\n", pid);
             }
        }

        if (strincmp(headers[i].name, "SSL_CLIENT_VERIFY", fmin(headers[i].name_len, sizeof("SSL_CLIENT_VERIFY"))) == 0) {
            fprintf(stderr, "[%d] Client tried to add our custom header (SSL_CLIENT_VERIFY), exiting\n", pid);
            exit(1);
        }

        if (strincmp(headers[i].name, "SSL_CLIENT_S_DN", fmin(headers[i].name_len, sizeof("SSL_CLIENT_S_DN"))) == 0) {
            fprintf(stderr, "[%d] Client tried to add our custom header (SSL_CLIENT_S_DN), exiting\n", pid);
            exit(1);
        }

        /* size(STUNNEL_HEADER) - 2 because we want to skip the extra space at the end of the string */
        if (strincmp(headers[i].name, STUNNEL_HEADER, fmin(headers[i].name_len, sizeof(STUNNEL_HEADER) - 2)) == 0) {
          fprintf(stderr, "[%d] Client tried to add our custom header (X-Stunnel-ID), exiting\n", pid);
          exit(1);
        }

#ifdef SQUID_FIX
        if (strincmp(headers[i].name, "Host", fmin(headers[i].name_len, sizeof("Host"))) == 0) {
          host = &headers[i];
        }
#endif
    }

#define WRITE(fd, ptr, len) \
    do { \
        if (write(fd, ptr, len) == -1) { \
            my_perror_exit("write: ", pid, errno); \
        } \
        if (debug_lvl >= 1) stream_print("C-> ", ptr,len); \
    } while (0)

    /* fix relative URLs for GET in order to go through Squid

     The request:
     GET /yourpath HTTP/1.1
     Host: test.com

     Becomes:
     GET http://test.com/yourpath HTTP/1.1
     Host: test.com
     */
#ifdef SQUID_FIX
    if (strincmp(method, "GET", fmin(sizeof("GET"), method_len)) == 0  && \
         path_len != 0 && path[0] == '/' && host) {
        fprintf(stderr, "[%d] Path is relative, make it absolute\n", pid);

        WRITE(srv, method, (path - method));
        WRITE(srv, "http://", sizeof("http://") -1);
        WRITE(srv, host->value, host->value_len);
        WRITE(srv, path, (headers[0].name - path));
    } else
#endif
    /* send method */
    WRITE(srv, method, (headers[0].name - method));

    /* send our custom head */
    WRITE(srv, DN_HEADER, sizeof(DN_HEADER) -1);
    WRITE(srv, dn, strlen(dn));
    WRITE(srv, "\r\n", 2);
    WRITE(srv, VERIFY, sizeof(VERIFY) -1);
    WRITE(srv, STUNNEL_HEADER, sizeof(STUNNEL_HEADER) -1);
    WRITE(srv, stunnel_id, sizeof(stunnel_id));
    WRITE(srv, "\r\n", 2);

    /* send the rest of the request */
    if (num_headers)
      WRITE(srv, headers[0].name, pret - (headers[0].name - method));

    if (content_length) {
      if (buflen  - pret < content_length) { /* partial body content */
        long unsigned int  partial_content_length = buflen - pret;
        WRITE(srv, gbuf + pret, partial_content_length);
        content_length -= partial_content_length;
        while (content_length != 0) {
          /* read more body content (up to BUFSIZE) */
          /* note: we can overwrite gbuf here, because we are we won't
             reuse it in the function to process another HTTP request */ 
          while ((partial_content_length = read(clt, gbuf, content_length < BUFSIZE ? content_length : BUFSIZE)) == -1 && errno == EINTR) ;

          if (content_length == 0 && errno != EINTR) {
            fprintf(stderr, "[%d] Read 0 byte from client while reading partial fragment of the request body\n", pid);
            return 1;
          }

          cread += partial_content_length;

          WRITE(srv, gbuf, partial_content_length);
          content_length -= partial_content_length;
        }
        /* end of a HTTP request */
        return 0;
      } else { /* we got the whole body, and possibly some more */
        WRITE(srv, gbuf + pret, content_length);
        pret += content_length;
      }
    }

#undef WRITE

    /* we actually have pipelined requests */
    if (pret < buflen) {
        fprintf(stderr, "[%d] Pipelined request (already read %zu bytes from next request)\n", pid, buflen - pret);
        /* copy the end of the first buffer into second buffer */
        if (gbuf == gbuf1) {
            /* toggle buffer */
            memset(gbuf2, 0, BUFSIZE); /* not mandatory here */
            memcpy(gbuf2, gbuf1 + pret, buflen - pret);
            gbuf = gbuf2;
        } else {
            /* toggle buffer */
            memset(gbuf1, 0, BUFSIZE); /* not mandatory here */
            memcpy(gbuf1, gbuf2 + pret, buflen - pret);
            gbuf = gbuf1;
        }

        buflen -= pret;
        prevbuflen = 0;

        /* parse the new request */
        goto resume_parse;
    }

    return 0;
}

/* Handle stream passthrough
 * we don't actually need to process the response at all
 * so we simply copy everything to the client
 *
 * input: input FD
 * ouput: output FD
 *
 * return 0 on success, 1 on FD closed, -1 on failure */
int handle_passthrough(int input, int ouput) {
    ssize_t nread;
    memset(gbuf, 0, BUFSIZE);

    errno = 0;

    while ( (nread = read(input, gbuf, BUFSIZE)) == -1 && errno == EINTR)
      ;

    if (nread == -1) {
      /* input can be set to O_NONBLOCK to read the last piece of data
         on the socket.
         Here, we catch the case when there is nothing to read. */
      if (errno == EAGAIN || errno == EWOULDBLOCK)
        return -1;
      /* ECONNRESET can only come from a socket, hence we know its from the server */
      if (errno == ECONNRESET) {
        fprintf(stderr, "[%d] read failed: server closed its connexion\n", pid);
        return -1;
      }

      my_perror_exit("read fd: ", pid, errno);
    }

    if (nread == 0) {
      if (input == STDIN_FILENO)
        fprintf(stderr, "[%d] read 0 byte from client [errno=%d]\n", pid, errno);
      else
        fprintf(stderr, "[%d] read 0 byte from server [errno=%d]\n", pid, errno);

      /* we ought to have a closed fd here */
      return 1;
    }

    if (input == STDIN_FILENO)
      cread += nread;
    else
      sread += nread;

    if (write(ouput, gbuf, nread) == -1) {
         my_perror_exit("write: ", pid, errno);
    }
    if (debug_lvl >= 2) stream_print(input == STDIN_FILENO ? "C-> " : "S-> ", gbuf, nread);
    return 0;
}

/* handle to request (client) and reponses (server) */
void proxify(int clt, int srv, const char * dn) {
    int epfd, j, ret, flags;
    struct epoll_event ev = { 0 };
    struct epoll_event evlist[MAX_EVENT];

    epfd = epoll_create(2);
    if (epfd == -1) {
        my_perror_exit("epoll_create: ", pid, errno);
    }

    ev.events = EPOLLIN|EPOLLERR|EPOLLHUP|EPOLLRDHUP;
    ev.data.fd = clt;

    if (epoll_ctl(epfd, EPOLL_CTL_ADD, clt, &ev) == -1) {
         my_perror_exit("epoll_ctl - stdin: ", pid, errno);
    }

    ev.data.fd = srv;
    if (epoll_ctl(epfd, EPOLL_CTL_ADD, srv, &ev) == -1) {
         my_perror_exit("epoll_ctl - sock: ", pid, errno);
    }

    /* loop until the client or the server closes its side */
    while (1) {
        int ready = epoll_wait(epfd, evlist, MAX_EVENT, -1);
         if (ready == -1) {
             if (errno == EINTR) continue;
             else {
                  my_perror_exit("epoll_wait: ", pid, errno);
             }
         }
         for(j = 0; j < ready; j++) {
#if 0
             fprintf(stderr, "[%d] debug: playing with poll %d [%d]\n", pid, j, evlist[j].data.fd);
#endif
             ret = 0;

             if (evlist[j].events & EPOLLIN) {
                 errno = 0;

                 if (evlist[j].data.fd == clt) {
                     switch (cs) {
                         case MORE_REQUEST:
                            ret = handle_client_request(clt, srv, dn);
                            break;
                         case NO_MORE_REQUEST:
                            ret = handle_passthrough(clt, srv);
                            break;
                        default:
                            fprintf(stderr,  "[%d] State machine got corrupted, exiting\n", pid);
                            exit(1);
                     }
                 } else if (evlist[j].data.fd == srv) {
                     ret = handle_passthrough(srv, STDOUT_FILENO);
                 } else {
                      fprintf(stderr, "[%d] read from FD we don't know of ?!\n", pid);
                      exit(1);
                 }

                 switch (ret) {
                     case -1:
                         my_perror_exit("proxify: ", pid, errno);
                         break;
                     case 1: // FD is supposedely closed
                         if (evlist[j].data.fd == clt) {
                           /* If STDIN is closed, it means the client disconnected.
                              Hence, the work here is done (see the corresponding stunnel code) */
                           fprintf(stderr, "[%d] client disconnected after a read\n", pid);
                         } else {
                           fprintf(stderr, "[%d] server shut down its connection\n", pid);
                         }

                         /* we don't monitor events on this fd anymore */
                         if (epoll_ctl(epfd, EPOLL_CTL_DEL, evlist[j].data.fd, NULL) == -1) {
                           my_perror_exit("epoll_ctl_del: ", pid, errno);
                         }

                         goto empty_pipe;
                         break;
                 }
             } else if (evlist[j].events & (EPOLLHUP | EPOLLRDHUP | EPOLLERR)) {

               if (evlist[j].data.fd == clt) {
                 fprintf(stderr, "[%d] client disconnected\n", pid);
                 /* If STDIN is closed, it means the client disconnected.
                    Hence, the work here is done (see the corresponding stunnel code) */
               } else {
                 fprintf(stderr, "[%d] socket is closing\n", pid);
               }

               /* XXX: remove the following statement? */
               /* we don't monitor events on this fd anymore */
               if (epoll_ctl(epfd, EPOLL_CTL_DEL, evlist[j].data.fd, NULL) == -1) {
                 my_perror_exit("epoll_ctl_del: ", pid, errno);
               }

               goto empty_pipe;
             }
         }
    }
empty_pipe:
    /* read the last bits of data from the client and the server, if any */
    //flags = fcntl(srv, F_GETFL, 0);
    //fcntl(srv, F_SETFL, flags | O_NONBLOCK);
    flags = fcntl(clt, F_GETFL, 0);
    fcntl(clt, F_SETFL, flags | O_NONBLOCK);

    /* return value is 0 while reading succeed */
    while (handle_passthrough(clt, srv) == 0)
      ;
    while (handle_passthrough(srv, STDOUT_FILENO) == 0)
      ;
}

/* Create socket and connect it to the address specified by
  'host' + 'service'/'type'. Return socket descriptor on success,
  or -1 on error
  stolen from TLPI example code */
int
inetConnect(const char *host, const char *service, int type)
{
    struct addrinfo hints;
    struct addrinfo *result, *rp;
    int sfd = 0, s;

    memset(&hints, 0, sizeof(struct addrinfo));
    hints.ai_canonname = NULL;
    hints.ai_addr = NULL;
    hints.ai_next = NULL;
    hints.ai_family = AF_UNSPEC;        /* Allows IPv4 or IPv6 */
    hints.ai_socktype = type;

    s = getaddrinfo(host, service, &hints, &result);
    if (s != 0) {
        errno = ENOSYS;
        return -1;
    }

    /* Walk through returned list until we find an address structure
       that can be used to successfully connect a socket */

    for (rp = result; rp != NULL; rp = rp->ai_next) {
        sfd = socket(rp->ai_family, rp->ai_socktype, rp->ai_protocol);
        if (sfd == -1)
            continue;                   /* On error, try next address */

        if (connect(sfd, rp->ai_addr, rp->ai_addrlen) != -1)
            break;                      /* Success */

        /* Connect failed: close this socket and try next address */

        close(sfd);
    }

    freeaddrinfo(result);

    return (rp == NULL) ? -1 : sfd;
}

/* normalize DN string
 * - in_s: input string
 * - out_s: output string
 * - size: size of the output string
 *
 *   example:
 *   "C=FR, ST=Paris, L=Paris, O=ANSSI, OU=clip/test-gtw, CN=Gateway-1.clip.local"
 *   becomes:
 *   "/C=FR/ST=Paris/L=Paris/O=ANSSI/OU=clip/test-gtw/CN=Gateway-1.clip.local"
 *   */
int normalize_dn(const char * in_s, char * out_s, unsigned int size) {
    int i = 0, j = 0;
    char last = '\0'; /* last char we read */
    memset(out_s, 0, size);

    /* output string is too small */
    if (size <= 1) return -1;

    if (in_s[0] != '/') {
        out_s[0] = '/';
        j = 1;
    }

    for (i=0; j < size && in_s[i] != '\0'; i++, j++) {
        if(last == ',' && in_s[i] == ' ') {
            --j; /* overwrite last period we copied */
            out_s[j] = '/';
            continue;
        }
        last = out_s[j] = in_s[i];
    }

    return 0;
}

int main(int argc, char *argv[])
{
    int opt, sock, verbose = 0;
    char * hostname = NULL;
    char * port = NULL;
    char * dn = NULL; /* distinguished name of certificate */

    struct timeval time = { 0 };

    pid = getpid();

    if (argc == 1) {
        usage_and_exit(argv[0], 1);
    }

    while ((opt = getopt(argc, argv, "c:dp:hv")) != -1 ) {
        switch (opt) {
            case 'c':
                hostname = optarg;
                break;
            case 'd':
                debug_lvl++;
                break;
            case 'p':
                port = optarg;
                break;
            case 'h':
                usage_and_exit(argv[0], 0);
                break;
            case 'v':
                verbose=1;
                break;
            default:
                fprintf(stderr, "[%d] Unrecognized option: %c\n", pid, opt);
                usage_and_exit(argv[0], 1);
        }
    }

    if (hostname == NULL || port == NULL) {
         fprintf(stderr, "[%d] options are missing\n", pid);
         usage_and_exit(argv[0], 1);
    }

    dn=getenv(DN_NAME);
    if (dn == NULL) {
        fprintf(stderr, "[%d] Variable %s is missing from environment\n", pid, DN_NAME);
        exit(1);
    }

    if (normalize_dn(dn, normalized_dn, DN_MAXSIZE) != 0) {
        fprintf(stderr, "[%d] Unable to normalized DN: %s\n", pid, dn);
        exit(1);
    }

    if (! verbose) less_verbose();

    fprintf(stderr, "[%d] Establishing connection for \"%s\"\n", pid, normalized_dn);

    /* connecting to remote host */
    sock = inetConnect(hostname, port, SOCK_STREAM);
    if (sock <= 0) {
        my_perror_exit("socket: ", pid, errno);
    }

    /* initialize stunnel ID
       This is a random value that is drawn for each new connexion.
       This helps us track the request on the downstream proxies */
    gettimeofday(&time, NULL);
    srandom(time.tv_sec * 1000000 + time.tv_usec);
    for(int i=0; i<sizeof(stunnel_id); ++i) {
      stunnel_id[i] = (random() % 10) + '0'; /* does not need to be cryptographically strong */
    }

    fprintf(stderr, "[%d] Stunnel ID for the session is: %.*s\n", pid, (int) sizeof(stunnel_id), stunnel_id);

    fprintf(stderr, "[%d] Entering proxify mode\n", pid);
    proxify(STDIN_FILENO, sock, normalized_dn);

    fprintf(stderr, "[%d] Client data received: %llu; Server data received: %llu\n", pid, cread, sread);
    fprintf(stderr, "[%d] Program exiting normally\n", pid);

    /* cleaning up */
    close(STDIN_FILENO); 
    shutdown(sock, SHUT_RDWR);

    return 0;
}
