
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; MODULE      : spreadsheet0.scm
;; DESCRIPTION : Example plugin for a simple spreadsheet
;; AUTHOR      : Henri Lesourd, Uni Saarbrucken, 2007.
;;
;; This software falls under the GNU general public license and comes WITHOUT
;; ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for details.
;; If you don't have this file, write to the Free Software Foundation, Inc.,
;; 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(texmacs-module (spreadsheet0))

;; Some utilities (put that in some file somewhere)
(define-macro (foreach i . b)
  `(for-each (lambda
                (,(car i))
                ,(cons 'begin b))
            ,(cadr i)))

(define-macro (foreach-number i . b)
  `(do ((,(car i) ,(cadr i)
        (,(if (memq (caddr i) '(> >=)) '- '+) ,(car i) 1))
       )
       ((,(if (eq? (caddr i) '>)
             '<=
              (if (eq? (caddr i) '<)
                 '>=
                  (if (eq? (caddr i) '>=) '< '>)))
         ,(car i) ,(cadddr i)
        )
       ,(car i)
       )
      ,(cons 'begin b)))

;; Traversing trees
(define (path-traverse-up p func)
 ;(display* "p=" p "\n")
  (if (func (path->tree p))
      p
      (if (> (length p) 2)
          (path-traverse-up (reverse (cdr (reverse p))) func)
          #f)))

(define (path-upper p tags)
  (path-traverse-up p (lambda (t) (in? (tree-label t) tags))))

(tm-define (tree-traverse t func)
 ;(display "Traverse\n")
  (if (eq? (tree-label t) 'string)
      #t
      (foreach-number (i 0 < (tree-arity t))
         (with child (tree-ref t i)
            (func child)
            (tree-traverse child func)))))

;; An example of tree-traversal
(tm-define (xmlize-tree t ind)
; Try this using (xmlize-tree (path->tree '(0)) 0)
; from a Scheme session inside TeXmacs
  (define (indent ind)
     (foreach-number (i 0 < ind)
        (display " "))
  )
  (indent ind)
  (if (tree? t)
      (if (tree-atomic? t)
          (if (== (tree-arity t) 0)
              (display* "<emptyline/>\n")
              (display* (tree->stree t) "\n"))
          (let* ((n (tree-arity t))
                 (l (tree-label t))
             )
             (cond
                ((== l 'point)
                 (display* "<point x1='" (tree->stree (tree-ref t 0))
                                "' x2='" (tree->stree (tree-ref t 1))
                                "'/>\n"))
                ((== l 'with)
                   (display* "<with>")
                   (foreach-number (i 0 < (/ (- n 1) 2))
                      (display "\n")(indent (+ ind 2))
                      (display* "<attribute name='" (tree->stree (tree-ref t (* 2 i))) "'>")
                      (display "\n")
                      (xmlize-tree (tree-ref t (+ (* 2 i) 1)) (+ ind 4))
                      (indent (+ ind 2))
                      (display* "</attribute>")
                   )
                   (display* "\n")
                   (xmlize-tree (tree-ref t (- n 1)) (+ ind 2))
                   (indent ind)
                   (display* "</with>\n"))
                (else
                   (display* "<" (tree-label t) ">\n")
                   (foreach-number (i 0 < (tree-arity t))
                      (xmlize-tree (tree-ref t i) (+ ind 2))
                     ;(display* " ")
                   )
                   (indent ind)
                   (display* "</" (tree-label t) ">\n")
                ))))))

;; Reload
(tm-define (spreadsheet0-reload) ;; For interactive development from inside TeXmacs
  (display "Reloading...\n")
  (load (string-append (getenv "TEXMACS_HOME_PATH") "/plugins/spreadsheet0/progs/spreadsheet0.scm")))

;; Keyboard events
(tm-define (in-spreadsheet0?)
  (path-upper (cDr (cursor-path)) '(spreadsheet))) ;; Try this one, at some point

(tm-define (in-spreadsheet0?)
  #t)

(kbd-map
  (:mode in-spreadsheet0?)
;; Shift-Ctrl-H
  ("C-H"
     (begin
        (display* "cp=" (cursor-path) "\n")
        (tree-set! (path->tree (cDDr (cursor-path))) 0 "Hello"))) ;; tree-set! example
;; Shift-Ctrl-I
  ("C-I" (tree-insert (path->tree (cDr (cursor-path))) 0 "Hello2")) ;; tree-insert example
;; Shift-Ctrl-J
  ("C-J" (insert-go-to '(tuple "abcde" "fghij" "klmno") '(1 2))) ;; insert-go-to example
;; Ctrl-J
  ("C-j" (insert "Hello !"))
;; Shift-R
  ("C-r" (tree-remove (path->tree (cDDr (cursor-path))) (cAr (cDr (cursor-path))) 1)) ;; tree-remove example
;; Shift-Tab
  ("S-tab" (toggle-preamble))
;; Shift-Ctrl-X
  ("C-X"
     (with p (path-upper (cDr (cursor-path)) '(spreadsheet))
        (if p
            (xmlize-tree (path->tree p) 0)
            (display* "No embedding <spreadsheet|...> tag !\n"))))
;; Ctrl-X
  ("C-x" (begin
            (write (tree->stree (path->tree `(,(car (cursor-path))))))
            (newline)))
;; Shift-Ctrl-R
  ("C-R" (spreadsheet0-reload)))

;; Initialization
(tm-define (spreadsheet0-initialize)
  (if (not spreadsheet0-initialized)
  (begin
     (set! spreadsheet0-initialized #t)))
  (display* "Loading Spreadsheet0...\n"))

(if (not (defined? 'spreadsheet0-initialized))
    (define-public spreadsheet0-initialized #f))
