;;; hhelp.scm
;;;
;;; TeXmacs procedures to help me learn how to use TeXmacs procedures ;)
;;;
;;; See: hhelp/doc/hhelp-help.tm (which can be loaded with C-h C-h)
;;; 
;;; Jarvis's notes:
;;; * Write code for the automatic generation of documentation.
;;; * Need `describe-key', `describe-function', etc.
;;; * Look closer at:
;;;   a. (kbd-find-inv-binding com)
;;;   b. (kbd-find-key-binding key)
;;;   c. (kbd-get-command key)
;;; * Filters:
;;;   a. Only search keystrokes specific to current mode
;;;   b. Figure out a way to group keybindings.
;;; * Something for the automatic generation of help for functions.
;;;
;;; S. Edward Dolan <bytecolor@yahoo.com>
;;; Wednesday, August 12 2009

(define texmacs-help help)

(texmacs-module (hhelp)
  (:use (ice-9 format)
        (ice-9 ls)                      ; XXX may not need this any more
        (ice-9 regex)
        (ice-9 pretty-print)
        (ice-9 session))
  ;; (help) from TeXmacs, not (ice-9 session) (does not seem to work)
  (:duplicates (noop)))

;; I can not figure out how to display a message box informing the user
;; when she gives a malformed regex.
;; It seems the catch code is working correctly.
(define (bad-re-handler . args)
  "Handle malformed regexps obtained from the user."
  ;(:synopsis "Handle malformed regexps obtained from the user.")
   (format #t "~A~%" args))

;; TODO Return the list sorted by name, type, module?
(tm-define (symbol-apropos-info regex)
  (:synopsis "Find all symbols that match REGEX. Collect information about
those symbols. Return a list of the form:
 ((name value module) (name value module) ...)")
  (let ((outlst '()))
    (apropos-fold
     (lambda (m n v d)
       (set! outlst (cons `(,n ,v ,m) outlst)))
     #f regex apropos-fold-all)
    outlst))

(tm-define (get-fn-docstring fn)
  (:synopsis "Get the document string for the given function: FN")
  (cond
   ((procedure-documentation fn))
   ((texmacs-help fn) => car)
   (else "Not documented.")))

(tm-define (get-source-as-indented-string fn)
  (:synopsis "Pretty print the given FN into a string. Return that
string. Default width is 79 columns.")
  (string-trim-both (with-output-to-string
                      (lambda ()
                        (pretty-print (procedure-source fn))))))

(tm-define (get-source-as-string-list fn)
  (:synopsis "Return a list of FN source lines.
e.g. (\"(define (foo x)\" \"  x)\")")
  (string-split (get-source-as-indented-string fn) #\Newline))

;; TODO look at: (symbol-binding o s)
(tm-define (get-symbol-value s)
  (:synopsis "Return the value of the symbol S")
  (cond ((string? s)
         (eval (string->symbol s) (current-module)))
        ((symbol? s)
         (eval s (current-module)))
        (else
         "(get-symbol->value s) Unknown symbol. Should not be here!")))

(tm-define (make-tm-doc-fn-data title name apropos-info docstring source)
  (:synopsis "Create a data tree that can be inserted into a tm-doc body. The
tree contains any information collected on the symbol NAME. APROPOS-INFO is
the value returned from SYMBOL-APROPOS-INFO. DOCSTRING is the value returned
from GET-FN-DOCSTRING. SOURCE is the value returned from ROCEDURE-SOURCE.")
  (let* ((mod-name (object->string (module-name (caddar apropos-info))))
         (printed-rep (object->string (cadar apropos-info)))
         (func-type (substring printed-rep
                               2 (string-index printed-rep #\Space)))
         (func-source (get-source-as-string-list
                       (get-symbol-value name))))
    `(document
      (tmdoc-title ,title)
      (tabular
       (tformat (cwith "1" "-1" "1" "1" "cell-halign" "r")
                (cwith "1" "-1" "1" "1" "cell-valign" "t")
                (cwith "1" "-1" "2" "2" "cell-valign" "t")
                (table (row (cell (strong "Docstring"))
                            (cell ,docstring))
                       (row (cell (strong "Type"))
                            (cell ,func-type))
                       (row (cell (strong "Module"))
                            (cell ,mod-name)))))
      (scm-fragment
       (document ,@func-source)))))

(tm-define (make-unbound-symbol-document symbol-name)
  (display-in-help-buffer
   (string-append "Unbound symbol: " symbol-name)
   (string-append "The symbol " symbol-name " is unbound")))

(tm-define (describe-function fn-name)
  (:synopsis "Find procedure given by the string FN-NAME. Collect information
about the procedure:

 * Guile document string or TeXmacs :synopsis
 * type: procedure, macro, closure, etc.
 * originating module (foo bar baaz)
 * source code of the function

Display this information in a Help buffer.")
  (:interactive #t)
  (let* ((quoted-fn-name (regexp->match fn-name))
         (apropos-info (symbol-apropos-info quoted-fn-name)))
    (cond ((null? apropos-info)
           (make-unbound-symbol-document fn-name))
          (else
           (let* ((sym-fn (eval (string->symbol fn-name)
                                (current-module)))
                  (docstring (get-fn-docstring sym-fn))
                  (source-code (procedure-source sym-fn))
                  (title (string-append "Information on function: " fn-name)))
             (display-in-help-buffer
              title
              (make-tm-doc-fn-data
               title
               fn-name apropos-info docstring
               source-code)))))))

(tm-define (display-in-help-buffer title body-data)
  (:synopsis "Create and load a doc with TITLE and BODY-DATA")
  (set-help-buffer
   title
   `(document (style "tmdoc")
              (body ,body-data)
              (initial (collection
                        (associate "language" ,(get-output-language)))))))

(define (print-type type name value)
  (format #t "~A: ~A ~A~%" type name value))

(define (regexp->match name)
  "Escape special regex characters in NAME, add prefix ^ and suffix $. This
should produce a string that will match NAME exactly."
  (string-append "^" (regexp-quote name) "$"))

(tm-define (describe-hook name)
  (let ((hook-info (symbol-apropos-info (regexp->match name))))
    (format #t "hook-info: ~A~%" hook-info)))

(tm-define (display-symbol-doc name value)
  (:synopsis "Generate and display any information about the symbol S in the
Help Buffer.")
  (:secure #t)
  (cond ((procedure? value)
         (describe-function name))
        ((macro? value)
         (print-type "macro" name value))
        ((primitive-macro? value)
         (print-type "primitive-macro" name value))
        ((hook? value)
         (describe-hook name))
        ((list? value)
         (print-type "list" name value))
        (else
         (print-type "unknown type" name value))))

;; (action "foo" "(display 'foo)")
(define (make-action label func . args)
  `(action ,label ,(format #f "(~A ~{~A~^~_~})" func args)))

;; Emacs `apropos-command' takes a few arguments that could be added. At
;; present a symbol is a symbol is a symbol; No filter for funcs, vars, or
;; even the ability to search only those symbols TeXmacs has defined.
(tm-define (apropos-command regex)
  (:synopsis "Find all symbols that match REGEX. Display a sorted list in the
Help Buffer. Each item in the list is a link which calls DISPLAY-SYMBOL-DOC")
  (:interactive #t)
  (let ((symre #f))
    ;; catch a malformed regex
    (catch
     'regular-expression-syntax
     (lambda ()
       (when (!= regex "")
             (let* ((symre (make-regexp regex))
                    (matched-symbol-list
                     (sort (filter (lambda (sym)
                                     (regexp-exec symre
                                                  (symbol->string sym)))
                                   (ls))
                           symbol<=?))
                    (list-items
                     (map (lambda (found-sym)
                            `(concat
                              (item)
                              ,(make-action (symbol->string found-sym)
                                            "display-symbol-doc"
                                            (string-append "\""
                                                           (symbol->string
                                                            found-sym)
                                                           "\"")
                                            found-sym)))
                          matched-symbol-list))
                    (partial-title (format #f "~D result~:p for: "
                                           (length matched-symbol-list))))
               ;; generate and display the doc
               (display-in-help-buffer
                (string-append partial-title regex)
                `(document
                  (tmdoc-title
                   (concat
                    ,partial-title
                    (strong ,regex)))
                  (enumerate
                   (document
                    ,@list-items)))))))
     bad-re-handler)))

;;; ========================================================================
;;; Key functions

(tm-define (self-inserting-char? key-seq)
  (:synopsis "Return non #f if key-seq is a character that would be displayed
as-is, such as S-r, u, etc., in the current mode.")
  (or (= (string-length key-seq)
         1)
      (and (= (string-length key-seq) 3)
           (string-starts? key-seq "S-"))))

(tm-define (get-kbd-key-info key-seq)
  (:synopsis "Generate documentation for the given KEY-SEQ.")
  (let ((key-val (kbd-find-key-binding key-seq)))
    (if key-val
        ;; key-seq is bound to `something'
        (cond ((procedure? (car key-val))
               (format #f "~A" (procedure-source (car key-val))))
              ((string? (car key-val))
               (format #f "Prefix: ~A" key-val))
              (else
               (format #f "Unknown Key Sequence: ~A" key-val)))
        ;; key-seq is unbound or a literal character
        (if (self-inserting-char? key-seq)
            "Simply insert the character"
            (format #f "~A is unbound in the current context." key-seq)))))

(tm-define (show-kbd-key-info key-seq)
  (:synopsis "Collect any information about the given KEY-SEQ and display the
results in a help buffer.")
  (:interactive #t)
  (let ((doc-dat #f)
        (doc-title (format #f "Information on Key Sequence: ~A" key-seq)))
    ;; for some odd reason the key-seq "`" is producing #\nul
    (if (string-index key-seq #\nul)
        (begin
          (set! doc-title "Internal Error")
          (set! doc-dat "Bug in hhelp.scm: show-kbd-key-info: #\\nul in str."))
        (set! doc-dat (get-kbd-key-info key-seq)))
    (display-in-help-buffer doc-title doc-dat)))


;; I'm not sure if this is how to bind `C-h a'. It seems to work, though :)
(kbd-wildcards pre
               ("emacs" "C-")               
               ("emacs:prefix-help" "emacs h"))

(kbd-map
 (:mode prevail?)                       ; supersede any other binding?
 ("emacs:prefix-help" "" "Emacs Help prefix command.")
 ("emacs:prefix-help a" (interactive apropos-command "Symbol Regex:")
  "Show symbols that match the given regular expression.")
 ("emacs:prefix-help d" (interactive docgrep-in-doc)
  "Search the TeXmacs documentation.")
 ("emacs:prefix-help f" (interactive describe-function "Function Name:")
  "Display information about the given function.")
 ("emacs:prefix-help k" (interactive show-kbd-key-info "Key Sequence:")
  "Display info on the given key sequence.")
 ("emacs:prefix-help t" (load-help-buffer "tutorial/tut-tutorial")
  "Start the TeXmacs tutorial.")
 ("emacs:prefix-help C-a" (load-help-article "about/about-summary")
  "Show TeXmacs information.")
 ("emacs:prefix-help C-c" (load-help-buffer "$TEXMACS_PATH/LICENSE")
  "Show TeXmacs license.")
 ("emacs:prefix-help C-h"
  (load-help-buffer "$TEXMACS_HOME_PATH/plugins/hhelp/doc/hhelp-help.tm")
  "Show the hhelp plugin documentation.")
 ("emacs:prefix-help C-n" (load-help-article "about/changes/change-log")
  "Show TeXmacs Change Log.")
 ("emacs:prefix-help C-t" (load-help-article "about/projects/future")
  "Show TeXmacs future plans."))
