/**
 * Copyright (c) 2004, Technische Universitaet Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universitaet Berlin nor the names
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Revision -------------------------------------------------------------
 * $Revision: 1.1.2.9 $
 * $Date: 2006/01/31 18:43:40 $
 * @author: Jan Hauer <hauer@tkn.tu-berlin.de>
 * ========================================================================
 */

/**
 * Please refer to TEP 101 for more information about this component and its
 * intended use. This component represents the HAL2 of the MSP430 ADC12
 * subsystem. Clients SHOULD NOT wire to <code>AdcC</code> directly but should
 * go via <code>AdcReadClientC</code>, <code>AdcReadNowClientC</code> or
 * <code>AdcReadStreamClientC</code>.
 *
 * @author Jan Hauer
 */

module AdcC {
  provides {
    interface Read<uint16_t> as Read[uint8_t client];
    interface ReadNow<uint16_t> as ReadNow[uint8_t client];
    interface ReadStream<uint16_t> as ReadStream[uint8_t rsClient];
  }
  uses {
    // for Read only:
    interface Resource as Resource[uint8_t client];
    // for Read and ReadNow:
    interface Msp430Adc12Config as Config[uint8_t client];
    interface Msp430Adc12 as Msp430Adc12[uint8_t client];
    // for ReadStream only:
    interface Msp430Adc12Config as ConfigReadStream[uint8_t rsClient];
    interface Msp430Adc12 as Msp430Adc12ReadStream[uint8_t rsClient];
    interface Resource as ResourceReadStream[uint8_t rsClient];
  }
}
implementation
{
  enum { // state
    READ,
    READ_NOW,
    READ_STREAM
  };

  struct list_entry_t {
    uint16_t count;
    struct list_entry_t *next;
  };

  // Resource interface makes norace safe
  norace uint8_t state;
  norace uint16_t value;
  // atomic section in postBuffer() makes norace safe
  norace struct list_entry_t *streamBuf[uniqueCount(ADCC_READ_STREAM_SERVICE)];
  uint32_t usPeriod[uniqueCount(ADCC_READ_STREAM_SERVICE)];
  uint8_t owner;
  bool ignore;
  uint16_t *resultBuf;
  msp430adc12_channel_config_t streamSettings;

  void task finishStreamRequest();
  void task signalBufferDone();
  void nextReadStreamRequest(uint8_t rsClient);

  command error_t Read.read[uint8_t client]()
  {
    return call Resource.request[client]();
  }

  async command error_t ReadNow.read[uint8_t client]()
  {
    msp430adc12_channel_config_t *settings;
    error_t hal1request;

    settings = call Config.getSettings[client]();
    if (!settings || !settings->count)
      return EINVAL; // Config not wired ?!

    // There is no automatic Resource reservation for ReadNow,
    // but startSingle() will fail if the client has not
    // reserved, because HAL1 checks ownership at runtime
    hal1request = call Msp430Adc12.start[client](settings);
    if (hal1request == SUCCESS)
      state = READ_NOW;
    return hal1request;
  }

  event void Resource.granted[uint8_t client]()
  {
    // signalled only for Read
    msp430adc12_channel_config_t *settings;
    error_t hal1request;

    settings = call Config.getSettings[client]();
    if (!settings || !settings->count) {
      call Resource.release[client]();
      signal Read.readDone[client](EINVAL, 0);
      return;
    }
    hal1request = call Msp430Adc12.start[client](settings);
    if (hal1request == SUCCESS) {
      state = READ;
      owner = client;
    } else {
      call Resource.release[client]();
      signal Read.readDone[client](FAIL, 0);
    }
  }

  void task readDone()
  {
    call Resource.release[owner]();
    signal Read.readDone[owner](SUCCESS, value);
  }

  async event bool Msp430Adc12.singleReady[uint8_t client](uint16_t data)
  {
    switch (state)
    {
      case READ:
        value = data;
        post readDone();
        break;
      case READ_NOW:
        if (ignore == TRUE)
          ignore = FALSE;
        else
          signal ReadNow.readDone[client](SUCCESS, data);
        break;
      default:
        break;
    }
    return SUCCESS;
  }

  async event bool Msp430Adc12.sequenceReady[uint8_t client](
      uint16_t *data, uint8_t len)
  {
    // won't happen
    return FALSE;
  }


  command error_t ReadStream.postBuffer[uint8_t rsClient]( uint16_t* buf, uint16_t count )
  {
    struct list_entry_t *newEntry = (struct list_entry_t *) buf;

#ifdef CHECK_ARGS
    if (!buf || !count)
      return FAIL;
#endif

    newEntry->count = count;
    newEntry->next = 0;
    atomic {
      if (!streamBuf[rsClient])
        streamBuf[rsClient] = newEntry;
      else {
        struct list_entry_t *tmp = streamBuf[rsClient];
        while (tmp->next)
          tmp = tmp->next;
        tmp->next = newEntry;
      }
    }
    return SUCCESS;
  }

  command error_t ReadStream.read[uint8_t rsClient]( uint32_t _usPeriod )
  {
    if (!streamBuf[rsClient] ||
        call ResourceReadStream.request[rsClient]() != SUCCESS)
      return FAIL;
    usPeriod[rsClient] = _usPeriod;
    return SUCCESS;
  }

  event void ResourceReadStream.granted[uint8_t rsClient]()
  {
    msp430adc12_channel_config_t *settings;

    owner = rsClient;
    settings = call ConfigReadStream.getSettings[rsClient]();
    if (!settings || !settings->count) {
      post finishStreamRequest();
      return;
    }
    streamSettings = *settings;
    /* assumption: SMCLK runs at 1 MHz */
    streamSettings.sampcon_ssel = SAMPCON_SOURCE_SMCLK;
    streamSettings.sampcon_id = SAMPCON_CLOCK_DIV_1;
    if (call Msp430Adc12ReadStream.startRepeat[rsClient](&streamSettings,
        usPeriod[rsClient]) != SUCCESS) {
      post finishStreamRequest();
      return;
    }
  }

  void task finishStreamRequest()
  {
    if (!streamBuf[owner]) {
      call ResourceReadStream.release[owner]();
      signal ReadStream.readDone[owner](SUCCESS, usPeriod[owner]);
    } else {
      do {
        struct list_entry_t *entry = streamBuf[owner];
        streamBuf[owner] = streamBuf[owner]->next;
        signal ReadStream.bufferDone[owner](FAIL, (uint16_t *)entry, 0);
      } while (streamBuf[owner]);
      call ResourceReadStream.release[owner]();
      signal ReadStream.readDone[owner](FAIL, 0);
    }
  }

  async event bool Msp430Adc12ReadStream.sequenceReady[uint8_t rsClient](
      uint16_t *data, uint8_t len)
  {
    if (!resultBuf) {
      struct list_entry_t *entry = streamBuf[rsClient];
      if (entry) {
        streamBuf[rsClient] = entry->next;
	resultBuf = (uint16_t *)entry;
        value = (len <= entry->count) ? len : entry->count;
        memcpy(resultBuf, data, value * sizeof(uint16_t));
        post signalBufferDone();
        return TRUE;
      }
    }
    post finishStreamRequest();
    return FALSE;
  }

  void task signalBufferDone()
  {
    uint16_t *tmp;
    atomic {
      tmp = resultBuf;
      resultBuf = 0;
    }
    signal ReadStream.bufferDone[owner]( SUCCESS, tmp, value);
  }

  async event bool
      Msp430Adc12ReadStream.singleReady[uint8_t rsClient](uint16_t data)
  {
    // won't happen
    return FALSE;
  }

  default async command error_t Resource.request[uint8_t client]() { return FAIL; }
  default async command error_t Resource.immediateRequest[uint8_t client]() { return FAIL; }
  default async command error_t Resource.release[uint8_t client]() { return FAIL; }
  default event void Read.readDone[uint8_t client]( error_t result, uint16_t val ){}
  default async event void ReadNow.readDone[uint8_t client]( error_t result, uint16_t val ){}

  default async command error_t ResourceReadStream.request[uint8_t rsClient]() { return FAIL; }
  default async command error_t ResourceReadStream.release[uint8_t rsClient]() { return FAIL; }
  default event void ReadStream.bufferDone[uint8_t rsClient]( error_t result,
			 uint16_t* buf, uint16_t count ){}
  default event void ReadStream.readDone[uint8_t rsClient]( error_t result, uint32_t actualPeriod ){ }

  default async command error_t
    Msp430Adc12.start[uint8_t client](
        const msp430adc12_channel_config_t *config)
  {
    return EINVAL;
  }

  default async command error_t
    Msp430Adc12.startRepeat[uint8_t client](
        const msp430adc12_channel_config_t *config, uint16_t jiffies)
  {
    return EINVAL;
  }

  default async command msp430adc12_channel_config_t
      *Config.getSettings[uint8_t client]()
  {
    return 0;
  }

  default async command error_t
    Msp430Adc12ReadStream.start[uint8_t client](
      const msp430adc12_channel_config_t *config)
  {
    return EINVAL;
  }

  default async command error_t
    Msp430Adc12ReadStream.startRepeat[uint8_t client](
      const msp430adc12_channel_config_t *config, uint16_t jiffies)
  {
    return EINVAL;
  }

  default async command msp430adc12_channel_config_t
      *ConfigReadStream.getSettings[uint8_t client]()
  {
    return 0;
  }
}

