/*
 * Copyright (c) 2004, Technische Universitaet Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universitaet Berlin nor the names
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Revision -------------------------------------------------------------
 * $Revision: 1.1.2.6 $
 * $Date: 2006/01/31 18:43:40 $
 * @author: Jan Hauer <hauer@tkn.tu-berlin.de>
 * ========================================================================
 */

#ifndef MSP430ADC12_H
#define MSP430ADC12_H
#include "Msp430RefVoltGenerator.h"

#if 1 /* FIXME: we need a better way ... */
#undef P6PIN_AUTO_CONFIGURE
#undef REF_VOLT_AUTO_CONFIGURE
#else
#define P6PIN_AUTO_CONFIGURE
#define REF_VOLT_AUTO_CONFIGURE
#endif
#define CHECK_ARGS

/*
 * The msp430adc12_channel_config_t encapsulates all relevant flags for
 * sampling a single ADC12 channel on a per-client basis.  They are taken from
 * the following MSP430 registers: ADC12CTL0, ADC12CTL1, ADC12MCTLx and TACTL
 * of TimerA (if applicable) and named according to section "17.3 ADC12
 * Registers" of the "MSP430x1xx Family User's Guide",
 * http://focus.ti.com/lit/ug/slau049e/slau049e.pdf).
 *
 * For msp430adc12_channel_config_t
 * ================================
 *
 * .ref2_5v: Reference generator voltage level (ADC12CTL0 register). See
 *        explanation to "sref" flag.
 *
 * .adc12ssel: ADC12 clock source select for the sample-hold time (ADC12CTL1
 *        register). In combination the "adc12ssel", "adc12div" and "sht"
 *        define the sample-hold-time: "adc12ssel" defines the clock source,
 *        "adc12div" defines the ADC12 clock divider and "sht" defines the
 *        sample-hold time expressed in jiffies (ticks of the divided clock).
 *
 * .adc12div: ADC12 clock divider (ADC12CTL1 register). See "adc12ssel".
 *
 * .sht: Sample-and-hold time (ADC12CTL1 register). See "adc12ssel".
 *
 * .sampcon_ssel: In combination with "sampcon_id" and the "jiffies"
 *        parameter in the Msp430Adc12 interface, "sampcon_ssel" defines the
 *        sampling rate (TASSEL in TACTL register, TimerA).  It is the clock
 *        source for the SAMPCON signal, which triggers the actual sampling. It
 *        is ignored when Msp430Adc12.startSingle() is used or if the "jiffies"
 *        parameter is zero.  Otherwise the SAMPCON signal is sourced from
 *        TimerA so that the multiple conversion mode can be made with the user
 *        defined sampling rate.
 *
 * .sampcon_id: Input divider for "sampcon_ssel"  (IDx in TACTL register,
 *        TimerA). See "sampcon_ssel".
 *
 * .count: The number of channels in the sequence to be converted.
 *
 * .channels: For a single-channel conversion, "count" is 1 and the "channels"
 *        union contains "single", an adcmemctl_t describing the channel.  If
 *        "count" > 1 then the union holds "sequence", a pointer to an array
 *        of adcmemctl_t's, one for each channel in the sequence.
 *
 * For adcmemctl_t
 * ===============
 *
 * .inch: ADC12 input channel (ADC12MCTLx register). An (external) input channel
 *        maps to one of msp430's A0-A7 pins (see device specific data sheet).
 *
 * .sref: reference voltage (ADC12MCTLx register). If
 *        REFERENCE_VREFplus_AVss or REFERENCE_VREFplus_VREFnegterm is chosen
 *        AND the client wires to Msp430Adc12RefVoltAutoClientC (or
 *        REF_VOLT_AUTO_CONFIGURE is defined and the client wires to
 *        AdcReadClientC, AdcReadNowClientC or AdcReadStreamClientC) then the
 *        reference voltage generator is automatically switched on to the
 *        voltage level defined by the "ref2_5v" flag (see below) whenever the
 *        client accesses the ADC12. Otherwise both flags are ignored.
 *
 *
 *                   **********************************
 *
 * EXAMPLE: Say we want to sample a sequence of 4 ADC channels 100 times per
 * second.  Assuming that SMCLK runs at 1 MHz the following commands return
 * conversion results for channels A0 through A3 (count = 4) with conversion of
 * the sequence 100 times per second.  Because count > 1, the samples are
 * returned by the Msp430Adc12 signalling event sequenceReady() instead of
 * singleReady().  Note that the sampling rate is defined by the combination of
 * SAMPCON_SOURCE_SMCLK, SAMPCON_CLOCK_DIV_1, the count of channels (4) and the
 * "jiffies" parameter of 2500.  The formula for determining jiffies is:
 *
 *   jiffies = (SAMPCON_CLOCK / SAMPCON_CLOCK_DIV) / (SAMPLE_FREQ * CHAN_COUNT)
 *
 * adcmemctl_t config_sequence[] = {
 *   { inch: INPUT_CHANNEL_A0, sref: REFERENCE_VREFplus_AVss },
 *   { inch: INPUT_CHANNEL_A1, sref: REFERENCE_VREFplus_AVss },
 *   { inch: INPUT_CHANNEL_A2, sref: REFERENCE_VREFplus_AVss },
 *   { inch: INPUT_CHANNEL_A3, sref: REFERENCE_VREFplus_AVss }
 * };
 *
 * msp430adc12_channel_config_t config = {
 *   rev2_5v: REFVOLT_LEVEL_1_5,
 *   adc12ssel: SHT_SOURCE_SMCLK,
 *   adc12div: SHT_CLOCK_DIV_1,
 *   sht: SAMPLE_HOLD_64_CYCLES,
 *   sampcon_sssel: SAMPCON_SOURCE_SMCLK,
 *   sampcon_id: SAMPCON_CLOCK_DIV_1,
 *   count: sizeof(config_sequence)/sizeof(adcmemctl_t),
 *   channels: { sequence: config_sequence }
 * };
 *
 * event void Resource.granted()
 * {
 *  if (call Msp430Adc12.startRepeat(config, channels, numChannels, 250)
 *       == SUCCESS)
 *   {
 *     // .. sequenceReady() event every 10 msec containing 4 ADC results
 *   } else {
 *     // check error
 *   }
 * }
 */

enum { ADC12_MAX_SEQUENCE = 16 };

typedef struct
{
  volatile unsigned
  inch: 4,                                     // input channel
  sref: 3,                                     // reference voltage
  eos: 1;                                      // end of sequence flag
} __attribute__ ((packed)) adc12memctl_t;

typedef struct {
  unsigned int ref2_5v: 1;         // reference voltage level
  unsigned int adc12ssel: 2;       // clock source sample-hold-time
  unsigned int adc12div: 3;        // clock divider sample-hold-time
  unsigned int sht: 4;             // sample-hold-time
  unsigned int sampcon_ssel: 2;    // clock source sampcon signal
  unsigned int sampcon_id: 2;      // clock divider sampcon
  unsigned int count: 4;           // number of channels in the sequence
  union {
    adc12memctl_t *sequence;       // used only when count > 1
    adc12memctl_t single;          // used only when count == 1
  } channels;
} msp430adc12_channel_config_t;

enum inch_enum
{
   // see device specific data sheet which pin Ax is mapped to
   INPUT_CHANNEL_A0 = 0,                    // input channel A0
   INPUT_CHANNEL_A1 = 1,                    // input channel A1
   INPUT_CHANNEL_A2 = 2,                    // input channel A2
   INPUT_CHANNEL_A3 = 3,                    // input channel A3
   INPUT_CHANNEL_A4 = 4,                    // input channel A4
   INPUT_CHANNEL_A5 = 5,                    // input channel A5
   INPUT_CHANNEL_A6 = 6,                    // input channel A6
   INPUT_CHANNEL_A7 = 7,                    // input channel A7
   EXTERNAL_REF_VOLTAGE_CHANNEL = 8,        // VeREF+ (input channel 8)
   REF_VOLTAGE_NEG_TERMINAL_CHANNEL = 9,    // VREF-/VeREF- (input channel 9)
   TEMPERATURE_DIODE_CHANNEL = 10,          // Temperature diode (input channel 10)
   SUPPLY_VOLTAGE_HALF_CHANNEL = 11,        // (AVcc-AVss)/2 (input channel 11-15)
   INPUT_CHANNEL_NONE = 12                  // illegal (identifies invalid settings)
};

enum sref_enum
{
   REFERENCE_AVcc_AVss = 0,                 // VR+ = AVcc   and VR-= AVss
   REFERENCE_VREFplus_AVss = 1,             // VR+ = VREF+  and VR-= AVss
   REFERENCE_VeREFplus_AVss = 2,            // VR+ = VeREF+ and VR-= AVss
   REFERENCE_AVcc_VREFnegterm = 4,          // VR+ = AVcc   and VR-= VREF-/VeREF-
   REFERENCE_VREFplus_VREFnegterm = 5,      // VR+ = VREF+  and VR-= VREF-/VeREF-
   REFERENCE_VeREFplus_VREFnegterm = 6      // VR+ = VeREF+ and VR-= VREF-/VeREF-
};

enum ref2_5v_enum
{
  REFVOLT_LEVEL_1_5 = 0,                    // reference voltage of 1.5 V
  REFVOLT_LEVEL_2_5 = 1,                    // reference voltage of 2.5 V
  REFVOLT_LEVEL_NONE = 0,                   // if e.g. AVcc is chosen
};

enum adc12ssel_enum
{
   SHT_SOURCE_ADC12OSC = 0,                // ADC12OSC
   SHT_SOURCE_ACLK = 1,                    // ACLK
   SHT_SOURCE_MCLK = 2,                    // MCLK
   SHT_SOURCE_SMCLK = 3                    // SMCLK
};

enum adc12div_enum
{
   SHT_CLOCK_DIV_1 = 0,                         // ADC12 clock divider of 1
   SHT_CLOCK_DIV_2 = 1,                         // ADC12 clock divider of 2
   SHT_CLOCK_DIV_3 = 2,                         // ADC12 clock divider of 3
   SHT_CLOCK_DIV_4 = 3,                         // ADC12 clock divider of 4
   SHT_CLOCK_DIV_5 = 4,                         // ADC12 clock divider of 5
   SHT_CLOCK_DIV_6 = 5,                         // ADC12 clock divider of 6
   SHT_CLOCK_DIV_7 = 6,                         // ADC12 clock divider of 7
   SHT_CLOCK_DIV_8 = 7,                         // ADC12 clock divider of 8
};

/* The number of ADC12CLK ticks that the ADC samples the input if SHP=1.  A
 * general formula for determining the sample time is:
 *   t(sample) >= (Rs + 2Kohm) x 9.011 x 40pF + 800ns,
 *     where Rs is the external source resistance.
 * If Rs = 10kOhm, then t(sample) >= 5.13us
 */
enum sht_enum
{
   SAMPLE_HOLD_4_CYCLES = 0,
   SAMPLE_HOLD_8_CYCLES = 1,
   SAMPLE_HOLD_16_CYCLES = 2,
   SAMPLE_HOLD_32_CYCLES = 3,
   SAMPLE_HOLD_64_CYCLES = 4,
   SAMPLE_HOLD_96_CYCLES = 5,
   SAMPLE_HOLD_123_CYCLES = 6,
   SAMPLE_HOLD_192_CYCLES = 7,
   SAMPLE_HOLD_256_CYCLES = 8,
   SAMPLE_HOLD_384_CYCLES = 9,
   SAMPLE_HOLD_512_CYCLES = 10,
   SAMPLE_HOLD_768_CYCLES = 11,
   SAMPLE_HOLD_1024_CYCLES = 12
};

/* The SHSx bits are set from one of these inputs to define the source of the
 * SHI signal, which upon its rising edge triggers a conversion.  In extended
 * sample mode (SHP=0), sampling occurs while SHI is high and conversion
 * starts upon its transition to low.  In pulse sample mode (SHP=1), the
 * rising edge of SHI starts a sampling period that lasts the number of ticks
 * of the ADC12CLK defined in SHT0x or SHT1x.
 *
 * See the latest revision of the SLAU09 document from TI
 */
enum shs_enum
{
   SHI_SOURCE_ADC12SC = 0,
   SHI_SOURCE_TIMERA_1 = 1,
   SHI_SOURCE_TIMERB_0 = 2,
   SHI_SOURCE_TIMERB_1 = 3
};

/* If SHSx is set to SHI_SOURCE_TIMERA_1, then the sampcon_ssel and sampcon_id
 * values in the structure define the source clock select and clock divider
 * for timer A, which is configured by Msp430Adc12P as the SHI source.
 */
enum sampcon_ssel_enum
{
   SAMPCON_SOURCE_TACLK = 0,        // Timer A clock source is (external) TACLK
   SAMPCON_SOURCE_ACLK = 1,         // Timer A clock source ACLK
   SAMPCON_SOURCE_SMCLK = 2,        // Timer A clock source SMCLK
   SAMPCON_SOURCE_INCLK = 3,        // Timer A clock source is (external) INCLK
};

enum sampcon_id_enum
{
   SAMPCON_CLOCK_DIV_1 = 0,             // SAMPCON clock divider of 1
   SAMPCON_CLOCK_DIV_2 = 1,             // SAMPCON clock divider of 2
   SAMPCON_CLOCK_DIV_3 = 2,             // SAMPCON clock divider of 3
   SAMPCON_CLOCK_DIV_4 = 3,             // SAMPCON clock divider of 4
};

// The unique string for allocating ADC resource interfaces
#define MSP430ADC12_RESOURCE "Msp430Adc12C.Resource"

// The unique string for accessing HAL2
#define ADCC_SERVICE "AdcC.Service"

// The unique string for accessing HAL2 via ReadStream
#define ADCC_READ_STREAM_SERVICE "AdcC.ReadStream.Client"


/* Test for GCC bug (bitfield access) - only version 3.2.3 is known to be stable */
// check: is this relevant anymore ?
#define GCC_VERSION (__GNUC__ * 100 + __GNUC_MINOR__ * 10 + __GNUC_PATCHLEVEL__)
#if GCC_VERSION == 332
#error "The msp430-gcc version (3.3.2) contains a bug which results in false accessing \
of bitfields in structs and makes MSP430ADC12M.nc fail ! Use version 3.2.3 instead."
#elif GCC_VERSION != 323
#warning "This version of msp430-gcc might contain a bug which results in false accessing \
of bitfields in structs (MSP430ADC12M.nc would fail). Use version 3.2.3 instead."
#endif

#endif
