/*
 * Copyright (c) 2004, Technische Universitaet Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universitaet Berlin nor the names
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Revision -------------------------------------------------------------
 * $Revision: 1.1.2.4 $
 * $Date: 2006/01/31 18:43:40 $
 * @author: Jan Hauer <hauer@tkn.tu-berlin.de>
 * ========================================================================
 */

/**
 *
 * Please refer to TEP 101 for more information about this interface and its
 * intended use.  This interface exports access to the ADC12 on the level of
 * HAL1 on a per-channel basis. It allows to sample a channel once or
 * repeatedly (and signal an event per conversion result) or perform multiple
 * conversions for the same channel once or repeatedly (and signal an event per
 * multiple conversion results). It does not allow to sample different channels
 * with the same command.
 *
 * @author Jan Hauer
 */

/*
 * FIXME: not sure about sequence sampling timing.  One strategy has all the
 * samples in the sequence happening as fast as possible and the timing of
 * sequence-start to sequence-start in repeating mode controlled by jiffies.
 * Another strategy is to have jiffies control the period between each
 * conversion, not between each sequence of conversions.
 */

#include <Msp430Adc12.h>

interface Msp430Adc12
{
  /**
   * Initializes the IO pins referenced in <code>config</code>.  Per the
   * MSP430 data sheet, we need to ensure IO pins receiving analog voltages
   * have their IO buffers disconnected to prevent parasitic current when
   * the pin voltage is near the gate transition threshold voltage.
   * 
   * If P6PIN_AUTO_CONFIGURE is defined, the IO pins are properly set before
   * a conversion and then reset (to normal GPIO mode) after the conversion
   * completes.  In the cases where this behavior may cause undesired parasitic
   * current, undefine P6PIN_AUTO_CONFIGURE and use
   * <code>configurePins()</code> instead.
   */
  async command void configurePins(const msp430adc12_channel_config_t *config);

  /**
   * Starts an ADC conversion sequence.  From 1 to 16 conversions can be made
   * within a single conversion sequence, as determined by the configuration
   * data (<code>config->count</code> and <code>config->channels</code>).  If
   * the method returns SUCCESS, then an event will be called when the
   * conversion sequence is complete.  If the sequence contains exactly one
   * conversion, <code>singleReady()</code> is sent.  If more than one
   * conversion is performed in the sequence, then
   * <code>sequenceReady()</code> is sent.
   *
   * @param config Pointer to the (read-only) ADC12 configuration data
   * @return SUCCESS means conversion will be signalled by
   * <code>singleReady()</code> or <code>sequenceReady()</code>.
   */
  async command error_t start(const msp430adc12_channel_config_t *config);

  /**
   * Starts a repeating ADC conversion sequence.  From 1 to 16 conversions can
   * be made within a single conversion sequence, as determined by the
   * configuration.  Once a sequence is complete, it repeats, based upon
   * <code>config->sampcon_ssel</code>, <code>config->sampcon_id</code> and the
   * <code>jiffies</code> parameter.  If the method returns SUCCESS, then an
   * event will be called when the conversion sequence is complete.  If the
   * sequence contains exactly one conversion, <code>singleReady()</code> is
   * sent.  If more than one conversion is performed in the sequence, then
   * <code>sequenceReady()</code> is sent.
   *
   * @param config Pointer to the ADC12 configuration data.
   * @param jiffies Used to clock the adc conversion process.  It zero, the
   * process proceeds as fast as the adc12 peripheral can go, while holding to
   * the specifications contained within <code>config</code>.  If non-zero, it
   * represents the inter-sample spacing.
   * <code>jiffies = (SAMPCON_SSEL / SAMPCON_ID) / (FREQ * CHANNELS)</code>.
   * SAMPCON_SSEL is the frequency of the clock specified by
   * <code>config->sampcon_ssel</code>.  SAMPCON_ID is the clock divider
   * indicated by <code>config->sampcon_id</code>.  FREQ is the desired
   * frequency with which to return conversion sequences.  CHANNELS is the
   * number of adc channel conversions within each sequence
   * (<code>config->count</code>).
   * @return SUCCESS means conversion will be signalled by
   * <code>singleReady()</code> or <code>sequenceReady()</code>.
   */
  async command error_t startRepeat(const msp430adc12_channel_config_t *config,
      uint16_t jiffies);

  /**
   * Signal the completion of a single-channel sequence, sending the conversion
   * result.  The code within this event should be as short as possible.  If
   * the conversion process was triggered by <code>startRepeat</code>, then the
   * return value of the event is used to continue or stop subsequent conversions.
   *
   * @param data Conversion result (lower 12 bits of the value are valid).
   * @return If triggered from <code>startRepeat</code>, a false return stops
   * a repeated conversion mode. A true value allows repeated conversion mode
   * to continue normally.  The return code is ignored if this event is in
   * response to <code>start()</code>.
   */
  async event bool singleReady(uint16_t data);

  /**
   * Signal the completion of a multi-channel sequence, sending a pointer to an
   * array of conversion results.  The code within this event should be as
   * short as possible.  If the conversion process was triggered by
   * <code>startRepeat</code>, then the return value of the event is used to
   * continue or stop subsequent sequence conversions.
   *
   * IMPORTANT: The MSP430 adc12 peripheral continues to assert its interrupt
   * flag(s) until the conversion values are read.  Unlike
   * <code>singleReady()</code>, which passes a single conversion result by
   * value, <code>sequenceReady()</code> passes a pointer into the adc12
   * peripheral where results are stored.  To prevent redundant interrupts from
   * the adc12, you should at least read the last result in the sequence, that
   * is data[len-1].
   *
   * @param data Pointer to an array of conversion results.  The lower 12 bits
   * of each value are valid.
   * @return If triggered from <code>startRepeat</code>, a false return stops
   * a repeated conversion mode. A true value allows repeated conversion mode
   * to continue normally.  The return code is ignored if this event is in
   * response to <code>start()</code>.
   */
  async event bool sequenceReady(uint16_t *data, uint8_t len);

  /**
   * Cancel the current conversion or sequence of conversions, as applicable.
   * <code>singleReady()</code> or <code>sequenceReady()</code> may still be
   * called after <code>cancel()</code> if valid data become available before
   * the cancel operation can take effect.
   */
  async command void cancel();
}
