/*
 * Copyright (c) 2004, Technische Universitaet Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universitaet Berlin nor the names
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Revision -------------------------------------------------------------
 * $Revision: 1.1.2.6 $
 * $Date: 2006/01/30 17:41:13 $
 * @author: Jan Hauer <hauer@tkn.tu-berlin.de>
 * ========================================================================
 */

#include <Msp430Adc12.h>
module Msp430Adc12P
{
  provides {
    interface Init;
    interface Msp430Adc12[uint8_t id];
  }
  uses {
    interface ArbiterInfo as ADCArbiterInfo;
    interface HplAdc12;
    interface Msp430Timer as TimerA;
    interface Msp430TimerControl as ControlA0;
    interface Msp430TimerControl as ControlA1;
    interface Msp430Compare as CompareA0;
    interface Msp430Compare as CompareA1;
    interface HplMsp430GeneralIO as Port60;
    interface HplMsp430GeneralIO as Port61;
    interface HplMsp430GeneralIO as Port62;
    interface HplMsp430GeneralIO as Port63;
    interface HplMsp430GeneralIO as Port64;
    interface HplMsp430GeneralIO as Port65;
    interface HplMsp430GeneralIO as Port66;
    interface HplMsp430GeneralIO as Port67;
  }
}
implementation
{
  enum { /* mode bit settings; see conversion mode enum */
    MODE_SINGLE = 0,
    MODE_SEQUENCE = 1,
    MODE_ONECONV = 0,
    MODE_REPEAT = 2
  };
  enum { /* conversion mode; also the value for CONSEQx */
    SINGLE_ONCE = MODE_SINGLE|MODE_ONECONV,
    SINGLE_REPEAT = MODE_SINGLE|MODE_REPEAT,
    SEQUENCE_ONCE = MODE_SEQUENCE|MODE_ONECONV,
    SEQUENCE_REPEAT = MODE_SEQUENCE|MODE_REPEAT
  };
  enum { /* flags */
    ADC_BUSY = 1,               /* request pending */
    TIMERA_USED = 2,            /* TimerA used for SAMPCON signal */
  };

  uint8_t clientID;             /* interface ID of current requestor */
  uint8_t flags;                /* current state, see above */

  /* Resource interface resolves conflicts for the following variables */
  norace uint8_t mode;		/* the ADC operating mode */
  norace uint8_t count;		/* count of channels in current sequence */

  command error_t Init.init()
  {
    call HplAdc12.disableConversion();
    call HplAdc12.adcOff();
    return SUCCESS;
  }

  error_t clientAccessRequest(uint8_t id)
  {
    if (call ADCArbiterInfo.userId() == id){
      atomic {
        if (flags & ADC_BUSY)
          return EBUSY;
        flags = ADC_BUSY;
        clientID = id;
      }
      return SUCCESS;
    }
    return ERESERVE;
  }

  inline void clientAccessFinished()
  {
    atomic flags = 0;
  }

  /* Configure timer A to generate a square wave having a 50% duty cycle with
   * a period defined by 'interval' ticks of the clock, and the clock defined
   * as the source 'csSAMPCON' which is pre-divided by 'cdSAMPCON'.
   */
  void prepareTimerA(uint16_t interval, uint16_t csSAMPCON, uint16_t cdSAMPCON)
  {
    msp430_compare_control_t ccResetSHI = {
      ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 0,
      outmod : 0, cap : 0, clld : 0, scs : 0, ccis : 0, cm : 0 };

    call TimerA.setMode(MSP430TIMER_STOP_MODE);
    call TimerA.clear();
    call TimerA.disableEvents();
    call TimerA.setClockSource(csSAMPCON);
    call TimerA.setInputDivider(cdSAMPCON);
    call ControlA0.setControl(ccResetSHI);
    call CompareA0.setEvent(interval-1);
    call CompareA1.setEvent((interval-1)/2);
  }

  void startTimerA()
  {
    msp430_compare_control_t ccSetSHI = {
      ccifg : 0, cov : 0, out : 1, cci : 0, ccie : 0,
      outmod : 0, cap : 0, clld : 0, scs : 0, ccis : 0, cm : 0 };
    msp430_compare_control_t ccResetSHI = {
      ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 0,
      outmod : 0, cap : 0, clld : 0, scs : 0, ccis : 0, cm : 0 };
    msp430_compare_control_t ccRSOutmod = {
      ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 0,
      outmod : 7, cap : 0, clld : 0, scs : 0, ccis : 0, cm : 0 };
    // manually trigger first conversion then switch to Reset/set mode
    call ControlA1.setControl(ccResetSHI);
    call ControlA1.setControl(ccSetSHI);
    //call ControlA1.setControl(ccResetSHI);
    call ControlA1.setControl(ccRSOutmod);
    call TimerA.setMode(MSP430TIMER_UP_MODE); // go!
  }

  void configureAdcPin( uint8_t inch )
  {
    switch (inch)
    {
      case 0: call Port60.selectModuleFunc(); call Port60.makeInput(); break;
      case 1: call Port61.selectModuleFunc(); call Port61.makeInput(); break;
      case 2: call Port62.selectModuleFunc(); call Port62.makeInput(); break;
      case 3: call Port63.selectModuleFunc(); call Port63.makeInput(); break;
      case 4: call Port64.selectModuleFunc(); call Port64.makeInput(); break;
      case 5: call Port65.selectModuleFunc(); call Port65.makeInput(); break;
      case 6: call Port66.selectModuleFunc(); call Port66.makeInput(); break;
      case 7: call Port67.selectModuleFunc(); call Port67.makeInput(); break;
    }
  }

  void resetAdcPin( uint8_t inch )
  {
    switch (inch)
    {
      case 0: call Port60.selectIOFunc(); break;
      case 1: call Port61.selectIOFunc(); break;
      case 2: call Port62.selectIOFunc(); break;
      case 3: call Port63.selectIOFunc(); break;
      case 4: call Port64.selectIOFunc(); break;
      case 5: call Port65.selectIOFunc(); break;
      case 6: call Port66.selectIOFunc(); break;
      case 7: call Port67.selectIOFunc(); break;
    }
  }

  void stopConversion()
  {
    if (flags & TIMERA_USED){
      call TimerA.setMode(MSP430TIMER_STOP_MODE);
    }
#ifdef P6PIN_AUTO_CONFIGURE
    while (count) {
      adc12memctl_t memctl = call HplAdc12.getMCtl(--count);
      resetAdcPin(memctl.inch);
    }
#endif
    call HplAdc12.stopConversion();
    call HplAdc12.resetOV();
    call HplAdc12.resetIFGs();
    call HplAdc12.setIEFlags(0);
    clientAccessFinished();
  }

  error_t configureAdc(uint8_t id, bool repeat,
      const msp430adc12_channel_config_t *config, uint16_t jiffies)
  {
    error_t result;

#ifdef CHECK_ARGS
    if (!config || !config->count || jiffies == 1 || jiffies == 2)
      return EINVAL;
#endif
    count = config->count;
    /* Derive mode/CONSEQx */
    mode = (repeat ? MODE_REPEAT : MODE_ONECONV) |
        ((count > 1) ? MODE_SEQUENCE : MODE_SINGLE);

    if ((result = clientAccessRequest(id)) == SUCCESS)
    {
      adc12ctl0_t ctl0 = {
        adc12sc: 0,
        enc: 0,
        adc12tovie: 0,
        adc12ovie: 0,
        adc12on: 1,
        refon: call HplAdc12.getRefon(),
        r2_5v: call HplAdc12.isRef2_5V(),
        msc: jiffies ? 0 : 1,
        sht0: config->sht,
        sht1: config->sht
      };
      adc12ctl1_t ctl1 = {
        adc12busy: 0,
        conseq: mode,
        adc12ssel: config->adc12ssel,
        adc12div: config->adc12div,
        issh: 0,
        shp: 1,
        shs: jiffies ? SHI_SOURCE_TIMERA_1 : SHI_SOURCE_ADC12SC,
        cstartadd: 0
      };
      adc12memctl_t channel;
      call HplAdc12.setCtl0(ctl0);
      call HplAdc12.setCtl1(ctl1);
      if (count == 1) {
	/* The channel must be in the channels union */
        channel = config->channels.single;
        channel.eos = 1;
#ifdef P6PIN_AUTO_CONFIGURE
	configureAdcPin(channel.inch);
#endif
        call HplAdc12.setMCtl(0, channel);
        call HplAdc12.setIEFlags(1);
      } else {
      	uint8_t i;

	/* The channels union must contain a ptr to the sequence of channels */
        adc12memctl_t *sequence = config->channels.sequence;
        for (i = 0; i < count-1; i++) {
          channel = sequence[i];
#ifdef P6PIN_AUTO_CONFIGURE
          configureAdcPin(channel.inch);
#endif
          call HplAdc12.setMCtl(i, channel);
        }
        channel = sequence[i];
#ifdef P6PIN_AUTO_CONFIGURE
        configureAdcPin(channel.inch);
#endif
        channel.eos = 1;
        call HplAdc12.setMCtl(i, channel);
        call HplAdc12.setIEFlags((uint16_t)1 << i);
      }
      if (jiffies) {
        atomic flags |= TIMERA_USED;
        prepareTimerA(jiffies, config->sampcon_ssel, config->sampcon_id);
        call HplAdc12.startConversion();
        startTimerA(); // go!
      } else
        call HplAdc12.startConversion();
    }
    return result;
  }

  async command void Msp430Adc12.configurePins[uint8_t id](
      const msp430adc12_channel_config_t *config)
  {
#ifdef CHECK_ARGS
    if (!config)
      return;
#endif

    if (config->count == 1)
      configureAdcPin(config->channels.single.inch);
    else {
      int i;
      for (i = 0; i < config->count; i++)
        configureAdcPin(config->channels.sequence[i].inch);
    }
  }

  async command error_t Msp430Adc12.start[uint8_t id](
      const msp430adc12_channel_config_t *config)
  {
    return configureAdc(id, FALSE, config, 0);
  }

  async command error_t Msp430Adc12.startRepeat[uint8_t id](
      const msp430adc12_channel_config_t *config, uint16_t jiffies)
  {
    return configureAdc(id, TRUE, config, jiffies);
  }

  async command void Msp430Adc12.cancel[uint8_t id]()
  {
    stopConversion();
  }

  async event void TimerA.overflow(){}
  async event void CompareA0.fired(){}
  async event void CompareA1.fired(){}


  async event void HplAdc12.conversionDone(uint16_t iv)
  {
    switch (mode)
    {
      case SINGLE_ONCE:
        stopConversion();
        signal Msp430Adc12.singleReady[clientID](call HplAdc12.getMem(0));
        break;
      case SINGLE_REPEAT:
        if (!signal Msp430Adc12.singleReady[clientID](call HplAdc12.getMem(0)))
          stopConversion();
        break;
      case SEQUENCE_ONCE:
        /* FIXME: should ADC12MEM be Hpl.getAdc12Mem()? */
        stopConversion();
        signal Msp430Adc12.sequenceReady[clientID]((uint16_t *)ADC12MEM,
            count);
        break;
      case SEQUENCE_REPEAT:
        /* FIXME: should ADC12MEM be Hpl.getAdc12Mem()? */
        if (!signal Msp430Adc12.sequenceReady[clientID]((uint16_t *)ADC12MEM,
            count))
          stopConversion();
        break;
      } /* switch */
  }

  default async event bool Msp430Adc12.singleReady[uint8_t id](uint16_t data)
  {
    return FALSE;
  }

  default async event bool Msp430Adc12.sequenceReady[uint8_t id](
      uint16_t *data, uint8_t len)
  {
    return FALSE;
  }

  async event void HplAdc12.memOverflow(){}
  async event void HplAdc12.conversionTimeOverflow(){}
}

