#include <I2C.h>

module MasterI2CC {
  uses {
    interface Boot;
    interface Notify<uint8_t> as Button;
    interface Resource;
    interface I2CPacket<TI2CBasicAddr> as I2CBasicAddr;
    interface Timer<TMilli>;
    interface Leds;
    interface FlashLeds;
  }
}
implementation {
  enum {
    DATA_LEN = 16,
    SLAVE_ADDR = 0xc6 >> 1,
    TIME_WAIT_MS = 2048,
  };

  enum {
    STATE_IDLE = 0,
    STATE_WRITE,
    STATE_READ
  };

  norace error_t lastErr = FAIL;
  int state;
  char wData[] = "0123456789abcdef";
  char rData[DATA_LEN];

  void localError(int e)
  {
    call FlashLeds.ledAll(e);
    call Leds.led0Off();
    call Leds.led1Off();
    call Leds.led2Off();
    call Button.enable();
  }

  event void Boot.booted()
  {
    call Button.enable();
  }

  event void Button.notify(uint8_t v)
  {
    state = STATE_WRITE;
    if (call Resource.request() == SUCCESS) {
      call Leds.led0On();
      return;
    }
    localError(2);
  }

  void writeLine()
  {
    uint8_t* l = (uint8_t*)wData;
    call Leds.led1On();
    call Leds.led2Off();
    if (call I2CBasicAddr.write(I2C_START|I2C_STOP, SLAVE_ADDR, strlen(l), l) !=
        SUCCESS)
      localError(3);
  }

  void readLine()
  {
    call Leds.led1Off();
    call Leds.led2On();
    if (call I2CBasicAddr.read(I2C_START|I2C_STOP, SLAVE_ADDR, sizeof(rData),
        (uint8_t*)rData) != SUCCESS)
      localError(4);
  }

  event void Resource.granted()
  {
    switch (state) {
      case STATE_WRITE:
	writeLine();
	break;
      case STATE_READ:
	readLine();
	break;
      default:
	localError(5);
    }
  }

  task void i2cWriteDone()
  {
    error_t err;
    atomic err = lastErr;
    call Resource.release();
    if (err == SUCCESS) {
      call Timer.startOneShot(TIME_WAIT_MS);
      call Leds.led0Off();
      call Leds.led1Off();
      call Leds.led2Off();
      return;
    }
    localError(6);
  }

  async event void I2CBasicAddr.writeDone(error_t error, uint16_t addr,
      uint8_t len, uint8_t* data)
  {
    lastErr = error;
    post i2cWriteDone();
  }

  task void i2cReadDone()
  {
    error_t err;
    atomic err = lastErr;
    call Resource.release();
    state = STATE_IDLE;
    if (err == SUCCESS) {
      /* Compare the read and write buffers; they should be the same */
      if (strncmp(rData, wData, sizeof(rData)) == 0) {
	call Leds.led0Off();
	call Leds.led1Off();
	call Leds.led2Off();
	call FlashLeds.ledAll(1);
	call Button.enable();
	return;
      } else
	localError(7);
    }
    localError(8);
  }

  async event void I2CBasicAddr.readDone(error_t error, uint16_t addr,
      uint8_t len, uint8_t* data)
  {
    lastErr = error;
    post i2cReadDone();
  }

  event void Timer.fired()
  {
    state = STATE_READ;
    if (call Resource.request() == SUCCESS) {
      call Leds.led0On();
      return;
    }
    localError(9);
  }
}
