#ifdef AG_USE_MULTIHOP
  #include "routing_table.h"
#else
  #include "AM.h"
#endif

#include "Serial.h"
#include "TestTymoLoop.h"

//#include "printf.h"

module TestTymoLoopP @safe() {
  uses {
    interface Boot;
    interface SplitControl as SerialControl;
    interface SplitControl as RadioControl;

    interface AMSend as UartSend[am_id_t id];
    interface Receive as UartReceive[am_id_t id];
    interface Packet as UartPacket;
    interface AMPacket as UartAMPacket;

#ifdef AG_USE_MULTIHOP
    interface AMPacket as RadioAMPacket;
    interface Packet as RadioPacket;
    interface Receive as RadioReceive;
    interface Intercept as RadioIntercept;
    interface AMSend as RadioSend;
#else
    interface AMSend as RadioSend[am_id_t id];
    interface Receive as RadioReceive[am_id_t id];
    interface Receive as RadioSnoop[am_id_t id];
    interface Packet as RadioPacket;
    interface AMPacket as RadioAMPacket;
#endif

    interface Leds;
    interface Timer<TMilli>;
  }
#ifdef DYMO_MONITORING
  uses interface DymoMonitor;
#endif
}

implementation
{

  // Tymo
  enum {
    ORIGIN = 0,
    TARGET = 1,
  };

  am_addr_t addr_my;
  am_addr_t addr_to;

  message_t packet;
  test_tymo_loop_msg_t payload_current;

  task void uartSendTask();
  task void radioSendTask();

  void dropBlink() {
    call Leds.led2Toggle();
  }

  void failBlink() {
    call Leds.led2Toggle();
  }

  event void Boot.booted() {

    call RadioControl.start();
    call SerialControl.start();

    addr_my = call RadioAMPacket.address();

    dbg("ag", "AG: mote %u booted.\n", addr_my);
  }

  event void RadioControl.startDone(error_t error) {
    if (error != SUCCESS) {
       dbg("ag", "AG: mote %u RADIO init FAILED!!!!!\n", addr_my);
    }
  }

  event void SerialControl.startDone(error_t error) {
    if (error != SUCCESS) {
       dbg("ag", "AG: mote %u SERIAL init FAILED!!!!!\n", addr_my);
    }
  }

  event void SerialControl.stopDone(error_t error) {}
  event void RadioControl.stopDone(error_t error) {}

  event void Timer.fired(void){
  }


  message_t* ONE receive(message_t* ONE msg, void* payload, uint8_t len);

#ifndef AG_USE_MULTIHOP
  event message_t *RadioSnoop.receive[am_id_t id](message_t *msg,
						    void *payload,
						    uint8_t len) {
    return receive(msg, payload, len);
  }
#endif

#ifndef AG_USE_MULTIHOP
  event message_t *RadioReceive.receive[am_id_t id](message_t *msg, void *payload, uint8_t len) {
#else
  event message_t *RadioReceive.receive            (message_t *msg, void *payload, uint8_t len) {
#endif
    return receive(msg, payload, len);
  }

  message_t* receive(message_t *msg, void *payload, uint8_t len) {
    message_t *ret = msg;

    am_addr_t source_addr;

    source_addr = call RadioAMPacket.source(msg);

    dbg("ag", "AG: received message via radio from %u.\n", source_addr);
    dbg("ag", "AG: payload: %u.\n", ((test_tymo_loop_msg_t*) payload)->counter);

    payload_current.counter = ((test_tymo_loop_msg_t*) payload)->counter;

    if ( addr_my == TARGET ) {
       addr_to = ORIGIN;
       call Leds.led2Toggle();
       dbg("ag", "AG: putting message to radio.\n");
       post radioSendTask();
    }

    if( addr_my == ORIGIN) {
       addr_to = ORIGIN;
       dbg("ag", "AG: putting message to serial.\n");
       post uartSendTask();
    }

    return ret;
  }

  task void uartSendTask() {

    test_tymo_loop_msg_t* payload_new;

    error_t error;

    payload_new = call RadioPacket.getPayload(&packet, sizeof(test_tymo_loop_msg_t));

    call UartPacket.clear(&packet);

    payload_new->counter = payload_current.counter;
    error = call UartSend.send[9](addr_to, &packet, sizeof(*payload_new));
    if(error == SUCCESS){
        dbg("ag", "AG: Uart: sending...\n");
        call Leds.led1Toggle();
    } else {
        dbg("ag", "AG: Uart: unable to send.\n");
	//failBlink();
    }

  }

// With conventional communication there is no problem
// With Tymo the client program on virtual serial port receives always 513,
// while between motes there is the proper number in payload.

  event void UartSend.sendDone[am_id_t id](message_t* msg, error_t error) {
    if (error != SUCCESS || msg != &packet) {
      dbg("ag", "AG: Uart.sendDone Failed\n");
      //failBlink();
    }
    else {
      dbg("ag", "AG: Uart.sendDone OK\n");
    }
  }

  event message_t *UartReceive.receive[am_id_t id](message_t *msg,
						   void *payload,
						   uint8_t len) {
    message_t *ret = msg;

    dbg("ag", "AG: received message via uart from %u.\n", call UartAMPacket.source(msg));
    dbg("ag", "AG: payload: %u.\n", ((test_tymo_loop_msg_t*) payload)->counter);

    payload_current.counter = ((test_tymo_loop_msg_t*) payload)->counter;

    addr_to = TARGET;
    dbg("ag", "AG: putting message to radio.\n");
    post radioSendTask();

    return ret;
  }

  task void radioSendTask() {

    test_tymo_loop_msg_t* payload_new;

    error_t error;

    payload_new = call RadioPacket.getPayload(&packet, sizeof(test_tymo_loop_msg_t));

    call RadioPacket.clear(&packet);

#ifdef AG_USE_MULTIHOP
// Without these lines there is only one debug message from Tymo:
// DEBUG (0): MHE: Somebody wants a route, let's see...
// and no radio communication at all.
// But for conventional ActiveMessage there is no need to set these data.

    call RadioAMPacket.setSource(&packet,addr_my);
    call RadioAMPacket.setDestination(&packet,addr_to);
#endif

    payload_new->counter = payload_current.counter;
#ifdef AG_USE_MULTIHOP
    error = call RadioSend.send   (addr_to, &packet, sizeof(*payload_new));
#else
    error = call RadioSend.send[9](addr_to, &packet, sizeof(*payload_new));
#endif
    if(error == SUCCESS){
        dbg("ag", "AG: Radio: sending to %u...\n", addr_to);
        call Leds.led0Toggle();
    } else {
        dbg("ag", "AG: Radio: unable to send to %u.\n", addr_to);
	//failBlink();
    }

  }

#ifdef AG_USE_MULTIHOP
  event void RadioSend.sendDone            (message_t* msg, error_t error) {
#else
  event void RadioSend.sendDone[am_id_t id](message_t* msg, error_t error) {
#endif

    if (error != SUCCESS || &packet != msg ) {
      failBlink();
      dbg("ag", "AG: sending message via radio to %u DONE FAILED.\n", addr_to);
    }
    else {
      call Leds.led0Toggle();
      dbg("ag", "AG: sending message via radio to %u DONE OK.\n", addr_to);
    }

  }

#ifdef AG_USE_MULTIHOP
  event bool RadioIntercept.forward(message_t * msg, void * payload, uint8_t len){
    //setLeds(2);
    call Leds.led1Toggle();
    return TRUE;
  }
#endif

}  
