import os, sys
from syslog import syslog

import Util

class FlatFileStore:
    """
    Very inefficient file store.  Use only if you're too lazy
    to install any db libraries. :)
    """
    def __init__(self, name):
        try:
            self.key_list = Util.file_to_list(name)
        except:
            exc_type, exc_value = sys.exc_info()[:2]
            syslog('could not read "%s": %s (%s)' % (name, exc_type, exc_value))
            self.key_list = []

    def contains(self, keys):
        return Util.findmatch(self.key_list, keys)

class CdbStore:
    """
    Search DJB's constant databases; see <http:/cr.yp.to/cdb.html>.
    """
    def __init__(self, name):
        import cdb
        try:
            self.db = cdb.init(pathname)
        except:
            exc_type, exc_value = sys.exc_info()[:2]
            syslog('could not read "%s": %s (%s)' % (name, exc_type, exc_value))
            self.db = None

    def contains(self, keys):
        if self.db != None:
            for key in keys:
                if key and self.db.has_key(string.lower(key)):
                    return 1
        return 0


class DbmStore:
    """
    Search a DBM-style database.
    """
    
    def __init__(self, name):
        import anydbm
        try:
            self.db = anydbm.open(pathname, 'r')
        except:
            exc_type, exc_value = sys.exc_info()[:2]
            syslog('could not read "%s": %s (%s)' % (name, exc_type, exc_value))
            self.db = None

    def contains(self, keys):
        if self.db != None:
            for key in keys:
                if key and self.db.has_key(string.lower(key)):
                    return 1
        return 0


class EzmlmStore:
    """ ezmlm subscriber directories """

    def __init__(self, name):
        self.ezmlm_list = []
        dir = os.path.join(name, 'subscribers')
        # See ezmlm(5) for dir/subscribers format.
        try:
            for file in os.listdir(dir):
                fp = open(os.path.join(dir, file), 'r')
                subs = fp.read().split('\x00')
                for sub in subs:
                    if sub:
                        self.ezmlm_list.append(sub.split('T', 1)[1].lower())
        except:
            exc_type, exc_value = sys.exc_info()[:2]
            syslog('could not read "%s": %s (%s)' % (name, exc_type, exc_value))
            self.keys = []
        
    def contains(self, keys):
        for key in keys:
            if key and key.lower() in self.ezmlm_list:
                return 1
        return 0


class MailmanStore:
    """
    Mailman configuration databases
    (note that this init takes a 2nd arg)
    """
    
    def __init__(self, name, mmkey):
        # Find the Mailman configuration database.
        # 'config.db' is a Python marshal used in MM 2.0, and
        # 'config.pck' is a Python pickle used in MM 2.1.
        config_db = os.path.join(name, 'config.db')
        config_pck = os.path.join(name, 'config.pck')
        if os.path.contains(config_pck):
            dbfile = config_pck
            import cPickle as Serializer
        elif os.path.contains(config_db):
            dbfile = config_db
            import marshal as Serializer
        else:
            syslog('could not read mailman config database')
            self.mmdb_list = []
            return

        try:
            mmdb_file = open(dbfile, 'r')
            mmdb_data = Serializer.load(mmdb_file)
            mmdb_file.close()
            self.mmdb_list = mmdb_data[mmdb_key]

            # Make sure self.mmdb_list is a list of e-mail addresses.
            if type(self.mmdb_list) is types.DictType:
                self.mmdb_list = mmdb_data[mmdb_key].keys()
        except:
            exc_type, exc_value = sys.exc_info()[:2]
            syslog('could not read "%s": %s (%s)' % (name, exc_type, exc_value))
            self.mmdb_list = []
        

    def contains(self, keys):
        for addy in keys:
            if addy and addy.lower() in self.mmdb_list:
                return 1
        return 0


def getStore(type, name, args):
    exname = os.path.expanduser(name)
    
    # 'from-file' or 'to-file', including autocdb functionality
    if type in ('from-file', 'to-file'):
        if args.has_key('autocdb'):
            if Util.autobuild_db(exname, '.cdb', Util.build_cdb):
                return CdbStore(exname + '.cdb')
            else:
                syslog('could not autobuild cdb for "%s"' % basename)
        elif args.has_key('autodbm'):
            if Util.autobuild_db(exname, '.dbm', Util.build_dbm):
                return DbmStore(exname + '.dbm')
            else:
                syslog('could not autobuild dbm for "%s"' % basename)
        # db types fall through to here on failure -- 
        # we'll try flatfile, in case the problem was with the db
        return FlatFileStore(exname)

    if type in ('from-dbm', 'to-dbm'):
        return DbmStore(exname)

    if type in ('from-cdb', 'to-cdb'):
        return CdbStore(exname)

    if type in ('from-ezmlm', 'to-ezmlm'):
        return EzmlmStore(exname)

    if type in ('from-mailman', 'to-mailman'):
        try:
            mmdb_key = args['attr']
        except KeyError:
            raise NameError(lineno,
                             '"%s" missing -attr argument' % type)
        return MailmanStore(exname, mmdb_key)

    return None
