package org.apache.maven.cvslib;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *    "Apache Maven" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Maven", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

import java.io.File;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Date;
import java.text.SimpleDateFormat;

// maven imports
import org.apache.maven.changelog.ChangeLog;
import org.apache.maven.changelog.ChangeLogGenerator;
import org.apache.maven.changelog.ChangeLogParser;
import org.apache.maven.util.AsyncStreamReader;

// ant imports
import org.apache.tools.ant.taskdefs.ExecuteStreamHandler;
import org.apache.tools.ant.taskdefs.Execute;
import org.apache.tools.ant.types.Commandline;

/**
 * A CVS implementation of the {@link org.apache.maven.changelog.ChangeLog}
 * interface.
 *
 * @todo Check CVS exists first by running cvs -version instead of current hack
 * @author Glenn McAllister
 * @author <a href="mailto:jeff.martin@synamic.co.uk">Jeff Martin</a>
 * @author <a href="mailto:jason@zenplex.com">Jason van Zyl</a>
 * @author <a href="mailto:dion@multitask.com.au">dIon Gillard</a>
 * @author <a href="mailto:bodewig@apache.org">Stefan Bodewig</a>
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 * @version $Id$
 */
public class CvsChangeLogGenerator 
implements ChangeLogGenerator, ExecuteStreamHandler
{
    /** 
     * Reference to the enclosing ChangeLog instance - used to obtain any
     * necessary configuration information. 
     * */
    private ChangeLog changeLogExecutor;
    
    /** 
     * stderr stream eater 
     * */
    private AsyncStreamReader errorReader;

    /** 
     * The cvs process input stream. 
     * */
    private InputStream in;

    /** 
     * The working directory for cvs. 
     * */
    private File base;

    /**
     * The CVS date range.
     */
    private String dateRange;

    /**
     * The parser that takes the cvs log output and transforms it into a
     * collection of ChangeLogEntry's.
     */
    private ChangeLogParser clParser;

    /**
     * The collection of ChangeLogEntry's returned from clParser.
     */
    private Collection entries;

    // doc comment inherited from interface
    public void init( ChangeLog changeLog )
    {
        changeLogExecutor = changeLog;
        
        base = changeLogExecutor.getBasedir();

        if (changeLogExecutor.getRange() != null)
        {
            setDateRange( changeLogExecutor.getRange() );
        }
 
    }

    /**
     * Set the dateRange member based on the number of days obtained from the
     * ChangeLog.
     */
    private void setDateRange( String numDaysString )
    {
        int days = Integer.parseInt( numDaysString );

        Date before = new Date(
            System.currentTimeMillis() - (long) days * 24 * 60 * 60 * 1000);
        Date to = new Date(
            System.currentTimeMillis() + (long) 1 * 24 * 60 * 60 * 1000);

        SimpleDateFormat outputDate = new SimpleDateFormat("yyyy-MM-dd");
        // We want something of the form: -d ">=YYYY-MM-dd"
        dateRange = "-d " + outputDate.format(before) + "<" + 
            outputDate.format(to);

    }

    // doc comment inherited from ChangeLogGenerator interface
    public Collection getEntries( ChangeLogParser parser ) throws IOException
    {
        if (parser == null)
        {
            throw new NullPointerException("parser cannot be null");
        }

        if (base == null)
        {
            throw new NullPointerException("basedir must be set");
        }

        if (!base.exists())
        {
            throw new FileNotFoundException(
                "Cannot find base dir " + base.getAbsolutePath());
        }

        clParser = parser;
        try 
        {
            Execute exe = new Execute(this);
            exe.setCommandline(getCvsLogCommand().getCommandline());
            exe.setAntRun(changeLogExecutor.getProject());
            exe.setWorkingDirectory(base);
            exe.execute();


            // log messages from stderr
            String errors = errorReader.toString().trim();
            if (errors.length() > 0)
            {
                changeLogExecutor.log(errors);
            }
        }
        catch (IOException ioe)
        {
            if (ioe.getMessage().indexOf("CreateProcess")  != -1 ||
                ioe.getMessage().indexOf("cvs: not found") != -1) 
            {
                // can't find CVS on Win32 or Linux...
                changeLogExecutor.log(
                    "Unable to find cvs executable. Changelog will be empty");
            }
            else
            {
                throw ioe;
            }
        }

        return entries;
    }

    // doc comment inherited from interface - nothing to do
    public void cleanup()
    {
    }

    /**
     * @return the cvs command line to be executed.
     */
    private Commandline getCvsLogCommand() 
    {
        Commandline command = new Commandline();

        command.setExecutable("cvs");
        command.createArgument().setValue("log");

        if (dateRange != null)
        {
            command.createArgument().setValue(dateRange);
        }
        
        return command;
    }


    /**
     * Stop the process - currently unimplemented
     */
    public void stop()
    {
    }

    /**
     * Set the input stream for the cvs process. cvs requires no input so this
     * is not used.
     * @param os - an {@link java.io.OutputStream}
     */
    public void setProcessInputStream(OutputStream os)
    {
    }

    /**
     * Set the error stream for reading from cvs log. This stream will be read
     * on a separate thread.
     * @param is - an {@link java.io.InputStream}
     */
    public void setProcessErrorStream(InputStream is)
    {
        errorReader = new AsyncStreamReader(is);
    }

    /**
     * Set the input stream used to read from cvs log
     * @param is - a stream of cvs log output to be read from
     */
    public void setProcessOutputStream(InputStream is)
    {
        in = is;
    }

    /**
     * Start read from the cvs log.
     * @throws IOException when there are errors reading from the streams
     *      previously provided
     */
    public void start() throws IOException
    {
        errorReader.start();
        entries = clParser.parse(in);
    }


}
