package org.apache.maven.cvslib;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *    "Apache Maven" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Maven", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TreeMap;

import org.apache.maven.changelog.ChangeLog;
import org.apache.maven.changelog.ChangeLogParser;
import org.apache.maven.changelog.ChangeLogEntry;
import org.apache.maven.changelog.ChangeLogFile;

/**
 * A class to parse cvs log output
 *
 * @author <a href="mailto:dion@multitask.com.au">dIon Gillard</a>
 * @version $Id: ChangeLogParser.java,v 1.18 2002/04/22 00:15:04 dion Exp $
 */
public class CvsChangeLogParser implements ChangeLogParser
{
    
    /**
     * rcs entries, in reverse (date, time, author, comment) order
     */
    private Map entries = new TreeMap(Collections.reverseOrder());

    // state machine constants for reading cvs output
    /** expecting file information */
    private static final int GET_FILE = 1;
    /** expecting date */
    private static final int GET_DATE = 2;
    /** expecting comments */
    private static final int GET_COMMENT = 3;
    /** expecting revision */
    private static final int GET_REVISION = 4;
    /** Marks start of file data*/
    private static final String START_FILE = "Working file: ";
    /** Marks end of file */
    private static final String END_FILE = "==================================="
        + "==========================================";
    /** Marks start of revision */
    private static final String START_REVISION = "----------------------------";
    /** Marks revision data */
    private static final String REVISION_TAG = "revision ";
    /** Marks date data */
    private static final String DATE_TAG = "date: ";

    /** current status of the parser */
    private int status = GET_FILE;
    
    /** the current log entry being processed by the parser*/
    private ChangeLogEntry currentLogEntry = null;
    
    /** the current file being processed by the parser */
    private ChangeLogFile currentFile = null;

    /**
     * Create a new ChangeLogParser.
     */
    public CvsChangeLogParser()
    {
    }

    // doc comment inherited from ChangeLogParer interface
    public void init( ChangeLog changeLog )
    {
    }

    // doc comment inherited from ChangeLogParer interface
    public void cleanup()
    {
    }
    
    /**
     * Parse the input stream into a collection.
     * @param anInputStream an input stream containing cvs log output
     * @return a collection of ChangeLogEntry's
     * @throws IOException when there are errors reading the provided stream
     */
    public Collection parse(InputStream anInputStream) throws IOException
    {
        BufferedReader stream = new BufferedReader( 
            new InputStreamReader(anInputStream) );

        // current state transitions in the state machine - starts with Get File
        //      Get File                -> Get Revision
        //      Get Revision            -> Get Date or Get File
        //      Get Date                -> Get Comment
        //      Get Comment             -> Get Comment or Get Revision
        String line = null;
        while ((line = stream.readLine()) != null)
        {
            switch (status)
            {
                case GET_FILE:
                    processGetFile(line);
                    break;
                case GET_REVISION:
                    processGetRevision(line);
                    break;
                case GET_DATE:
                    processGetDate(line);
                    break;
                case GET_COMMENT:
                    processGetComment(line);
                    break;
                default:
                    throw new IllegalStateException("Unknown state: " + status);
            }
        }
        
        return entries.values();
     }
    
    /**
     * Add a change log entry to the list (if it's not already there)
     * with the given file.
     * @param entry a {@link ChangeLogEntry} to be added to the list if another
     *      with the same key doesn't exist already. If the entry's author
     *      is null, the entry wont be added
     * @param file a {@link ChangeLogFile} to be added to the entry
     */
    private void addEntry(ChangeLogEntry entry, ChangeLogFile file)
    {
        // do not add if entry is not populated
        if (entry.getAuthor() == null)
        {
            return;
        }
        
        String key = entry.getDateFormatted() + entry.getTimeFormatted() + 
            entry.getAuthor() + entry.getComment();
        if (!entries.containsKey(key))
        {
            entry.addFile(file);
            entries.put(key, entry);
        }
        else
        {
            ChangeLogEntry existingEntry = (ChangeLogEntry) entries.get(key);
            existingEntry.addFile(file);
        }
    }
 
    /**
     * Process the current input line in the Get File state.
     * @param line a line of text from the cvs log output
     */
    private void processGetFile(String line) 
    {
        if (line.startsWith(START_FILE))
        {
            setCurrentLogEntry(new ChangeLogEntry());
            setCurrentFile(new ChangeLogFile(line.substring(START_FILE.length(),
                line.length())));
            setStatus(GET_REVISION);
        }
    }

    /**
     * Process the current input line in the Get Revision state.
     * @param line a line of text from the cvs log output
     */
    private void processGetRevision(String line) 
    {
        if (line.startsWith(REVISION_TAG))
        {
            getCurrentFile().setRevision(line.substring(REVISION_TAG.length()));
            setStatus(GET_DATE);
        }
        else if (line.startsWith(END_FILE))
        {
            // If we encounter an end of file line, it means there 
            // are no more revisions for the current file.
            // there could also be a file still being processed.
            setStatus(GET_FILE);
            addEntry(getCurrentLogEntry(), getCurrentFile());
        }
    }

    /**
     * Process the current input line in the Get Date state.
     * @param line a line of text from the cvs log output
     */
    private void processGetDate(String line)
    {
        if (line.startsWith(DATE_TAG))
        {
            StringTokenizer tokenizer = new StringTokenizer(line, " ;");
            // date: YYYY/mm/dd HH:mm:ss; author: name
            tokenizer.nextToken(); // date tag
            String date = tokenizer.nextToken(); 
            String time = tokenizer.nextToken();
            getCurrentLogEntry().setDate(date + " " + time);
            tokenizer.nextToken(); // author tag
            // assumes author can't contain spaces
            String author = tokenizer.nextToken();
            getCurrentLogEntry().setAuthor(author);
            setStatus(GET_COMMENT);
        }
    }

    /**
     * Process the current input line in the Get Comment state.
     * @param line a line of text from the cvs log output
     */
    private void processGetComment(String line)
    {
        if (line.startsWith(START_REVISION))
        {
            // add entry, and set state to get revision
            addEntry(getCurrentLogEntry(), getCurrentFile());
            // new change log entry
            setCurrentLogEntry(new ChangeLogEntry());
            // same file name, but different rev
            setCurrentFile(new ChangeLogFile(getCurrentFile().getName()));
            setStatus(GET_REVISION);
        }
        else if (line.startsWith(END_FILE))
        {
            addEntry(getCurrentLogEntry(), getCurrentFile());
            setStatus(GET_FILE);
        }
        else
        {
            // keep gathering comments
            getCurrentLogEntry().setComment(
                getCurrentLogEntry().getComment() + line + "\n");
        }
    }

    /**
     * Getter for property currentFile.
     * @return Value of property currentFile.
     */
    private ChangeLogFile getCurrentFile()
    {
        return currentFile;
    }
    
    /**
     * Setter for property currentFile.
     * @param currentFile New value of property currentFile.
     */
    private void setCurrentFile(ChangeLogFile currentFile)
    {
        this.currentFile = currentFile;
    }
    
    /**
     * Getter for property currentLogEntry.
     * @return Value of property currentLogEntry.
     */
    private ChangeLogEntry getCurrentLogEntry()
    {
        return currentLogEntry;
    }
    
    /**
     * Setter for property currentLogEntry.
     * @param currentLogEntry New value of property currentLogEntry.
     */
    private void setCurrentLogEntry(ChangeLogEntry currentLogEntry)
    {
        this.currentLogEntry = currentLogEntry;
    }
    
    /**
     * Getter for property status.
     * @return Value of property status.
     */
    private int getStatus()
    {
        return status;
    }
    
    /**
     * Setter for property status.
     * @param status New value of property status.
     */
    private void setStatus(int status)
    {
        this.status = status;
    }
}
