package org.apache.maven.changelog;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *    "Apache Maven" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Maven", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;

/**
 * Tests for the {@link ChangeLogEntry} class
 *
 * @author dion
 * @version $Id: ChangeLogEntryTest.java,v 1.7 2002/04/21 11:14:22 dion Exp $
 */
public class ChangeLogEntryTest extends TestCase {
    
    /** the {@link ChangeLogEntry} used for testing */
    private ChangeLogEntry instance;
    
    /**
     * Create a test with the given name
     * @param testName the name of the test
     */
    public ChangeLogEntryTest(String testName) {
        super(testName);
    }
    
    /**
     * Run the test using the {@link TestRunner}
     * @param args command line provided arguments
     */
    public static void main(String[] args) {
        TestRunner.run(suite());
    }
    
    /**
     * Create a test suite for this class
     * @return a {@link TestSuite} for all tests in this class
     */
    public static Test suite() {
        return new TestSuite(ChangeLogEntryTest.class);
    }
    
    /**
     * Initialize per test data
     */
    public void setUp() throws Exception
    {
        instance = new ChangeLogEntry();
        instance.setAuthor("dion");
        instance.setComment("comment");
        instance.setDate("2002/04/01 00:00:00");
    }
    
    /** 
     * Test of addFile methods: using ChangeLogFile
     */
    public void testAddFileWithFile() {
        ChangeLogFile file = new ChangeLogFile("maven:dummy");
        instance.addFile(file);
        assertTrue("File name not found in list",
            instance.toString().indexOf("maven:dummy") != -1 );
    }

    /** 
     * Test of addFile methods: using file & revision
     */
    public void testAddFileWithFileAndRevision() {
        instance.addFile("maven:dummy", "x.y");
        assertTrue("File name not found in list", 
            instance.toString().indexOf("maven:dummy") != -1);
        assertTrue("Revision not found in list", 
            instance.toString().indexOf("x.y") != -1);
    }

    /** 
     * Test of toString method
     */
    public void testToString() {
        //dion, Mon Apr 01 00:00:00 EST 2002, [], comment
        String value = instance.toString();
        assertTrue("author not found in string", value.indexOf("dion") != -1);
        assertTrue("comment not found in string", 
            value.indexOf("comment") != -1);
        assertTrue("date not found in string", 
            value.indexOf("Mon Apr 01") != -1);
        assertTrue("empty file list not found in string", 
            value.indexOf("[]") != -1);
    }
    
    /**
     * Test of toXML method
     */
    public void testToXML() {
        String value = instance.toXML();
        String trimmedValue = value.trim();
        assertTrue("XML doesn't start with changelog-entry",
            trimmedValue.startsWith("<changelog-entry>"));
        assertTrue("XML doesn't contain date", 
            value.indexOf("<date>2002-04-01</date>") != -1);
        assertTrue("XML doesn't contain author CDATA", 
            value.indexOf("<author><![CDATA[dion]]></author>") != -1);
        assertTrue("XML doesn't contain comment CDATA",
            value.indexOf("<msg><![CDATA[comment]]></msg>") != -1);
    }
    
    /**
     * Test of getAuthor method
     */
    public void testGetAuthor() {
        assertEquals("Author value not retrieved correctly", "dion",  
            instance.getAuthor());
    }
    
    /** 
     * Test of setAuthor method
     */
    public void testSetAuthor() {
        instance.setAuthor("maven:dion");
        assertEquals("Author not set correctly", "maven:dion", 
            instance.getAuthor());
    }
    
    /** 
     * Test of getComment method
     */
    public void testGetComment() {
        assertEquals("Comment value not retrieved correctly", "comment", 
            instance.getComment());
    }
    
    /**
     * Test of setComment method
     */
    public void testSetComment() {
        instance.setComment("maven:comment");
        assertEquals("Comment not set correctly", "maven:comment", 
            instance.getComment());
    }
    
    /**
     * Test of getDate method
     */
    public void testGetDate() {
        Calendar cal = Calendar.getInstance();
        cal.set(2002, 3, 1, 0, 0, 0);
        cal.set(Calendar.MILLISECOND, 0);
        assertEquals("Date value not retrieved correctly",  cal.getTime(), 
            instance.getDate());
    }
    
    /**
     * Test of setDate method with Date object
     */
    public void testSetDate() {
        Calendar cal = Calendar.getInstance();
        Date date = cal.getTime();
        instance.setDate(date);
        assertEquals("Date value not set correctly", date, instance.getDate());
    }
    
    /**
     * Test of setDate method with String
     */
    public void testSetDateFromString() {
        instance.setDate("2002/03/04 00:00:00");
        Calendar cal = Calendar.getInstance();
        cal.set(2002, 2, 4, 0, 0, 0);
        cal.set(Calendar.MILLISECOND, 0);
        assertEquals("Date value not set correctly from a string", 
            cal.getTime(), instance.getDate());
    }

    
    /** 
     * Test of getDateFormatted method
     */
    public void testGetDateFormatted() {
        assertEquals("Date not formatted correctly", "2002-04-01",
            instance.getDateFormatted());
    }

    /** 
     * Test of getDateFormatted method
     */
    public void testGetTimeFormatted() {
        assertEquals("Time not formatted correctly", "00:00:00",
            instance.getTimeFormatted());
    }

    // Add test methods here, they have to start with 'test' name.
    // for example:
    // public void testHello() {}
    
    
}
