#!/bin/sh

################################################################################
#
# WARNING: Tread lightly in here. This is the primary driver script used on many
#          different Unix platforms. Be sure to rely only on lowest-common-
#          denominator functionality available on all systems. See
#          scripts/platform/platform.sh for platform-specific
#          functionality.
#
################################################################################

BN=$(basename "$0" .sh)

# Source in system utilties based on the platform.
SYMPATH="$(cd "$(dirname "$0")" && pwd -P)"
. "${SYMPATH}/scripts/base.sh"

################################################################################

start()
{
    local services service
    services="$@"

    for service in $services; do
        start_service $service
    done
}

stop()
{
    local services service running i port

    services="$@"
    failed=false
    for service in $services; do
        stop_service $service || failed=true
    done

    if $failed; then
        echo "Failed to shutdown" 1>&2
    fi

    if echo "$services" | grep -q contrib; then
        port=`get_contrib_port`
        adjust_firewall Contribution remove tcp $port
        adjust_firewall Contribution remove udp $port
    fi
}

################################################################################

cmd=$1
[ $# -gt 0 ] && shift

if [ 0 -eq $# ]; then
    ALL_SERVICES=true
    SERVICES='contrib sync web'
else
    ALL_SERVICES=false
    SERVICES="$@"
fi

case "$cmd" in
    start)
        adjust_cron true $SERVICES
        start $SERVICES

        for service in $SERVICES; do
            if [ $service = "web" ]; then
                sleep 2
                outfn=$(util_mktemp /tmp/testweb.XXXXXX)
                if ! util_test_url "http://127.0.0.1:$(get_webui_port)/" 1 10 "$outfn"; then
                    echo "Web service is not healthy" 1>&2
                    cat "$outfn" 1>&2
                fi
                rm -f "$outfn"
            fi
        done
        ;;

    stop)
        if [ `id -u` -ne 0 ]; then
            echo "Must be root to stop ${SERVICES}" 1>&2
            exit 1
        fi

        adjust_cron false $SERVICES
        stop $SERVICES
        ;;

    restart)
        for service in $SERVICES; do
            stop $service
            sleep 1
            start $service
        done
        ;;

    status)
        echo "${BN}: $(util_symver -long)"
        echo
        for service in $SERVICES; do
            pid=`util_pidof $service`
            if [ -z "$pid" ]; then
                echo "symform${service} not running"
            else
                echo "symform${service} running as $pid"
            fi
        done
        ;;

    statusexit)
        for service in $SERVICES; do
            pid=`util_pidof $service`
            if [ -z "$pid" ]; then
                exit 1
            fi
        done
        exit 0
        ;;

    service)
        adjust_cron true $SERVICES

        for service in $SERVICES; do
            symservice $service
        done
        ;;

    dump)
        for service in $SERVICES; do
            echo "generating thread dump of symform${service}."
            dump_service $service
        done
        ;;

    runMono)
        default_args='mono'
        # allow user to skip passing in "asroot" with default to run as root
        [ "$1" = 'true' -o "$1" = 'false' ] || default_args="${default_args} true"
        run_mono $default_args "$@"
        ;;

    configure)
        symconfig http://127.0.0.1:$(get_webui_port)/
        ;;

    uploadLogs)
        logfn="${LOGSPATH}/symformloguploader-mono.log"
        util_rotate_log "$logfn"

        for logfn in "${LOGSPATH}/"*"-task.log"; do
            util_rotate_log "$logfn"
        done

        loguploaderpid="$(util_pidof loguploader)"
        if [ -z "$loguploaderpid" ]; then
            util_log "Starting LogUploader v$(util_symver) on ${SYMFORM_OS_VERSION_STRING}" >>"$logfn"
            run_mono symformloguploader true "${SYMPATH}/bin/LogUploader.exe" "${WORKDIR}/node.config" >>"$logfn" 2>&1 &
            renice 10 -p $!
        else
            util_log "LogUploader already running as pid $loguploaderpid" >>"$logfn"
        fi
        ;;

    rootDirectories)
        list_root_directories
        ;;

    adjustFirewall)
        adjust_firewall "$@"
        ;;

    install)
        base_install
        ;;

    # To support upgrade from 3.9.x
    postInstall)
        base_install
        ;;

    uninstall)
        base_uninstall
        ;;

    purge)
        base_purge
        ;;

    *)
        cat 1>&2 <<EOF

Usage: ${BN} {uploadLogs|status}
       ${BN} {start|stop|restart} [<service> [<service>...]]

  In the case of {start|stop|restart} additional parameters can specify one or
  more services (e.g. sync, contrib, and/or web).

  If no additional parameters are provided, the action is applied to all
  services.

EOF
        exit 1
esac
exit 0
