#!/bin/sh

################################################################################
#
# WARNING: Tread lightly in here. This is the updater script for making sure the
#          latest Symform Node software is installed. This script is used on
#          many different Unix platforms. Be sure to rely only on lowest-common-
#          denominator functionality available on all systems. See
#          platform/platform.sh for platform-specific functionality.
#
################################################################################

usage()
{
    cat <<EOF
Usage: $0 [--log-stdout] [--download-only]"
        --log-stdout     This script will produce logs to standard output as well as to a file.
        --download-only  This script is to run without any other dependencies or config files.
EOF
    exit 1
}

log()
{
    util_log "$*" >>"$UPDATE_LOG"
    $LOG_STDOUT && echo "$*" || :
}

fail()
{
    log "${1}"
    log 'Update script failed'
    exit 1
}

warn()
{
    log "${1}"
    return 1
}

downloadFile()
{
    local url outfile message

    url="${1}"
    outfile="${2}"
    message="${3}"

    rm -f "${outfile}"
    log "Downloading ${message} file: ${url}"

    util_download_file "${url}" "${outfile}" >>"$UPDATE_LOG" 2>&1

    if [ $? -ne 0 -o ! -f "${outfile}" ]; then
        warn "Failed to retrieve ${message}"
        return 1
    fi

    # read the 1st 5 characters to see if we got XML content
    read hdr rest_of_line < "${outfile}"
    if [ "${hdr}" = "<?xml" ]; then
        warn "Failed to download ${message}. ${outfile} seems be an xml file"
        if grep -q '<Error>' "${outfile}"; then
            warn "Contents of XML file follow:"
            warn "$(cat "${outfile}")"
        fi
        return 1
    fi
    return 0
}

validatePackage()
{
    local hash hash_error_code verified verify_error_code
    local package_path signature_path

    package_path="$1"
    signature_path="$2"

    hash=`openssl dgst -sha1 < "${package_path}" | sed 's/^.*[^0-9a-f]\([0-9a-f]*\)$/\1/'`
    hash_error_code=$?
    verified=`openssl rsautl -verify -inkey "${SYMFORM_CERT_PATH}" -keyform PEM -certin -in "${signature_path}" | sed 's/^.*[^0-9a-f]\([0-9a-f]*\)$/\1/'`
    verify_error_code=$?

    if [ "$hash_error_code" -eq 0 -a "$verify_error_code" -eq 0 ]; then
        if [ "${verified}" = "${hash}" ]; then
            log 'Package verification succeeded'
        else
            warn "Package verification failed - Signature hash: ${verified} File hash: ${hash}" || return $?
        fi
    else
        warn "Package verification failed - Hashing return code: ${hash_error_code} Verifing return code: ${verify_error_code}" || return $?
    fi
    return 0
}

#
# Ask cloud control is scope upgrade is available for this node
# If the answer is yes, set global variable TRY_SCOPED_UPGRADE
# to true
#
check_scoped_upgrade()
{
    local version platform arch node_config_path url httpcode tmpfn

    version="$1"
    platform="$2"
    arch="$3"
    node_config_path="$4"

    nodeId=$(sed -ne 's/.*nodeId="\([0-9A-F]\{40\}\)".*/\1/p' "${node_config_path}" 2>/dev/null)

    if [ -z "${version}" -o "${version}" = "${NO_LOCAL_VERSION}" -o \
         -z "${nodeId}"  -o -z "${platform}" -o -z "${arch}" ]; then
        return 1
    fi

    url="${SCOPE_DEPLOY_BASE_URL}?platform=${platform}&arch=${arch}&ver=${version}&nodeId=${nodeId}"

    log "Checking for scope deployment at $url"
    tmpfn="/tmp/symuphttpcode.$$"
    httpcode="$(util_download_file --http-status-code "$url" "${tmpfn}")"
    rm -f "${tmpfn}"
    if [ "${httpcode}" = "204" ]; then
        log "HTTP code: $httpcode. Scope upgrade available, will try to upgrade to it."
        return 0
    fi
    log "HTTP code: $httpcode. No scope upgrade available at this time. Trying regular upgrade."
    return 1
}

#
# Try to do upgrade
# Return 0 in case of successful upgrade or if no upgrade required
#        1 if upgrade fails for any reason
# 
do_upgrade()
{
    local workdir local_version version_url pkg_url signature_url upgrade_ext
    local version_file_path package_path signature_path error_code

    if [ $# -lt 6 ]; then
        cat >&2 <<EOF
Usage: do_upgrade workdir local_version version_url pkg_url signature_url upgrade_ext
    workdir       - working directory
    local_version - version currently installed
    version_url   - URL to get version file
    pkg_url       - URL to get the actual package
    signature_url - URL to get package signature
    upgrade_ext   - file extension for this platform
EOF
        return 1
    fi

    workdir="$1"
    local_version="$2"
    version_url="$3"
    pkg_url="$4"
    signature_url="$5"
    upgrade_ext="$6"

    version_file_path="${workdir}/currentversion"
    package_path="${workdir}/update.${upgrade_ext}"
    signature_path="${workdir}/signature2"

    downloadFile "${version_url}" "${version_file_path}" "version" || return $?

    current_version="$(util_symver "${version_file_path}")"
    result=`util_compare_versions "$local_version" "$current_version"` 

    if [ $result -eq 0 ]; then
        log "Up to date on version ${local_version}"
    elif [ $result -lt 0 ]; then
        log "Out of date version ${local_version}"
        log "Updating to ${current_version}"

        downloadFile "${pkg_url}" "${package_path}" "package" || return $?

        if $VALIDATE_PACKAGE; then
            downloadFile "${signature_url}" "${signature_path}" "signature" || return $?
            validatePackage "${package_path}" "${signature_path}" || return $?
        fi
    
        if ! $DOWNLOAD_ONLY; then
            # from here on out, we want to execute and catch errors ourselves
    
            if [ -f "${SYMPATH}/SymformNode.sh" ]; then
                log 'Stopping the service...'
    
                "${SYMPATH}/SymformNode.sh" stop >> "$UPDATE_LOG" 2>&1
    
                if [ $? -ne 0 ]; then
                    warn 'Failed to stop service for update' || return $?
                fi
            fi
    
            log 'Installing the update...'
            if install_update "$package_path" >> "$UPDATE_LOG" 2>&1; then
                rm -f "$package_path"
                log 'Update succeeded'
            else
                log 'Package installation failed'
                if [ -f "${SYMPATH}/SymformNode.sh" ]; then
                    log 'Starting the old service again...'
                    "${SYMPATH}/SymformNode.sh" start >> "$UPDATE_LOG" 2>&1
                fi
                return 1
            fi
        fi
    else
        log "Out of date version online ${current_version}"
        log "Up to date on version ${local_version}"
    fi
    return 0
}

start_upgrade()
{
    local try_scoped version_url pkg_url signature_url error_code

    try_scoped="$1"

    # Set up various download urls
    if $try_scoped; then
        version_url="${SCOPE_VERSION_URL}"
        pkg_url="${SCOPE_PACKAGE_URL}"
        signature_url="${SCOPE_SIGNATURE_URL}"
    else
        version_url="${VERSION_URL}"
        pkg_url="${PACKAGE_URL}"
        signature_url="${SIGNATURE_URL}"
    fi

    do_upgrade "${WORKDIR}" "${LOCAL_VERSION}" "$version_url" "$pkg_url" "$signature_url" "${SYMFORM_UPGRADE_EXT}"
    error_code=$?

    if [ $error_code -ne 0 ] && $try_scoped; then
        log "Scoped upgrade failed. Falling back to normal upgrade URL"
        do_upgrade "${WORKDIR}" "${LOCAL_VERSION}" "${VERSION_URL}" "${PACKAGE_URL}" "${SIGNATURE_URL}" "${SYMFORM_UPGRADE_EXT}"
        error_code=$?
    fi
    return $error_code
}

setup_environment()
{
    local sympath

    sympath="${1}"

    DOWNLOAD_ONLY=false
    LOG_STDOUT=false
    VALIDATE_PACKAGE=true
    NO_LOCAL_VERSION="NOT_INSTALLED"

    . "${sympath}/scripts/base.sh" || exit 1

    if [ "netgear" = "$SYMFORM_PLATFORM" ] && [ "deb" = "$SYMFORM_UPGRADE_EXT" ]; then
        SYMFORM_PLATFORM="linux"
    fi

    BASE_URL='http://download.symform.com'
    VERSION_URL="${BASE_URL}/${DEPLOYMENT}/current/${SYMFORM_PLATFORM}/${SYMFORM_ARCH}/version"
    PACKAGE_URL="${BASE_URL}/${DEPLOYMENT}/current/${SYMFORM_PLATFORM}/${SYMFORM_ARCH}/Symform${SYMFORM_PACKAGE_NAME_POSTFIX}.${SYMFORM_UPGRADE_EXT}"
    SIGNATURE_URL="${BASE_URL}/${DEPLOYMENT}/current/${SYMFORM_PLATFORM}/${SYMFORM_ARCH}/${SIGNATURE_PREFIX}signature${SYMFORM_PACKAGE_NAME_POSTFIX}2"

    SCOPE_VERSION_URL="${BASE_URL}/${DEPLOYMENT}/scope/${SYMFORM_PLATFORM}/${SYMFORM_ARCH}/version"
    SCOPE_PACKAGE_URL="${BASE_URL}/${DEPLOYMENT}/scope/${SYMFORM_PLATFORM}/${SYMFORM_ARCH}/Symform${SYMFORM_PACKAGE_NAME_POSTFIX}.${SYMFORM_UPGRADE_EXT}"
    SCOPE_SIGNATURE_URL="${BASE_URL}/${DEPLOYMENT}/scope/${SYMFORM_PLATFORM}/${SYMFORM_ARCH}/${SIGNATURE_PREFIX}signature${SYMFORM_PACKAGE_NAME_POSTFIX}2"

    echo "-------------------"
    echo ${PACKAGE_URL}
    echo ${SCOPE_PACKAGE_URL}
    echo "-------------------"
    
    NODE_CONFIG="${WORKDIR}/node.config"
    SCOPE_DEPLOY_BASE_URL="http://${DEPLOYMENT}.symform.com/upgrade"
}

if [ -z "${IGNORE_MAIN}" ]; then

    if [ `id -u` -ne 0 ]; then
        echo 'Must be root to update Symform installation' 1>&2
        exit 2
    fi

    [ "$1" = "--log-stdout" ] && shift && LOG_STDOUT=true
    [ "$1" = "--download-only" ] && shift && DOWNLOAD_ONLY=true

    if [ $# -ge 2 -o "$1" = "-h" -o "$1" = "--help" ]; then
        # usage is expected to exit and not return
        usage
    fi

    # This script expects to live one directory below SYMPATH.
    SYMPATH="$(cd "$(dirname "$0")/.." && pwd -P)"

    # setup various variables, read base.sh
    setup_environment "${SYMPATH}"

    UPDATE_LOG="${LOGSPATH}/symformupdater.log"

    trap "util_rotate_log \"$UPDATE_LOG\"" 0

    log 'Update script started'

    TRY_SCOPED_UPGRADE=false
    LOCAL_VERSION="$(util_symver 2>/dev/null || echo "${NO_LOCAL_VERSION}")"

    check_scoped_upgrade "${LOCAL_VERSION}" "${SYMFORM_PLATFORM}" "${SYMFORM_ARCH}" "${NODE_CONFIG}" && TRY_SCOPED_UPGRADE=true

    start_upgrade $TRY_SCOPED_UPGRADE
    error_code=$?

    log 'Update script finished'

    exit $error_code
fi
