#
# This script is sourced from SymformNode.sh and SymformUpdater.sh. See template.sh
# for more information.
#
# This script REQUIRES that SYMPATH is defined.
#

. "${SYMPATH}/scripts/util.sh"

#
# Overloadable
#

adjust_cron()
{
    util_adjust_cron "$@"
}

# Add firewall rules to allow Symform applications to work correctly.
adjust_firewall()
{
    :
}

post_install()
{
    :
}

post_uninstall()
{
    :
}

post_purge()
{
    :
}

non_volatile_dir()
{
    :
}

unprivileged_user()
{
    :
}

crimson_rijndael_buffer_block_size()
{
    :
}

crimson_sha1_buffer_block_size()
{
    :
}

crimson_sha256_buffer_block_size()
{
    :
}

# Establish the date formate for logging. Optionally overriden if some of the
# format specifiers aren't available.
get_date_string()
{
    echo $(date -u +'%Y-%m-%d %H:%M:%S,%03NZ')
}

run_unprivileged()
{
    user=$1
    shift
    exec su -c "$(for arg; do printf "'${arg}' "; done)" $user
}

symform_os_version_string()
{
    echo "$(uname -a)"
}

get_processor_count()
{
    echo `grep -c '^[Pp]rocessor[[:space:]]' /proc/cpuinfo 2>/dev/null || echo 1`
}

start_service()
{
    # we want to daemonize so try to locate nohup
    if [ -e "${SYMPATH}/bin/nohup" ]; then
        NOHUP="${SYMPATH}/bin/nohup"
    elif which nohup >/dev/null 2>&1; then
        NOHUP=nohup
    fi

    symservice $1 &
}

stop_service()
{
    local service pid
    service=$1
    pid="$(util_pidof $service)"

    util_really_kill "$pid" "Forcing ${service} (${pid}) to shut down"
}

################################################################################
# Legacy

post_adjust_cron()
{
    :
}

################################################################################

# Execute a command, optionally with lowered privileges (i.e. not as root).
symexec()
{
    local asroot args
    asroot=$1
    shift
    if $asroot; then
        exec "$@"
    else
        run_unprivileged $(unprivileged_user) "$@"
    fi
}

# Open web browser connection to web service for configuration.
symconfig()
{
    xdg-open "$1"
}

# Get the TCP port for the contribution service (might be an empty string if no
# port is configured).
get_contrib_port()
{
    if [ -f "${WORKDIR}/node.config" ]; then
        # consider optionally using xmllint --xpath to restict by parent element
        sed -n 's/^.* port="\([0-9][0-9]*\)".*$/\1/p' "${WORKDIR}/node.config"
    fi
}

get_webui_port()
{
    if [ -z "$SYMFORM_WEB_UI_PORT" ]; then
        echo "59234"
    else
        echo "$SYMFORM_WEB_UI_PORT"
    fi
}

base_install()
{
    # Call platform specific post_install
    post_install

    # Migrate to non_volatile_dir (for 3.9)
    if [ -n "$(non_volatile_dir)" ]; then
        if [ -e "$(non_volatile_dir)/node.config" -o -e "$(non_volatile_dir)/symform.conf" -o -e "$(non_volatile_dir)/.mono" ]; then
            super=$(cd "$(non_volatile_dir)/../"; pwd -P)
            rm -rf "${super}/tempsymform"
            mv "$(non_volatile_dir)" "${super}/tempsymform"
            mkdir "$(non_volatile_dir)"
            mv "${super}/tempsymform" "$(non_volatile_dir)/lib"
            mv -f "$(non_volatile_dir)/lib/logs" "$(non_volatile_dir)/log"
            rm -rf "$(non_volatile_dir)/lib/lib"
            rm -rf "$(non_volatile_dir)/lib/log"
            ln -s "$(non_volatile_dir)/log/" "$(non_volatile_dir)/logs"
        fi
    fi

    # Migrate updater.config to symform.conf (for 3.9)
    if [ -e "${WORKDIR}/updater.config" ]; then
        mv "${WORKDIR}/updater.config" "${WORKDIR}/symform.conf"
    fi

    if [ ! -e "${WORKDIR}/symform.conf" ]; then
        echo "DEPLOYMENT=${DEPLOYMENT}" >"${WORKDIR}/symform.conf"
    fi

    if [ ! -L /etc/symform.conf ]; then
        [ -e /etc/symform.conf ] && mv /etc/symform.conf "${WORKDIR}/symform.conf"
        ln -sf "${WORKDIR}/symform.conf" /etc/symform.conf
    fi

    # make sure we create the keypairs with the right permissions so that it's
    # only readable by the symform user
    if [ ! -d "${WORKDIR}/.mono/keypairs" ]; then
        mkdir -p "${WORKDIR}/.mono/keypairs"
        chmod 700 "${WORKDIR}/.mono/keypairs"
    fi

    # Set up our executables to be appropriately named for process listing
    for service in sync contrib web loguploader; do
        ln -sf "${SYMPATH}/mono/bin/mono" "${SYMPATH}/bin/symform${service}"
    done

    # point our services to a more standard logging location
    if [ ! -L "${SYMPATH}/bin/logs" -a "$LOGSPATH" != "${SYMPATH}/bin/logs" ]; then
        mkdir -p "$LOGSPATH"
        chmod 777 "$LOGSPATH"
        rm -rf "${SYMPATH}/bin/logs"
        mkdir -p "${SYMPATH}/bin"
        ln -s "$LOGSPATH" "${SYMPATH}/bin/logs"
    fi

    # Add needed certs for ssl communication to symform.com domains
    if ! HOME="${WORKDIR}" XDG_DATA_HOME="${WORKDIR}" XDG_CONFIG_HOME="${WORKDIR}" "${SYMPATH}/mono/bin/mono" "${SYMPATH}/mono/lib/mono/2.0/mozroots.exe" --import --sync --file "${SYMPATH}/certs/certdata.txt"; then
        echo "Failed to add certificates"
        exit 1
    fi

    # Make sure the basic directories have the correct permissions
    if [ -n "$(unprivileged_user)" ]; then
        setup_unprivileged_perms
    fi

    # Setup permissions
    chmod -R +rx "${SYMPATH}/mono"
    chmod -R +rx "${SYMPATH}/bin"
    chmod -R +rx "${SYMPATH}/scripts"
    [ -c /dev/crypto ] && chmod +x /dev/crypto

    # Best-effort attempt to get a machine keypair generated before the WebUI will need it
    # (this can take 40 seconds or more on underpowered machines)
    run_mono mono true "${SYMPATH}/bin/monokeypairtool.exe" >>"${LOGSPATH}/monokeypairtool.log" 2>&1 </dev/null &

    return 0
}

base_uninstall()
{
    [ -L "/var/lib/symform" ] && rm -f /var/lib/symform
    [ -L "/var/log/symform" ] && rm -f /var/log/symform

    rm -rf /var/run/symform
    
    # Call platform specific post_uninstall
    post_uninstall

    [ -L /opt/symform ] && rm -f /opt/symform
    [ -L /etc/symform.conf ] && rm -f /etc/symform.conf
}

base_purge()
{
    if [ -n "$(non_volatile_dir)" ]; then
        rm -rf "$(non_volatile_dir)/log"
        rm -rf "$(non_volatile_dir)/lib"
    fi

    util_clear_cron

    # Call platform specific purge
    post_purge

    base_uninstall
    
    rm -rf /var/lib/symform
    rm -rf /var/log/symform
    rm -rf /opt/symform
}

setup_unprivileged_perms()
{
    chown symform:symform "$WORKDIR"
    chown symform:symform "$LOGSPATH"
    chown symform:symform "$LOCKPATH"

    # need to always ensure that symform is the owner (especially for the
    # keypairs subdirectory, otherwise node.config reading will fail)
    chown -R symform:symform "${WORKDIR}/.mono"

    # The Sync and Contrib services will utilize a shared Symform directory in
    # the "ApplicationData" directories.
    mkdir -p "${WORKDIR}/Symform"
    chown symform:symform "${WORKDIR}/Symform"
    mkdir -p "${COMMONAPPDATA}/Symform"
    chown symform:symform "${COMMONAPPDATA}/Symform"
}

pre_mono_execute()
{
    if [ -n "$(crimson_rijndael_buffer_block_size)" ]; then
        export CRIMSON_AES_BUFFER_BLOCK_SIZE=$(crimson_rijndael_buffer_block_size)
        export CRIMSON_RIJNDAEL_BUFFER_BLOCK_SIZE=$(crimson_rijndael_buffer_block_size)
    fi

    if [ -n "$(crimson_sha1_buffer_block_size)" ]; then
        export CRIMSON_SHA1_BUFFER_BLOCK_SIZE=$(crimson_sha1_buffer_block_size)
    fi

    if [ -n "$(crimson_sha256_buffer_block_size)" ]; then
        export CRIMSON_SHA256_BUFFER_BLOCK_SIZE=$(crimson_sha256_buffer_block_size)
    fi

    export SYMFORM_WEB_UI_PORT="$(get_webui_port)"
}

symservice()
{
    local service pid asroot nodecfg logfn port bufferSize

    service="symform${1}"

    if [ `id -u` -ne 0 ]; then
        echo "Must be root to start ${service}" 1>&2
        exit 1
    fi

    pid=`util_pidof $1`
    if [ -n "$pid" ]; then
        echo "${service} appears to already be running as $pid"
        return 0
    fi

    if [ "$service" = 'symformcontrib' -o "$service" = 'symformsync' ]; then
        SYMFORM_MONO_EXTRA_PARAMS="${SYMFORM_MONO_EXTRA_PARAMS} --priority-class=BelowNormal --server"
    fi

    asroot=true
    if [ "$service" = 'symformcontrib' ]; then
        nodecfg="${WORKDIR}/node.config"
        if [ -e "$nodecfg" ]; then
            # we have to have full priviledges in these directories for contribution
            sed -n 's/^.* fragmentStorePath *= *"\([^"][^"]*\).*$/\1/p' "$nodecfg" | \
            while read d; do
                arrivals="${d}/arrivals"

                # If there is a tempfile signaling we should create the fragment store path, do so
                if [ -f /tmp/symform.createcontrib ]; then
                    mkdir -p "$arrivals"
                    rm -f /tmp/symform.createcontrib
                fi

                if [ ! -d "$arrivals" ]; then
                    echo "Wait 1 minute for contribution folder $d"
                    sleep 60
                fi

                if [ ! -d "$arrivals" ]; then
                    # We still can't find the folder, sleep for 30 minutes and exit with 1.
                    # The "service manager" will see the exit code 1 and restart the service.
                    echo "Can't find contribution folder $d; wait 30 minutes" 1>&2
                    sleep 1800
                    exit 1
                fi

                # ensure both the root and all children are owned by the right user
                [ -n "$(unprivileged_user)" ] && chown symform:symform "$d" "$d"/*
            done
        fi
        setup_unprivileged_perms
        port=`get_contrib_port`
        adjust_firewall Contribution add tcp $port
        adjust_firewall Contribution add udp $port

        [ -n "$(unprivileged_user)" ] && asroot=false
    fi

    # allow many file descriptors to be opened
    ulimit -n 10000

    bufferSize=2097152

    # ensure the the socket receive buffer mac is at least 4MB on linux
    if [ "`uname -s`" = "Linux" ]; then
        if [ "`sysctl -n net.core.rmem_max`" -lt ${bufferSize} ]; then 
            sysctl -w net.core.rmem_max=${bufferSize}
        fi

        if [ "`sysctl -n net.core.rmem_max`" -lt ${bufferSize} ]; then 
            echo Failed to set net.core.rmem_max to ${bufferSize}. Performance might be seriously degraded.
        fi
    fi

    # remove any lock files leftover from a crash (we've already checked for the
    # process, so this should be safe)
    rm -f "${LOCKPATH}/${service}.exe.lock"

    logfn="${LOGSPATH}/${service}-mono.log"

    # unlike normal logrotate functionality, we can only rotate mono logs when
    # the process is about to start since we can't tell it to reopen the
    # stdout/stderr file descriptors
    util_rotate_log "$logfn"

    util_log "Starting $service v$(util_symver) on ${SYMFORM_OS_VERSION_STRING}" >>"$logfn"

    pre_mono_execute ${1}

    MONO_OPTIONS="--process-title=${service}" run_mono "$service" $asroot "${SYMPATH}/mono/lib/mono/2.0/mono-service.exe" \
        -l:"${LOCKPATH}/${service}.exe.lock" -d:"$WORKDIR" -m:"$service" \
        --output=stderr --exclude-sigusr=true \
        "${SYMPATH}/bin/${service}.exe" >>"$logfn" 2>&1
}

run_mono()
{
    local service execname asroot mozup_fn

    if [ "$1" = 'mono' ]; then
        execname="$MONO"
    else
        service="$1"
        execname="${SYMPATH}/bin/$1"
    fi
    shift

    asroot=$1
    shift

    # set environment up for mono to use our work directory for its libraries,
    # certificates, and keypairs location
    export HOME="$WORKDIR"
    export XDG_DATA_HOME="$HOME"
    export XDG_CONFIG_HOME="$HOME"

    # contrib handles pathing safely, no need for help as it can impact file system access
    if [ "$service" != 'symformcontrib' ]; then
        export MONO_IOMAP=all
    fi

    export MONO_XMLSERIALIZER_THS=no
    export MONO_OPTIONS="${MONO_OPTIONS}"
    export MONO_PATH="${SYMPATH}/bin"

    symexec $asroot ${NOHUP} "$execname" ${MONO_OPTIONS} "$@" ${SYMFORM_MONO_EXTRA_PARAMS}
}

dump_service()
{
  local pid gdbcrash

  if [ $# -ne 1 ]; then
    echo 'usage:  dump_service <service>' 1>&2
    return 1
  fi

  pid=`util_pidof $1`
  if [ -z "$pid" ]; then
      echo "${service} does not appear to be running." 1>&2
      return 1
  fi

  if which lldb >/dev/null 2>&1; then

    lldbmonobacktrace="${TMP}/lldbmonobacktrace"
    cat > "${lldbmonobacktrace}" <<EOF
import lldb
import os, sys
import StringIO

def stop_reason_to_str(enum):
    """Returns the stopReason string given an enum."""
    if enum == lldb.eStopReasonInvalid:
        return "invalid"
    elif enum == lldb.eStopReasonNone:
        return "none"
    elif enum == lldb.eStopReasonTrace:
        return "trace"
    elif enum == lldb.eStopReasonBreakpoint:
        return "breakpoint"
    elif enum == lldb.eStopReasonWatchpoint:
        return "watchpoint"
    elif enum == lldb.eStopReasonSignal:
        return "signal"
    elif enum == lldb.eStopReasonException:
        return "exception"
    elif enum == lldb.eStopReasonPlanComplete:
        return "plancomplete"
    elif enum == lldb.eStopReasonThreadExiting:
        return "threadexiting"
    else:
        raise Exception("Unknown StopReason enum")
def get_args_as_string(frame, showFuncName=True):
    vars = frame.GetVariables(True, False, False, True) # type of SBValueList
    args = [] # list of strings
    for var in vars:
        args.append("(%s)%s=%s" % (var.GetTypeName(),
                                   var.GetName(),
                                   var.GetValue()))
    if frame.GetFunction():
        name = frame.GetFunction().GetName()
    elif frame.GetSymbol():
        name = frame.GetSymbol().GetName()
    else:
        name = ""
    if showFuncName:
        return "%s(%s)" % (name, ", ".join(args))
    else:
        return "(%s)" % (", ".join(args))
def get_function_names(thread):
    def GetFuncName(i):
        return thread.GetFrameAtIndex(i).GetFunctionName()
    return map(GetFuncName, range(thread.GetNumFrames()))
def get_symbol_names(thread):
    def GetSymbol(i):
        return thread.GetFrameAtIndex(i).GetSymbol().GetName()
    return map(GetSymbol, range(thread.GetNumFrames()))
def get_pc_addresses(thread):
    def GetPCAddress(i):
        return thread.GetFrameAtIndex(i).GetPCAddress()
    return map(GetPCAddress, range(thread.GetNumFrames()))
def get_filenames(thread):
    def GetFilename(i):
        return thread.GetFrameAtIndex(i).GetLineEntry().GetFileSpec().GetFilename()
    return map(GetFilename, range(thread.GetNumFrames()))
def get_line_numbers(thread):
    def GetLineNumber(i):
        return thread.GetFrameAtIndex(i).GetLineEntry().GetLine()
    return map(GetLineNumber, range(thread.GetNumFrames()))
def get_module_names(thread):
    def GetModuleName(i):
        return thread.GetFrameAtIndex(i).GetModule().GetFileSpec().GetFilename()
    return map(GetModuleName, range(thread.GetNumFrames()))
def get_stack_frames(thread):
    def GetStackFrame(i):
        return thread.GetFrameAtIndex(i)
    return map(GetStackFrame, range(thread.GetNumFrames()))
def print_stacktrace(thread, string_buffer = False):
    output = StringIO.StringIO() if string_buffer else sys.stdout
    target = thread.GetProcess().GetTarget()
    depth = thread.GetNumFrames()
    mods = get_module_names(thread)
    funcs = get_function_names(thread)
    symbols = get_symbol_names(thread)
    files = get_filenames(thread)
    lines = get_line_numbers(thread)
    addrs = get_pc_addresses(thread)
    if thread.GetStopReason() != lldb.eStopReasonInvalid:
        desc =  "stop reason=" + stop_reason_to_str(thread.GetStopReason())
    else:
        desc = ""
    print >> output, "Stack trace for thread id={0:#x} name={1} queue={2} ".format(
        thread.GetThreadID(), thread.GetName(), thread.GetQueueName()) + desc
    for i in range(depth):
        frame = thread.GetFrameAtIndex(i)
        function = frame.GetFunction()

        load_addr = addrs[i].GetLoadAddress(target)
        if str(frame.GetPCAddress())[0] == '0':
            monoframe=frame.EvaluateExpression('(char*)mono_pmip((void*)' + str(frame.GetPCAddress()) + ')')
            print >> output, "  frame #{num}: {addr:#016x} {value}".format(num=i, addr=load_addr, value=monoframe.GetSummary())
        elif not function:
            file_addr = addrs[i].GetFileAddress()
            start_addr = frame.GetSymbol().GetStartAddress().GetFileAddress()
            symbol_offset = file_addr - start_addr
            print >> output, "  frame #{num}: {addr:#016x} {mod}'{symbol} + {offset}".format(
                num=i, addr=load_addr, mod=mods[i], symbol=symbols[i], offset=symbol_offset)
        else:
            print >> output, "  frame #{num}: {addr:#016x} {mod}'{func} at {file}:{line} {args}".format(
                num=i, addr=load_addr, mod=mods[i],
                func='%s [inlined]' % funcs[i] if frame.IsInlined() else funcs[i],
                file=files[i], line=lines[i],
                args=get_args_as_string(frame, showFuncName=False) if not frame.IsInlined() else '()')
    if string_buffer:
        return output.getvalue()
def print_stacktraces(process, string_buffer = False):
    output = StringIO.StringIO() if string_buffer else sys.stdout
    print >> output, "Stack traces for " + str(process)
    for thread in process:
        print >> output, print_stacktrace(thread, string_buffer=True)
    if string_buffer:
        return output.getvalue()
def monobacktrace(debugger, args, result, dict):
  print_stacktraces(debugger.GetTargetAtIndex(0).GetProcess())
  return None
def __lldb_init_module (debugger, dict):
  debugger.HandleCommand('command script add -f monobacktrace.monobacktrace monobacktrace')
EOF

    lldbcrash="${TMP}/lldbcrash"
    cat > "${lldbcrash}" <<EOF
process attach --pid $pid
command script import "${lldbmonobacktrace}"
monobacktrace
detach
EOF

    lldb --source "$lldbcrash"
    
    rm -f "$lldbcrash"
    rm -f "$lldbmonobacktrace"

  elif which gdb >/dev/null 2>&1; then

    gdbcrash="${TMP}/gdbcrash"
    cat > "${gdbcrash}" <<EOF
define mono_backtrace
 select-frame 0
 set \$i = 0
 while (\$i < \$arg0)
   set \$foo = (char*) mono_pmip (\$pc)
   if (\$foo)
     printf "#%d %p in %s\n", \$i, \$pc, \$foo
   else
     frame
   end
   up-silently
   set \$i = \$i + 1
 end
end

attach $pid
info threads
thread apply all mono_backtrace 20
quit
EOF
    gdb -batch -quiet -x "$gdbcrash"
    rm -f "$gdbcrash"

  else
    echo "Error: No debugger found."
    return 1
  fi
}

################################################################################


# Add common paths (removing duplicates).
export PATH=`echo "/sbin:/usr/sbin:/bin:/usr/bin:/usr/local/sbin:/usr/local/bin:${PATH}" | \
awk -F: '{for(i=1;i<=NF;i++){if(!($i in a)){a[$i];printf s$i;s=":"}}}'`

#include debugtools in the path if bundled
if [ -d "${SYMPATH}/debugtools/bin" ]; then
  export PATH="${SYMPATH}/debugtools/bin":${PATH}
fi

SYMFORM_CERT_PATH="${SYMPATH}/scripts/SymformCodeSigningCertificate.pem"

# WORKDIR is also the LocalApplicationData folder
WORKDIR=/var/lib/symform
COMMONAPPDATA=/usr/share
LOGSPATH=/var/log/symform
LOCKPATH=/var/run/symform

MONO="${SYMPATH}/mono/bin/mono"

# Set up what service to use. Optionally overriden by the symform.conf if set
# from the web service.
DEPLOYMENT=control
if [ -e /etc/symform.conf ]; then
    . /etc/symform.conf
else
    [ -e "${WORKDIR}/symform.conf" ] && . "${WORKDIR}/symform.conf"
fi
export DEPLOYMENT

# Set up cron tasks so that the following actions will be taken on a daily basis
# (may be overriden based on the platform):
WEBTASK=start
SYNCTASK=start
CONTRIBTASK=restart

# If SYMPATH is not /opt/symform force create a symlink from SYMPATH to /opt/symform
#   if it is not already in place
if [ "${SYMPATH}" != /opt/symform ]; then
    if [ ! -e /opt/symform ] || [ "$(readlink /opt/symform)" != "${SYMPATH}" ]; then
        rm -rf /opt/symform
        mkdir -p /opt
        ln -sf "${SYMPATH}" /opt/symform
    fi
fi

# Source in platform specific script.
. "${SYMPATH}/scripts/platform/platform.sh"

export SYMFORM_OS_VERSION_STRING="$(symform_os_version_string)"
export SYMFORM_PLATFORM="$(symform_platform_string)"
export SYMFORM_ARCH="$(symform_arch)"

# Handle non-volitile setup
if [ -n "$(non_volatile_dir)" ]; then
    lib_dir="$(non_volatile_dir)/lib"
    mkdir -p "$lib_dir"
    lib_dir="$(cd "$lib_dir" ; pwd -P)"

    if [ ! -L /var/lib/symform ] || [ "$(readlink /var/lib/symform)" != "$lib_dir" ]; then
        rm -rf "/var/lib/symform"
        ln -sf "$lib_dir" /var/lib/symform
    fi

    log_dir="$(non_volatile_dir)/log"
    mkdir -p "$log_dir"
    log_dir="$(cd "$log_dir" ; pwd -P)"

    mkdir -p "$log_dir"
    if [ ! -L /var/log/symform ] || [ "$(readlink /var/log/symform)" != "$log_dir" ]; then
        rm -rf /var/log/symform
        ln -sf "$log_dir" /var/log/symform
    fi

    if [ ! -L "${SYMPATH}/bin/logs" ]; then
        rm -rf "${SYMPATH}/bin/logs"
        mkdir -p "${SYMPATH}/bin"
        ln -sf /var/log/symform "${SYMPATH}/bin/logs"
    fi
fi

# Force important directory locations to be absolute paths.
mkdir -p "$WORKDIR"
WORKDIR="$(cd "$WORKDIR" && pwd -P)"

mkdir -p "$LOGSPATH"
LOGSPATH="$(cd "$LOGSPATH" && pwd -P)"

mkdir -p "$COMMONAPPDATA"
COMMONAPPDATA="$(cd "$COMMONAPPDATA" && pwd -P)"

mkdir -p "$LOCKPATH"
