# Extract a Symform version value from current software or a file provided.
util_symver()
{
    local fn
    if [ -f "$1" ]; then
        fn="$1"
    elif [ "$1" = '-long' ]; then
        fn="${SYMPATH}/bin/version.long"
    else
        fn="${SYMPATH}/bin/version"
    fi
    # drop the carriage return
    awk '{gsub(/\r/,"");print}' "$fn"
}

# Behaves like seq but without support for any options
util_seq()
{
    local first=1
    local increment=1
    local last="$1"
    local comparison="<="

    if [ $# -lt 1 -o $# -gt 3 ]; then
        echo "Usage: util_seq [FIRST [INCREMENT]] LAST" 1>&2
        exit 1
    fi

    [ $# -gt 1 ] && first="$1" && last="$2"
    [ $# -gt 2 ] && increment="$2" && last="$3"
    [ $increment -lt 0 ] && comparison=">="

    echo | awk '{for(i='${first}'; i'${comparison}${last}'; i=i+'${increment}') {print i}}'
}

# Usage: util_test_url URL_TO_TEST MINIMUM_SUCCESSES MAXIMUM_TRIES TEMPFILE
util_test_url()
{
    local success_cnt i rc testcommand

    success_cnt=0
    for i in $(util_seq $3); do
        if [ "$(util_download_file --http-status-code $1 $4)" = '200' ]; then
            success_cnt=`expr $success_cnt + 1`
            [ $success_cnt -eq $2 ] && break
        else
            success_cnt=0
            sleep 5
        fi
    done

    [ $success_cnt -eq $2 ]
}

util_pidof()
{
    if which pidof >/dev/null 2>&1 && test -d /proc; then

        # For these systems, we can be more precise about first finding all mono
        # processes and then inspecting their command line directly. This is
        # used primarily for Busybox systems as that "ps" tool will restrict
        # command line output based on terminal size.

        local pid
        for pid in `pidof symform${1} mono`; do
            if test -f /proc/${pid}/cmdline && grep -q "symform${1}" /proc/${pid}/cmdline; then
                echo $pid
            fi
        done
    else
        # This should be safe on pretty much any Unix variant using a "normal"
        # ps report.
        ps -Awwww | awk '{if(/symform'"${1}"'/ && ! / awk .if/){print $1}}'
    fi
}

util_pidfromname()
{
    # This should be safe on pretty much any Unix variant using a "normal"
    # ps report.
    ps -Awwww | awk '{if(/'"${1}"'/ && ! / awk .if/){print $1}}'
}

# Kill, wait 30 seconds for it to die, if it doesn't kill -9 and wait 10 seconds.
# If it is still not dead error
util_really_kill()
{
    local pid="$1"
    local force_message="$2"

    # Kill and wait 30 seconds for it to take hold
    [ -n "$pid" ] || return 0
    kill $pid 2>/dev/null || return 0
    for i in $(util_seq 30); do
        kill -0 $pid 2>/dev/null || return 0
        sleep 1
    done

    # Force (kill -9) if pid not killed after 30 seconds
    echo "$force_message" 1>&2
    kill -9 $pid
    for i in $(util_seq 10); do
        kill -0 $pid 2>/dev/null || return 0
        sleep 1
    done
    return 1
}

util_mktemp()
{
    local filename=$1
    local not_done=true
    local tmpname=
    local number=

    [ -z "$MKTEMP_SEED" ] && MKTEMP_SEED=$$

    if [ $# -lt 1 ]; then
        filename=/tmp/tmp.XXXXXX
    fi

    while $not_done; do
        MKTEMP_SEED="$(expr $MKTEMP_SEED + 1)"
        number="$(echo | awk 'BEGIN {srand('$MKTEMP_SEED')} {print int(rand()*1000000)}')"
        tmpname=$(echo "$filename" | sed "s/XXXXXX/${number}/g")
        [ -e $tmpname ] || not_done=false
    done

    touch $tmpname
    echo $tmpname
}

# Usage [--http-status-code] [--timeout SECONDS] URL OUTPUT_FILE
#   --http-status-code  Status code is output and OUTPUT_FILE is where logging and errors go
#   URL                 The url to request
#   OUTPUT_FILE         The file to download to.  Pass a dash (-) to output to stdout
util_download_file()
{
    local result scheme curl_ca_option wget_ca_option

    HTTP_STATUS_CODE=false
    [ $1 = "--http-status-code" ] && shift && HTTP_STATUS_CODE=true
    TIMEOUT=10

    scheme=`echo "$1" | awk -F : '{print $1}'`

    if which curl >/dev/null 2>&1; then
        if $HTTP_STATUS_CODE; then
            ( set +e; curl -m 900 --connect-timeout $TIMEOUT -sS -w '%{http_code}' -o /dev/null "$1" 2>"$2" ; true )
        else
            curl -m 900 --connect-timeout $TIMEOUT -sS -o "$2" "$1"
        fi
    elif which wget >/dev/null 2>&1; then
        if $HTTP_STATUS_CODE; then
            ( set +e ; wget -O /dev/null --server-response --max-redirect 0 -T $TIMEOUT "$1" 2>"$2" ; true )
            result=$(awk '/^  *HTTP/{print $2}' "$2")
            if [ -z "$result" ]; then
                echo "000"
            else
                echo $result
            fi
        else
            wget -O "$2" -q -T $TIMEOUT "$1"
        fi
    else
        echo "Error: no curl or wget"
        exit 1
    fi
}

# Report how two dotted-value version strings compare to one another.
util_compare_versions()
{
    # the versions are stripped of anything but digits, periods, and spaces
    echo "$*" | awk '{
    gsub(/[^0-9. ]+/, "", $1);
    gsub(/[^0-9. ]+/, "", $2);
    n1 = split($1, v1, ".");
    n2 = split($2, v2, ".");
    for (i = 1; i <= n1; i++)
    {
        if (i in v2 == 0 || v1[i] > v2[i]){ print "1"; exit; }
        if (v1[i] < v2[i]){ print "-1"; exit; }
    }
    if (n1 < n2) print "-1";
    else print "0";
}'
}

# Rotate script-generated log files into a compressed file that will be
# recognized by the LogUploader. Expects to be passed the full path to the log
# file.
util_rotate_log()
{
    local ts logfn

    if [ -f "$1" ]; then
        # busybox may not have the nice "stat" command, so we have to rely on ls
        # output to get the file size (we don't want to bother rotating empty
        # files)
        if [ 0 -lt `ls -l "$1" | awk '{print $5}'` ]; then
            # we only rotate if we have not already generated one for this
            # hour--it's not as precise as the log4net rotatation, but good
            # enough for these purposes
            logfn="${1}.$(date -u '+%Y-%m-%dT%H00Z')"
            if [ ! -e "${logfn}.gz" ]; then
                mv -f "$1" "$logfn" && gzip "$logfn"
            fi
        fi
    fi
}

# Provide a common header for log content.
util_log()
{
    echo "$(get_date_string) [$$] $*"
}

util_adjust_cron()
{
    local starting tf service ups action min tasks

    starting=$1; shift

    # Seed the random number with the process id
    RANDOM=$$
    
    # Start the log uploader during the latter half of the hour
    loguploader_min="$(echo | awk 'BEGIN {srand('$seed')} {print int(rand()*25+30)}')"
    
    # Start the updater in the latter half of the hour after all the other tasks in the same hour. 
    # This ensures the job is not skipped when the crontab is regenerated by the start service jobs
    updater_min="$(echo | awk 'BEGIN {srand('$loguploader_min')} {print int(rand()*25+30)}')"

    # Restart the contribution service in the early half of the hour to allow the log uploader and updater to run in the latter half
    contribrestart_min="$(echo | awk 'BEGIN {srand('$updater_min')} {print int(rand()*20+5)}')"
    
    [ -d /etc/cron.d ] && CRONUSER=root

    if $starting; then
        # update our saved list of "starting" tasks so we continue to keep them
        # in the final list even if only one is stopped
        # SP-1267 Increase the auto start frequency for sync/contrib (SP-1059, SP-978)
        for service in "$@"; do
            if [ $service = 'web' -a -n "$WEBTASK" ]; then
                cat <<EOF >"${WORKDIR}/${service}.task"
0  0  * * * $CRONUSER /bin/sh '${SYMPATH}/SymformNode.sh' $WEBTASK web >>'${LOGSPATH}/${WEBTASK}web-task.log' 2>&1
EOF
            elif [ $service = 'sync' -a -n "$SYNCTASK" ]; then
                cat <<EOF >"${WORKDIR}/${service}.task"
2 */2 * * * $CRONUSER /bin/sh '${SYMPATH}/SymformNode.sh' $SYNCTASK sync >>'${LOGSPATH}/${SYNCTASK}sync-task.log' 2>&1
EOF
            elif [ $service = 'contrib' -a -n "$CONTRIBTASK" ]; then
                cat <<EOF >"${WORKDIR}/${service}.task"
${contribrestart_min} 0 * * * $CRONUSER /bin/sh '${SYMPATH}/SymformNode.sh' $CONTRIBTASK contrib >>'${LOGSPATH}/${CONTRIBTASK}contrib-task.log' 2>&1
4 */2 * * * $CRONUSER /bin/sh '${SYMPATH}/SymformNode.sh' start contrib >>'${LOGSPATH}/startcontrib-task.log' 2>&1
EOF
            fi
        done
    else
        for service in "$@"; do rm -f "${WORKDIR}/${service}.task"; done
    fi

    tf=$(util_mktemp /tmp/symformtasks.XXXXXX)

    # always make sure we stay up-to-date and push logs online--and make sure we
    # don't invoke a thundering herd (i.e. base time to run when we are
    # installing it
    cat <<EOF >"$tf"
${updater_min} */6 * * * $CRONUSER /bin/sh '${SYMPATH}/scripts/SymformUpdater.sh' >>'${LOGSPATH}/symformupdater-task.log' 2>&1
${loguploader_min} */6 * * * $CRONUSER /bin/sh '${SYMPATH}/SymformNode.sh' uploadLogs >>'${LOGSPATH}/uploadlogs-task.log' 2>&1
EOF

    # and then grab any new or existing tasks into our mix
    tasks="$(echo "${WORKDIR}"/*.task)"
    if [ "${tasks}" != "${WORKDIR}"'/*.task' ]; then
        cat "${WORKDIR}"/*.task >>"$tf"
    fi

    if [ -d /etc/cron.d ]; then
        mv -f "$tf" /etc/cron.d/symform-node
    else
        ( crontab -l 2>/dev/null | grep -Fvi "symform" || :; cat "$tf" ) | crontab -
        rm -f "$tf"
    fi

    post_adjust_cron
}

util_clear_cron()
{
    if [ -d /etc/cron.d ]; then
        rm -f /etc/cron.d/symform-node
    else
        crontab -l 2>/dev/null | grep -Fvi "symform" | crontab -
    fi

    post_adjust_cron
}
