/*
 * Copyright (c) 2000-2009 Canoo Engineering AG, Switzerland.
 */
package com.ulcjava.formmodel.interactive.hypo;

import static com.ulcjava.formmodel.interactive.hypo.HypoBean.*;

import java.math.BigDecimal;

import com.ulcjava.applicationframework.application.form.model.BigDecimalToDoubleMapper;
import com.ulcjava.applicationframework.application.form.model.FormModel;
import com.ulcjava.applicationframework.application.form.model.GreaterOrEqualValidator;
import com.ulcjava.applicationframework.application.form.model.IValidator;
import com.ulcjava.applicationframework.application.form.model.RangeValidator;
import com.ulcjava.base.shared.ErrorObject;

public class HypoFormModel extends FormModel<HypoBean> {
    
    public HypoFormModel(HypoBean bean) {
        super(bean);
    }
    
    @Override
    protected void addMappers() {
        addMapper(BigDecimal.class, new BigDecimalToDoubleMapper());
    }
    
    @Override
    protected IValidator[] createValidators() {
        return new IValidator[] {new RangeValidator<Double>(0.0, 18.0, FINANCING1_INTERESTRATE, FINANCING2_INTERESTRATE),
                new GreaterOrEqualValidator<Double>(0.0, OBJECTPRICE, OBJECTVALUE, EIGENMITTEL, YEARLYAMORTISATIONAMOUNT),
                new RangeValidator<Integer>(0, 25, AMORTISATIONYEARS), new IValidator() {
                    public void validate(FormModel<?> formModel) {
                        HypoBean hypoCalc = getBean();
                        BigDecimal loanAmount = hypoCalc.getLoanAmount();
                        BigDecimal objectValue = hypoCalc.getObjectValue();
                        if (loanAmount != null && objectValue != null) {
                            BigDecimal maxLoanAmount = objectValue.multiply(new BigDecimal("0.8"));
                            if (maxLoanAmount.compareTo(loanAmount) < 0) {
                                String errorMessage = "The total loan amount may not be greater then " + maxLoanAmount
                                        + " (80% of the objects value)";
                                setErrorForProperty(OBJECTVALUE, new ErrorObject(errorMessage, loanAmount));
                            }
                        }
                    }
                }};
    }
    
    @Override
    public void updatePresentationState() {
        BigDecimal amortisationAmount = getBean().getAmortisationAmount();
        boolean amortisationNecessary = amortisationAmount != null && amortisationAmount.doubleValue() > 0;
        setReadonly(AMORTISATIONYEARS, !amortisationNecessary);
        setReadonly(YEARLYAMORTISATIONAMOUNT, !amortisationNecessary);
        
        BigDecimal loanAmount1 = getBean().getFinancing(0).getLoanAmount();
        boolean loan1Necessary = loanAmount1 != null && loanAmount1.doubleValue() > 0;
        setReadonly(FINANCING1_INTERESTRATE, !loan1Necessary);
        setMandatory(FINANCING1_INTERESTRATE, loan1Necessary);
        
        BigDecimal loanAmount2 = getBean().getFinancing(1).getLoanAmount();
        boolean loan2Necessary = loanAmount2 != null && loanAmount2.doubleValue() > 0;
        setReadonly(FINANCING2_INTERESTRATE, !loan2Necessary);
        setMandatory(FINANCING2_INTERESTRATE, loan2Necessary);
    }
    
    public BigDecimal getMonthlyAmortisationAmmount() {
        BigDecimal yearlyAmortisationAmount = getBean().getYearlyAmortisationAmount();
        return yearlyAmortisationAmount == null ? null : roundAmount(yearlyAmortisationAmount.divide(BIG_DECIMAL_12, MC));
    }
    
    @Override
    protected void initState() {
        setReadonly(FINANCING1_LOANAMOUNT, FINANCING2_LOANAMOUNT);
        setMandatory(OBJECTPRICE, OBJECTVALUE);
    }
}
