/*

 ============================================================================
                   The Apache Software License, Version 1.1
 ============================================================================

 Copyright (C) 1999-2003 The Apache Software Foundation. All rights reserved.

 Redistribution and use in source and binary forms, with or without modifica-
 tion, are permitted provided that the following conditions are met:

 1. Redistributions of  source code must  retain the above copyright  notice,
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

 3. The end-user documentation included with the redistribution, if any, must
    include  the following  acknowledgment:  "This product includes  software
    developed  by the  Apache Software Foundation  (http://www.apache.org/)."
    Alternately, this  acknowledgment may  appear in the software itself,  if
    and wherever such third-party acknowledgments normally appear.

 4. The names "Apache Cocoon" and  "Apache Software Foundation" must  not  be
    used to  endorse or promote  products derived from  this software without
    prior written permission. For written permission, please contact
    apache@apache.org.

 5. Products  derived from this software may not  be called "Apache", nor may
    "Apache" appear  in their name,  without prior written permission  of the
    Apache Software Foundation.

 THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 FITNESS  FOR A PARTICULAR  PURPOSE ARE  DISCLAIMED.  IN NO  EVENT SHALL  THE
 APACHE SOFTWARE  FOUNDATION  OR ITS CONTRIBUTORS  BE LIABLE FOR  ANY DIRECT,
 INDIRECT, INCIDENTAL, SPECIAL,  EXEMPLARY, OR CONSEQUENTIAL  DAMAGES (INCLU-
 DING, BUT NOT LIMITED TO, PROCUREMENT  OF SUBSTITUTE GOODS OR SERVICES; LOSS
 OF USE, DATA, OR  PROFITS; OR BUSINESS  INTERRUPTION)  HOWEVER CAUSED AND ON
 ANY  THEORY OF LIABILITY,  WHETHER  IN CONTRACT,  STRICT LIABILITY,  OR TORT
 (INCLUDING  NEGLIGENCE OR  OTHERWISE) ARISING IN  ANY WAY OUT OF THE  USE OF
 THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 This software  consists of voluntary contributions made  by many individuals
 on  behalf of the Apache Software  Foundation and was  originally created by
 Stefano Mazzocchi  <stefano@apache.org>. For more  information on the Apache
 Software Foundation, please see <http://www.apache.org/>.

*/
package org.apache.cocoon.components.upload;


import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.cocoon.Constants;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.cocoon.servlet.multipart.PartOnDisk;

/**
 * Component to upload file 
 * @author <a href="mailto:n.maisonneuve@hotpop.com">Nicolas Maisonneuve</a>
 * @version 1.0
 */
public class FileUploadManagerImpl
    extends AbstractLogEnabled
    implements FileUploadManager, Configurable, Contextualizable {

  static final String DEFAULT_DIRECTORY = "uploadfolder";
  static final int BUFFER = 4096;

  private Context env_context;
  File uploadFolder;

  public void contextualize(Context context) throws ContextException {
    this.uploadFolder = (File) context.get(Constants.CONTEXT_UPLOAD_DIR);
  }

  public void configure(Configuration conf) throws ConfigurationException {
    Configuration dirconf = conf.getChild(DEFAULT_DIRECTORY,false);
    if (dirconf != null ) {
		String dir=dirconf.getValue("");
		if (!dir.equals("")){
			this.setUploadFolder(dir);
		}
	}
  }

  public String getUploadFolder() {
    return uploadFolder.getAbsolutePath();
  }

  public void setUploadFolder(String uploadfolder) {
    this.uploadFolder = new File(uploadfolder);
    // Create the upload folder if need be
    if (!uploadFolder.exists()) {
      uploadFolder.mkdirs();
    }
  }

  public String upload(Part source) throws Exception {
    //For some reason Part.getFileName() returns the full path instead of just the filename, so we have to substract that ourselves   	
  	String fileName = source.getFileName().substring(source.getFileName().lastIndexOf(File.separator) + 1, source.getFileName().length());
  	System.out.println("this.getUploadFolder = " + this.getUploadFolder());
  	System.out.println("this.uploadFolder.getAbsolutePath = " + this.uploadFolder.getAbsolutePath());

    String destfilename=this.uploadFolder.getAbsolutePath() + File.separator + fileName;
    System.out.println("destfilename = " + destfilename);
    this.upload(source,destfilename);
    return destfilename;
  }

  public void upload(Part source, String destfilename) throws Exception {
    String filename = source.getFileName();
    File destFile = new File(destfilename);

    // Check to see if the object is a PartOnDisk object as this can
    // simply be renamed if it is on the same file system
    if (source instanceof PartOnDisk) {

      getLogger().debug("Renaming " + source.getFileName() + " to " + destFile);
      if ( ( (PartOnDisk) source).getFile().renameTo(destFile)) {
        getLogger().debug("Successfully renamed ");
      }
      // Maybe it's on a different filesystem...
      else if (copyFile(source.getInputStream(), new FileOutputStream(destFile))) {
        getLogger().debug("Successfully copied " + filename + " to " + destFile);
      }
      else {
        getLogger().error("FAILED to rename/copy " + filename + " to " + destFile);
        throw new Exception("FAILED to rename/copy " + filename + " to " + destFile);
      }
    }
    else {
      getLogger().debug("Streaming file to" + destFile);
      if (copyFile(source.getInputStream(), new FileOutputStream(destFile))) {
        getLogger().debug("Successfully copied stream to " + destFile);
      }
      else {
        getLogger().error("FAILED to copy stream to " + destFile);
        throw new Exception("FAILED to copy stream to " + destFile);
      }
    }
  }


  /**
   * Copy Stream
   * @param in
   * @param out
   * @return
   */
  private boolean copyFile(InputStream in, FileOutputStream out) {
    byte[] buffer = new byte[BUFFER];
    int read;
    try {
      read = in.read(buffer);

      while (read > 0) {
        out.write(buffer, 0, read);
        read = in.read(buffer);
      }
      out.close();
    }
    catch (IOException ex) {
      return false;
    }
    return true;
  }

}