/*
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */

/**
 *  listener.cpp: 
 *
 *  This program is one of three programs designed to be used
 *  together. ecl
 *  
 *    create_queues.cpp:
 *
 *      Creates a queue named "message_queue" on a broker, binding the
 *      queue to the "amq.direct" exchange, using the routing key
 *      "routing_key".
 *
 *    direct_producer.cpp 
 *
 *      Publishes to the "amq.direct" exchange, specifying the routing
 *      key "routing_key"
 *
 *    listener.cpp (this program):
 *
 *      Reads  from the "message_queue"  queue on  the broker  using a
 *      message listener.
 */


#include <qpid/client/Connection.h>
#include <qpid/client/Session.h>
#include <qpid/client/Message.h>
#include <qpid/client/MessageListener.h>
#include <qpid/client/SubscriptionManager.h>

#include <unistd.h>
#include <cstdlib>
#include <iostream>

using namespace qpid::client;
using namespace qpid::framing;

      
class Listener : public MessageListener{
  private:
    SubscriptionManager& subscriptions;
  public:
  Listener(SubscriptionManager& subscriptions);
    virtual void received(Message& message);
};

Listener::Listener(SubscriptionManager& subs) : subscriptions(subs)
{
}

void Listener::received(Message& m) {

  std::cout << "holding message " << m.getData() << std::endl;
  sleep(20);
  
  Subscription sub = subscriptions.getSubscription("message_queue");
  
  SequenceSet set;
  set.add(m.getId());

  std::cout << "releasing message" << m.getData() << std::endl;
  sub.release(set);

  sleep(1);
  //sub.grantMessageCredit(1);
}

int main(int argc, char** argv) {
    const char* host = "127.0.0.1";
    int port = 5672;

    Connection connection;

    try {
      connection.open(host, port);
      Session session =  connection.newSession();

  //--------- Main body of program --------------------------------------------

      SubscriptionManager subscriptions(session);
      // Create a listener and subscribe it to the queue named "message_queue"
      // Receive messages until the subscription is cancelled
      // by Listener::received()
      SubscriptionSettings settings;
      settings.completionMode = COMPLETE_ON_ACCEPT;
      settings.acceptMode = ACCEPT_MODE_EXPLICIT;
      settings.autoAck = 0;
      settings.flowControl = FlowControl::messageCredit(1);

      Listener listener(subscriptions);
      subscriptions.subscribe(listener, "message_queue", settings);

      subscriptions.run();


  //---------------------------------------------------------------------------

      connection.close();
      return 0;
    } catch(const std::exception& error) {
        std::cout << error.what() << std::endl;
    }
    return 1;   
}


