/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ru.pronto.wicketextras;

import org.apache.wicket.Component;
import org.apache.wicket.Component.IVisitor;
import org.apache.wicket.MarkupContainer;
import org.apache.wicket.Page;
import org.apache.wicket.application.IComponentOnBeforeRenderListener;

/**
 * Stateless checker. Checks if components with {@link StatelessComponent} annotation are really stateless.
 * @author Marat Radchenko
 * @see StatelessComponent
 */
public class StatelessChecker implements IComponentOnBeforeRenderListener {

  /**
   * Returns <code>true</code> if checker must check given component, <code>false</code> otherwise.
   * @param c component to check.
   * @return <code>true</code> if checker must check given component.
   */
  private static boolean mustCheck(final Component c) {
    final StatelessComponent ann = c.getClass().getAnnotation(StatelessComponent.class);
    return ann != null && ann.enabled();
  }

  /** {@inheritDoc} */
  public void onBeforeRender(final Component c) {
    if (StatelessChecker.mustCheck(c)) {
      final IVisitor<Component> visitor = new Component.IVisitor<Component>() {
        public Object component(final Component comp) {
          if (c instanceof Page && StatelessChecker.mustCheck(comp)) {
            // Do not go deeper, because this component will be checked by checker itself.
            // Actually we could go deeper but that would mean we traverse it twice
            // (for current component and for inspected one).
            // We go deeper for Page because full tree will be inspected during isPageStateless call.
            return IVisitor.CONTINUE_TRAVERSAL_BUT_DONT_GO_DEEPER;
          } else if (!comp.isStateless()) {
            return comp;
          } else {
            return IVisitor.CONTINUE_TRAVERSAL;
          }
        }
      };

      final String msg = "'" + c + "' claims to be stateless but isn't.";
      if (!c.isStateless()) {
        throw new IllegalArgumentException(msg + " Possible reasons: no stateless hint, statefull behaviors");
      }
      if (c instanceof MarkupContainer) {
        // Traverse children
        final Object o = ((MarkupContainer) c).visitChildren(visitor);
        if (o == null) {
          throw new IllegalArgumentException(msg + " Offending component: " + o);
        }
      }

      if (c instanceof Page) {
        final Page p = (Page) c;
        if (!p.isBookmarkable()) {
          throw new IllegalArgumentException(msg + " Only bookmarkable pages can be stateless");
        }
        if (!p.isPageStateless()) {
          throw new IllegalArgumentException(msg + " for unknown reason");
        }
      }
    }
  }
}
