/* $Id: VBoxVirtualPciDevice.cpp $ */
/** @file
 * VBox Virtual Pci Device.
 */

/*
 * Copyright (C) 2009 Sun Microsystems, Inc.
 *
 * This file is part of VirtualBox Open Source Edition (OSE), as
 * available from http://www.virtualbox.org. This file is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License (GPL) as published by the Free Software
 * Foundation, in version 2 as it comes in the "COPYING" file of the
 * VirtualBox OSE distribution. VirtualBox OSE is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY of any kind.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa
 * Clara, CA 95054 USA or visit http://www.sun.com if you need
 * additional information or have any questions.
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/

#include <VBox/pdmdev.h>
#include <VBox/pgm.h>
#include <VBox/version.h>
#include <VBox/err.h>
#include <VBox/log.h>

#include <iprt/assert.h>

#include <cstring>
#include <stdio.h>


#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/shm.h>

#define BETA 1
//#undef DEBUG_LEVEL
//#define DEBUG_LEVEL -1

#if BETA
	#define BETA_STRING ".beta"
#else
	#define BETA_STRING
#endif

#if !defined(DEBUG_LEVEL) && defined(BETA) && BETA
	#define DEBUG_LEVEL 1
#endif

#if !defined(DEBUG_LEVEL) || !defined(BETA) || !BETA
	#undef DEBUG_LEVEL
	#define DEBUG_LEVEL -1	// minus one for no debug, zero for minimum debug, 9 for maximum debug
	#define dPrintf(level, format, ...)
#else
	#define dPrintf(level, format, ...) doPrintf (level, format, ## __VA_ARGS__)

	inline void doPrintf(int const level, char const* const format, ...)
	{
	#if DEBUG_LEVEL >= 0
		if (level <= DEBUG_LEVEL)
		{
			FILE * file = fopen ("/vbox/VirtualPciLog.txt", "a");
			if (file)
			{
				va_list args;
				va_start(args, format);
				vfprintf(file, format, args); fflush(0);
				va_end(args);
				fclose(file);
			}

		}
	#endif
	}
#endif


static int getShmInputSize(int shmSize)
{
	if (shmSize == 0)
	{
		// FIXME replace with VBox config
		char const* const shmSizeStr = 0;// getEnv("IVSHM_shmSize");
		if (shmSizeStr != 0)
		{
			sscanf(shmSizeStr, "%d", &shmSize);
		}
		else
		{
			shmSize = 16; // default to 16 MBytes
		}
	}
	int const maxSize = 1024;
	int const oneMegByte = 1024 * 1024;
	if (shmSize > oneMegByte)
		shmSize /= oneMegByte;
	for (int i = 1; i < maxSize; i *= 2)
		if (i >= shmSize)
			return i * oneMegByte;
	return maxSize * oneMegByte;
}





/*******************************************************************************
*   Structures and Typedefs                                                    *
*******************************************************************************/
/**
 * Device Instance Data.
 */
typedef struct VBOXVIRTUALPCIDEVICE
{
	PCIDevice   pciDevice;
	int shmFileDesc;
	unsigned int shmSize;
	RTGCPHYS32	shmPhyAddress;
	R3PTRTYPE(uint8_t *)shmPtrR3;
	RCPTRTYPE(uint8_t *)shmPtrRC;
    //uint32_t    Whatever;
} VBOXVIRTUALPCIDEVICE;
typedef VBOXVIRTUALPCIDEVICE *PVBOXVIRTUALPCIDEVICE;




//FNPDMDEVCONSTRUCT  pfnConstruct;
///** Destructor instance - optional. */
//FNPDMDEVDESTRUCT   pfnDestruct;

static DECLCALLBACK(int) devVirtualPciDestruct(PPDMDEVINS pDevIns);
static DECLCALLBACK(int) devVirtualPciConstruct(PPDMDEVINS pDevIns, int iInstance, PCFGMNODE pCfgHandle);

static DECLCALLBACK(int) virtualPciR3IORegionMap(PPCIDEVICE pPciDev, /*unsigned*/ int iRegion, RTGCPHYS GCPhysAddress, uint32_t cb, PCIADDRESSSPACE enmType);
/**
 * The device registration structure.
 */
static const PDMDEVREG g_DeviceVirtualPci =
{
    /* u32Version */
    PDM_DEVREG_VERSION,
    /* szDeviceName */
    "virtualPci",

    "VBoxDDGC.gc", /* szRCMod */
    "", //"VBoxDDR0.r0", /* szR0Mod */

    /* pszDescription */
    "VBox Virtual PCI Device.\n",
    /* fFlags */

    PDM_DEVREG_FLAGS_DEFAULT_BITS | PDM_DEVREG_FLAGS_RC,// | PDM_DEVREG_FLAGS_R0,

    //PDM_DEVREG_FLAGS_DEFAULT_BITS  | PDM_DEVREG_FLAGS_GUEST_BITS_32_64,
    /* fClass */
    PDM_DEVREG_CLASS_MISC,
    /* cMaxInstances */
    1,
    /* cbInstance */
    sizeof(VBOXVIRTUALPCIDEVICE),
    /* pfnConstruct */
    devVirtualPciConstruct,
    /* pfnDestruct */
    devVirtualPciDestruct,
    /* pfnRelocate */
    NULL,
    /* pfnIOCtl */
    NULL,
    /* pfnPowerOn */
    NULL,
    /* pfnReset */
    NULL,
    /* pfnSuspend */
    NULL,
    /* pfnResume */
    NULL,
    /* pfnAttach */
    NULL,
    /* pfnDetach */
    NULL,
    /* pfnQueryInterface */
    NULL,
    /* pfnInitComplete */
    NULL,
    /* pfnPowerOff */
    NULL,
    /* pfnSoftReset */
    NULL,
    /* u32VersionEnd */
    PDM_DEVREG_VERSION
};



static int openSharedFile(PVBOXVIRTUALPCIDEVICE pThis)
{
	// attempt to Create, will fail if already exists.
	// On success returns non-negative integer (question does this include 0)
	// Otherwise returns -1 and errno to indicate error.
	mode_t mode = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH;
	char const * const shmName = "/IVSHM"; // FIXME replace with VBox config
	pThis->shmSize = 16 * 1024 * 1024; // the default size if no existing file and no configuration
    pThis->shmFileDesc = shm_open(shmName, O_CREAT | O_RDWR | O_EXCL, mode);
	dPrintf(1, "openSharedFile: %s size=%d shm_open result = %d\n", shmName, pThis->shmSize, pThis->shmFileDesc);

	bool const isNewFile = pThis->shmFileDesc != -1;	// record if shared memory regions was created

	if (pThis->shmFileDesc == -1)
	{
		dPrintf(1, "openSharedFile: 1st Create call shmFileDesc = %d errno=%d %s\n", pThis->shmFileDesc, errno, strerror(errno));
		// FAILED to create NEW --> try to open existing
		pThis->shmFileDesc = shm_open(shmName, O_RDWR, S_IRUSR | S_IWUSR);

		if (pThis->shmFileDesc == -1)
		{
			dPrintf(1, "openSharedFile: 2nd Create call shmFileDesc = %d errno=%d %s\n", pThis->shmFileDesc, errno, strerror(errno));
			// report error and exit.
			// FIXME add better error report
			fprintf(stderr, "\nshm_open FAILED with %s\n", shmName);
			return VERR_GENERAL_FAILURE;
		}
	}
	else
	{
		dPrintf(1, "openSharedFile: ATTEMPT to set mode = %o\n", mode);
		struct stat file_stats;
		int error;
		error = fstat(pThis->shmFileDesc, &file_stats);
		if (error == -1)
		{
			// report error, errno has error
			fprintf(stderr, "fstat FAILED\n");
			return VERR_GENERAL_FAILURE;
		}
		dPrintf(1, "openSharedFile: BEFORE file mode = %o\n", file_stats.st_mode);

		error = fchmod(pThis->shmFileDesc, mode);
		if (error == -1)
			fprintf(stderr, "shmctl errno=%d %s\n", errno, strerror(errno));

		error = fstat(pThis->shmFileDesc, &file_stats);
		if (error == -1)
		{
			// report error, errno has error
			fprintf(stderr, "fstat FAILED\n");
			return VERR_GENERAL_FAILURE;
		}
		dPrintf(1, "openSharedFile: AFTER file mode = %o\n", file_stats.st_mode);
	}

	if (isNewFile)
	{
		// Only called if we created the shared memory region
		// On success return 0, otherwise return -1, errno has error code
		pThis->shmSize = getShmInputSize(pThis->shmSize);
		dPrintf(1, "openSharedFile: Calling ftruncate(shmFileDesc=%d, shmSize=%d\n", pThis->shmFileDesc, pThis->shmSize);
		int error = ftruncate(pThis->shmFileDesc, pThis->shmSize);
		if (error == -1)
		{
			// report error, errno has error
			fprintf(stderr, "ftrancate FAILED\n");
			return VERR_GENERAL_FAILURE;
		}
	}
	else
	{
		// Opened existing Shared memory region, get the size of existing regions
		struct stat file_stats;
		int error = fstat(pThis->shmFileDesc, &file_stats);
		if (error == -1)
		{
			// report error, errno has error
			fprintf(stderr, "fstat FAILED\n");
			return VERR_GENERAL_FAILURE;
		}
		pThis->shmSize = file_stats.st_size;
	}
	return VINF_SUCCESS;
}

static void initPciRegisters(PVBOXVIRTUALPCIDEVICE pThis)
{
    /* The PCI devices configuration. */
    PCIDevSetVendorId(         &pThis->pciDevice,   0x1af4);	/* PCI vendor, same as Cam Macdonell used for KVM, offset==0x00 */
    PCIDevSetDeviceId(         &pThis->pciDevice,   0x1110);	/* PCI device ID, same as Cam Macdonell used for KVM, offset==0x02 */

    PCIDevSetSubSystemVendorId(&pThis->pciDevice,   0x1af4);	/* PCI sub vendor, offset==0x2C */
    PCIDevSetSubSystemId(      &pThis->pciDevice,   0x1110);	/* PCI sub device ID, offset==0x02 */

    PCIDevSetClassBase(        &pThis->pciDevice,   0x05);		/* Memory controller,  offset==0x0B */
    PCIDevSetClassSub(         &pThis->pciDevice,   0x00);		/* Ram controller, offset==0x0A */
    PCIDevSetClassProg(        &pThis->pciDevice,   0x00);		/* Ram controller, offset==0x09 */

    PCIDevSetHeaderType(       &pThis->pciDevice,   0x00);		/* Header type, offset==0x0E */
    PCIDevSetInterruptPin(     &pThis->pciDevice,   0x00);

	pThis->pciDevice.config[0x10] = 0x00; /* MMIO Base */
	pThis->pciDevice.config[0x11] = 0x00;
	pThis->pciDevice.config[0x12] = 0x00;
	pThis->pciDevice.config[0x13] = 0x00;

}



static DECLCALLBACK(int) devVirtualPciDestruct(PPDMDEVINS pDevIns)
{
	dPrintf(5, "devVirtualPciDestruct: ENTER\n", 0);
    /*
     * Check the versions here as well since the destructor is *always* called.
     */
    AssertMsgReturn(pDevIns->u32Version            == PDM_DEVINS_VERSION, ("%#x, expected %#x\n", pDevIns->u32Version,            PDM_DEVINS_VERSION), VERR_VERSION_MISMATCH);
    AssertMsgReturn(pDevIns->pDevHlpR3->u32Version == PDM_DEVHLP_VERSION, ("%#x, expected %#x\n", pDevIns->pDevHlpR3->u32Version, PDM_DEVHLP_VERSION), VERR_VERSION_MISMATCH);

    return VINF_SUCCESS;
}

static DECLCALLBACK(int) devVirtualPciConstruct(PPDMDEVINS pDevIns, int iInstance, PCFGMNODE pCfgHandle)
{
	int         rc;
	int const barNumber = 0;
	dPrintf(5, "devVirtualPciConstruct: ENTER\n", 0);
    /*
     * Check that the device instance and device helper structures are compatible.
     */
    AssertLogRelMsgReturn(pDevIns->u32Version            == PDM_DEVINS_VERSION, ("%#x, expected %#x\n", pDevIns->u32Version,            PDM_DEVINS_VERSION), VERR_VERSION_MISMATCH);
    AssertLogRelMsgReturn(pDevIns->pDevHlpR3->u32Version == PDM_DEVHLP_VERSION, ("%#x, expected %#x\n", pDevIns->pDevHlpR3->u32Version, PDM_DEVHLP_VERSION), VERR_VERSION_MISMATCH);


    /*
     * Initialise the instance data so that the destructor won't mess up.
     */
    PVBOXVIRTUALPCIDEVICE pThis = PDMINS_2_DATA(pDevIns, PVBOXVIRTUALPCIDEVICE);

    /* initialise the PCI registers */
    initPciRegisters(pThis);

    /*
	  * Register our PCI device.
	  */
	rc = PDMDevHlpPCIRegister(pDevIns, &pThis->pciDevice);
	if (RT_FAILURE(rc))
		return rc;

	dPrintf(1, "PCI Register DONE\n", 0);

	/* open the shared file and get size */
	rc = openSharedFile(pThis);
	if (RT_FAILURE(rc))
		return rc;
	dPrintf(1, "size = %d\n", pThis->shmSize);
	// We now have the size of the shared region, either as size of existing file, as configured or the default of 16MBytes

	dPrintf(1, "calling  PDMDevHlpMMIO2Register size = 0x%x\n", pThis->shmSize);
	rc = PDMDevHlpMMIO2Register(pDevIns, barNumber /* iRegion */, pThis->shmSize, 0, (void **)&pThis->shmPtrR3, "virtualPci");
	 AssertLogRelMsgRCReturn(rc, ("PDMDevHlpMMIO2Register(%#x,) -> %Rrc\n", pThis->shmSize, rc), rc);
	dPrintf(1, "Return  PDMDevHlpMMIO2Register size = 0x%x ptrR3=%p\n", pThis->shmSize, pThis->shmPtrR3);

    RTRCPTR pRCMapping = 0;
    rc = PDMDevHlpMMHyperMapMMIO2(pDevIns, barNumber /* iRegion */, 0 /* off */,  pThis->shmSize, "Guest virtualPci", &pRCMapping);
    AssertLogRelMsgRCReturn(rc, ("PDMDevHlpMMHyperMapMMIO2(%#x,) -> %Rrc\n", pThis->shmSize, rc), rc);
    pThis->shmPtrRC = pRCMapping;
    dPrintf(1, "return  PDMDevHlpMMHyperMapMMIO2 size = 0x%x ptrRC=%p\n", pThis->shmSize, pThis->shmPtrRC);

    dPrintf(1, "calling mmap  ptrR3 = %p\n", pThis->shmPtrR3);
	void * pMemory = mmap(pThis->shmPtrR3, pThis->shmSize, PROT_READ | PROT_WRITE, MAP_SHARED | MAP_FIXED, pThis->shmFileDesc, 0);
	if (pMemory == MAP_FAILED)
	{
		// report failed to map file
		fprintf(stderr, "mmap FAILED \n");
		dPrintf(1, "mmap FAILED\n", 0);
		return VERR_GENERAL_FAILURE;
	}
	dPrintf(1, "mmap returned pMemory = %p\n", pMemory);

	unsigned char * p = pThis->shmPtrR3;
	for (int i = 1; i <= 16; ++i)
	{
		dPrintf(1, " %2.2x %2.2x %2.2x %2.2x ", p[0], p[1], p[2], p[3]);
		p += 4;
		if ((i & 3) == 0)
			dPrintf(1, "\n", 0);
	}
	//dPrintf(1, "\n", 0);

//	pThis->shm_ptrR0 = (RTR0PTR)pThis->shmPtrR3; /** @todo #1865 Map parts into R0 or just use PGM access (Mac only). */

//	if (pThis->fGCEnabled)
//	{
//		RTRCPTR pRCMapping = 0;
//		rc = PDMDevHlpMMHyperMapMMIO2(pDevIns, 0 /* iRegion */, 0 /* off */,  VGA_MAPPING_SIZE, "VGA VRam", &pRCMapping);
//		AssertLogRelMsgRCReturn(rc, ("PDMDevHlpMMHyperMapMMIO2(%#x,) -> %Rrc\n", VGA_MAPPING_SIZE, rc), rc);
//		pThis->vram_ptrRC = pRCMapping;
//	}
//
//#if defined(VBOX_WITH_2X_4GB_ADDR_SPACE)
//	if (pThis->fR0Enabled)
//	{
//		RTR0PTR pR0Mapping = 0;
//		rc = PDMDevHlpMMIO2MapKernel(pDevIns, 0 /* iRegion */, 0 /* off */,  VGA_MAPPING_SIZE, "VGA VRam", &pR0Mapping);
//		AssertLogRelMsgRCReturn(rc, ("PDMDevHlpMapMMIO2IntoR0(%#x,) -> %Rrc\n", VGA_MAPPING_SIZE, rc), rc);
//		pThis->vram_ptrR0 = pR0Mapping;
//	}
//#endif

	dPrintf(1, "calling  PDMDevHlpPCIIORegionRegister size = %p\n", pThis->shmSize);
	rc = PDMDevHlpPCIIORegionRegister(pDevIns, barNumber /* iRegion */, pThis->shmSize, PCI_ADDRESS_SPACE_MEM_PREFETCH, virtualPciR3IORegionMap);
	if (RT_FAILURE(rc))
		return rc;

	dPrintf(1, "VBoxVirtualPciDevice devVirtualPciConstruct pre config\n", 0);
    /*
     * Validate and read the configuration.
     */
    if (!CFGMR3AreValuesValid(pCfgHandle,
                              "Whatever1\0"
                              "Whatever2\0"))
        return VERR_PDM_DEVINS_UNKNOWN_CFG_VALUES;


    dPrintf(1, "VBoxVirtualPciDevice devVirtualPciConstruct EXIT\n", 0);
    return VINF_SUCCESS;
}




/* -=-=-=-=-=- Ring 3: PCI Device -=-=-=-=-=- */


///**
// * HC access handler for the LFB.
// *
// * @returns VINF_SUCCESS if the handler have carried out the operation.
// * @returns VINF_PGM_HANDLER_DO_DEFAULT if the caller should carry out the access operation.
// * @param   pVM             VM Handle.
// * @param   GCPhys          The physical address the guest is writing to.
// * @param   pvPhys          The HC mapping of that address.
// * @param   pvBuf           What the guest is reading/writing.
// * @param   cbBuf           How much it's reading/writing.
// * @param   enmAccessType   The access type.
// * @param   pvUser          User argument.
// */
//static DECLCALLBACK(int) virtualPciR3LFBAccessHandler(PVM pVM, RTGCPHYS GCPhys, void *pvPhys, void *pvBuf, size_t cbBuf, PGMACCESSTYPE enmAccessType, void *pvUser)
//{
//	PVBOXVIRTUALPCIDEVICE   pThis = (PVBOXVIRTUALPCIDEVICE)pvUser;
//    int         rc;
//    Assert(pThis);
//    Assert(GCPhys >= pThis->GCPhysVRAM);
//    rc = vgaLFBAccess(pVM, pThis, GCPhys, 0);
//    if (RT_SUCCESS(rc))
//        return VINF_PGM_HANDLER_DO_DEFAULT;
//    AssertMsg(rc <= VINF_SUCCESS, ("rc=%Rrc\n", rc));
//    return rc;
//}


/**
 * Callback function for unmapping and/or mapping the VirtualPci MMIO2 region (called by the PCI bus).
 *
 * @return VBox status code.
 * @param   pPciDev         Pointer to PCI device. Use pPciDev->pDevIns to get the device instance.
 * @param   iRegion         The region number.
 * @param   GCPhysAddress   Physical address of the region. If iType is PCI_ADDRESS_SPACE_IO, this is an
 *                          I/O port, else it's a physical address.
 *                          This address is *NOT* relative to pci_mem_base like earlier!
 * @param   enmType         One of the PCI_ADDRESS_SPACE_* values.
 */
static DECLCALLBACK(int) virtualPciR3IORegionMap(PPCIDEVICE pPciDev, /*unsigned*/ int iRegion, RTGCPHYS GCPhysAddress, uint32_t cb, PCIADDRESSSPACE enmType)
{
	dPrintf(1, "virtualPciR3IORegionMap: ENTER iRegion=%d\n", iRegion);
    int         rc = VINF_SUCCESS;
    PPDMDEVINS  pDevIns = pPciDev->pDevIns;
    PVBOXVIRTUALPCIDEVICE   pThis = PDMINS_2_DATA(pDevIns, PVBOXVIRTUALPCIDEVICE);
    LogFlow(("virtualPciR3IORegionMap: iRegion=%d GCPhysAddress=%RGp cb=%#x enmType=%d\n", iRegion, GCPhysAddress, cb, enmType));
    //AssertReturn(iRegion == 0 && enmType == PCI_ADDRESS_SPACE_MEM_PREFETCH, VERR_INTERNAL_ERROR);
    dPrintf(1, "virtualPciR3IORegionMap: 1 size0f = %d, phyAdr = %p NIL_RTGCPHYS = %p\n", sizeof(GCPhysAddress), GCPhysAddress, NIL_RTGCPHYS);
    if (GCPhysAddress != NIL_RTGCPHYS)
    {
        /*
         * Mapping the VRAM.
         */
    	dPrintf(1, "virtualPciR3IORegionMap: calling PDMDevHlpMMIO2Map\n", 0);
        rc = PDMDevHlpMMIO2Map(pDevIns, iRegion, GCPhysAddress);
        dPrintf(1, "virtualPciR3IORegionMap: return PDMDevHlpMMIO2Map rc = %d, RT_SUCCESS = %d\n", rc, RT_SUCCESS(rc));
        AssertRC(rc);
        if (RT_SUCCESS(rc))
        {
//            rc = PGMR3HandlerPhysicalRegister(PDMDevHlpGetVM(pDevIns),
//                                              PGMPHYSHANDLERTYPE_PHYSICAL_WRITE,
//                                              GCPhysAddress, GCPhysAddress + (pThis->shmSize - 1),
//                                              virtualPciR3LFBAccessHandler, pThis,
//                                              g_DeviceVirtualPci.szR0Mod, "virtualPciR0LFBAccessHandler", pDevIns->pvInstanceDataR0,
//                                              g_DeviceVirtualPci.szRCMod, "virtualPciGCLFBAccessHandler", pDevIns->pvInstanceDataRC,
//                                              "VirtualPci LFB");
//            AssertRC(rc);
//            if (RT_SUCCESS(rc))
                pThis->shmPhyAddress = GCPhysAddress;

			/* On success return pointer to memory, otherwise returns MAP_FAILED, errno has error code */



        }
        else
        {
        	dPrintf(1, "virtualPciR3IORegionMap: PDMDevHlpMMIO2Map FAILED rc = %d, RT_SUCCESS = %d\n", rc, RT_SUCCESS(rc));
        }
    }
    else
    {
    	dPrintf(1, "virtualPciR3IORegionMap: UN-MAP\n", 0);
        /*
         * Unmapping of the virtualPci in progress.
         * Deregister the access handler so PGM doesn't get upset.
         */
        Assert(pThis->shmPhyAddress);
//        rc = PGMHandlerPhysicalDeregister(PDMDevHlpGetVM(pDevIns), pThis->shmPhyAddress);
//        AssertRC(rc);
//        int r = munmap(pThis->shmPtrR3, pThis->shmSize);
//        dPrintf(1, "munmap rc = %d\n", r);
        pThis->shmPhyAddress = 0;
    }
    dPrintf(1, "virtualPciR3IORegionMap: EXIT phyAdr=%p\n", pThis->shmPhyAddress);
    return rc;
}




/**
 * Register builtin devices.
 *
 * @returns VBox status code.
 * @param   pCallbacks      Pointer to the callback table.
 * @param   u32Version      VBox version number.
 */
extern "C" DECLEXPORT(int) VBoxDevicesRegister(PPDMDEVREGCB pCallbacks, uint32_t u32Version)
{
	int rc;
	dPrintf(1, "VBoxVirtualPciDevice VBoxDevicesRegister ENTER\n", 0);
	LogRel(("VBoxVirtualPciDevice VBoxDevicesRegister ENTER"));
    LogFlow(("VBoxVirtualPciDevice::VBoxDevicesRegister: u32Version=%#x pCallbacks->u32Version=%#x\n", u32Version, pCallbacks->u32Version));
    dPrintf(1, "VBoxVirtualPciDevice VBoxDevicesRegister 1\n", 0);
    AssertLogRelMsgReturn(pCallbacks->u32Version == PDM_DEVREG_CB_VERSION,
                          ("%#x, expected %#x\n", pCallbacks->u32Version, PDM_DEVREG_CB_VERSION),
                          VERR_VERSION_MISMATCH);
    dPrintf(1, "VBoxVirtualPciDevice VBoxDevicesRegister 2\n", 0);
    rc = pCallbacks->pfnRegister(pCallbacks, &g_DeviceVirtualPci);
    if (VBOX_FAILURE(rc))
    	return rc;
    dPrintf(1, "VBoxVirtualPciDevice VBoxDevicesRegister 3\n", 0);
    return VINF_SUCCESS;
}

