/*
 * vdelivermail
 * part of the vpopmail package
 * 
 * Copyright (C) 1999 Inter7 Internet Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <dirent.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "config.h"
#include "vdelivermail.h"
#include "safestring.h"
#include "vpopmail.h"
#include "vauth.h"

#define BOUNCE_ALL "bounce-no-mailbox"
#define DELETE_ALL "delete"

#define TMP_BUF_SIZE 1024
#define MAX_SMALL_BUF 100
#define MAX_BUFF 512

char curdir[256];
char tmp_file[256];
char tmp_buf[TMP_BUF_SIZE];
char msgbuf[32768];

off_t msg_size;
#ifdef HARD_QUOTA
 off_t cur_msg_bytes;
 off_t per_user_limit;
#endif

char *return_path_field = "Return-Path: ";
char *from_field = "From: ";
char *subject_field = "Subject: ";

char TheUser[MAX_BUFF];
char TheDomain[MAX_BUFF];
int CurrentQuotaSizeFd;

#ifdef USE_DELIVER_FILTER
void do_filter_delivery( char *tmp_file );

/* filter vaiables */

char tmp_file_f[256];

#define MAX_LOCATION_LEN	25
#define MAX_SSTRING_LEN		256
#define MAX_TARGET_LEN		256
#define MAX_FBUFF_SIZE		2048
#define MAX_FOLDER_SIZE		512
#define MAX_NAME_LEN		1024

#define FLOC_T			1
#define FLOC_F			2
#define FLOC_S			3
#define FLOC_C			4
#define FLOC_R			5
#define FLOC_AH			6
#define FLOC_AR			7
#define FLOC_B			8
#define FLOC_X			9
#define FLOC_UNDEF		999

typedef struct  _flc {
        char	*loc;
        int	tag;
} flc;

static flc filter_locs[] = {
	{"To",			FLOC_T	},
	{"From",		FLOC_F	},
	{"Subject",		FLOC_S	},
	{"Cc",			FLOC_C	},
	{"Reply-To",		FLOC_R	},
	{"AnyHeaders",		FLOC_AH	},
	{"AnyRecipents",	FLOC_AR	},
	{"Body",		FLOC_B	},
	{"-",			FLOC_X	},
	{0,			0	}
};


typedef struct _fr_action {
	char			action;
	char			target[MAX_TARGET_LEN+1];
	struct _fr_action	*next;
} fr_action;

typedef struct _fr_location {
	char			location[MAX_LOCATION_LEN+1];
	char			sstring[MAX_SSTRING_LEN+1];
	char			operator;
	int			loctag;
	int			hit;
	struct _fr_location	*next;
} fr_location;

typedef struct _fr_header {
	long int		vf;
	long int		vt;
	char			name[MAX_NAME_LEN+1];
	fr_location		*loc_chain;
	fr_action		*act_chain;
	int			hit;
	char			act_operator;
	struct _fr_header	*next;
} fr_header;


fr_header	*fh_chain = (fr_header *)NULL;

int run_filter = 0, hit_filter = 0, hit_body = 0;

char header_text[265];
int position_tag;
#endif

/*
 * Delete the temporary file 
 */
void delete_tmp()
{
	char message[1024];

	if (unlink(tmp_file) != 0) {
		switch (errno) {
			case EACCES:
				scopy (message,"EACCES: permission denied",sizeof(message));
				break;
			case EPERM:
				scopy (message,"EPERM: permission denied",sizeof(message));
				break;
			case ENOENT:
				scopy (message,"ENOENT: path doesn't exist",sizeof(message));
				break;
			case ENOMEM:
				scopy (message,"ENOMEM: out of kernel memory",sizeof(message));
				break;
			case EROFS:
				scopy (message,"EROFS: filesystem read-only",sizeof(message));
				break;
			default: sprintf (message,"Other code: %d",errno);
		}
		puts ("Yikes! Could create but can't delete temporary file!!");
		puts (message);
	}
}


#ifdef USE_DELIVER_FILTER
/*
 * Delete the temporary file used by filter_copy
 */
void delete_tmp_f()
{
	char message[1024];

	if (unlink(tmp_file_f) != 0) {
		switch (errno) {
			case EACCES:
				scopy (message,"EACCES: permission denied",sizeof(message));
				break;
			case EPERM:
				scopy (message,"EPERM: permission denied",sizeof(message));
				break;
			case ENOENT:
				scopy (message,"ENOENT: path doesn't exist",sizeof(message));
				break;
			case ENOMEM:
				scopy (message,"ENOMEM: out of kernel memory",sizeof(message));
				break;
			case EROFS:
				scopy (message,"EROFS: filesystem read-only",sizeof(message));
				break;
			default: sprintf (message,"Other code: %d",errno);
		}
		puts ("Yikes! Could create but can't delete temporary file!! (filter)");
		puts (message);
	}
}
#endif


/****************************************************************************
** Temporary fail (exit 111), & display message */
int failtemp(char *err, ...)
{
	va_list args;

	va_start(args,err);
	vprintf(err,args);
	va_end(args);

	if (*tmp_file)
		delete_tmp();

	exit(111);
}

/****************************************************************************
** Permanant fail (exit 100), & display message */
int failperm(char *err, ...)
{
	va_list args;

	puts ("Reason for failure: ");
	va_start(args,err);
	vprintf(err,args);
	va_end(args);

	if (*tmp_file)
		delete_tmp();

	exit(100);
}

/****************************************************************************
** See if the POP user exists!! */
struct passwd* pop_user_exist(char *user, char *host, char *prefix, char *bounce)
{
 static struct passwd *pw_data;
 char localuser[MAX_SMALL_BUF];
 char user2[MAX_SMALL_BUF];
 int i;
 char *ttmp;
 char *tmpstr;
 char Dir[156];
 int uid;
 int gid;

	if ( user== NULL || host == NULL || prefix == NULL ) {
		printf("gag\n");
		exit(0);
	}

	for(ttmp=user;*ttmp!=0;++ttmp)
		if(isupper((int)*ttmp))
			*ttmp = tolower((int)*ttmp);
	for(ttmp=host;*ttmp!=0;++ttmp)
		if(isupper((int)*ttmp))
			*ttmp = tolower((int)*ttmp);
	for(ttmp=prefix;*ttmp!=0;++ttmp)
		if(isupper((int)*ttmp))
			*ttmp = tolower((int)*ttmp);


	if (*prefix) {
		scopy(localuser,prefix,sizeof(localuser));
		scat(localuser,user,sizeof(localuser));
	} else {
		scopy(localuser,user,sizeof(localuser));
	}

	if ( (pw_data = vauth_getpw(user, host))==NULL) { 
		for(i=0;i<(MAX_SMALL_BUF-1)&&user[i]!=0&&user[i]!='-';++i) {
			user2[i] = user[i];
		}
		user2[i] = 0;
		pw_data = vauth_getpw(user2, host);
	}


	/* if they don't have a dir, make one for them */
	if ( pw_data != NULL && 
		 (pw_data->pw_dir == NULL || slen(pw_data->pw_dir) == 0) ) {

		tmpstr = vget_assign(host, Dir, 156, &uid, &gid);
		chdir(Dir);
		tmpstr = make_user_dir(pw_data->pw_name, host, uid, gid);
		i = slen(tmpstr) + slen(Dir) + 12;
		
		pw_data->pw_dir = calloc(1,i); 
            	if ( slen(tmpstr) > 0 ) {
                	sprintf(pw_data->pw_dir, 
				"%s/%s/%s", Dir, tmpstr, user);
            	} else {
                	sprintf(pw_data->pw_dir, 
				"%s/%s", Dir, user);
            	}
		vauth_setpw( pw_data, host );
	}
	vclose();

	if (!pw_data && !*bounce) {
		failperm ("Unknown local POP user %s (#5.1.1)\n",
			localuser);
	}
	if (pw_data && pw_data->pw_gid & BOUNCE_MAIL ) {
		failperm ("Account locked and mail bounced %s (#5.1.2)\n",
			localuser);
	}
	return pw_data;
}


#ifdef USE_DELIVER_FILTER

/****************************************************************************
** output a message */
void print_message(char *err, ...)
{
	va_list args;

	va_start(args,err);
	vprintf(err,args);
	va_end(args);
}

/*****************************************************************************
** read filter sets */

void read_filter()
{
	FILE *ffp;
	char tmpbuff[MAX_FBUFF_SIZE+1];
	char *fbuff;
	int ec, rf = 0;
	int linenum = 1;
	fr_header *fh_chain_act = (fr_header *)NULL;
	fr_header *fh_chain_tmp = (fr_header *)NULL;
	fr_header *fh_chain_eoc = (fr_header *)NULL;
	flc *flcp;

	int header_flag = 0;
	int location_flag = 1;
	int action_flag = 1;
	fr_location *location_tmp, *location_act = (fr_location *)NULL;
	fr_action   *action_tmp,   *action_act   = (fr_action *)NULL;

	if ( (ffp = fopen( ".vpopfilter", "r" )) != NULL )
	{
		while ( fgets( tmpbuff, MAX_FBUFF_SIZE, ffp ) != NULL )
		{
			linenum++;
			fbuff = tmpbuff + 2;
			switch ( tmpbuff[0] )
			{
			case '@' : /* header found      */
				if ( header_flag == 1 && (location_flag == 0 || action_flag == 0) )
				{
					print_message ("FilterRuelRead: syntax error at line %d (#F.0.3)\n",linenum); 
					rf = 0;
					break;
				}

				if ( (fh_chain_tmp = (fr_header *) malloc( sizeof( fr_header ) )) == NULL )
					failtemp ("FilterRuelRead: Out of memory (#F.0.1)\n"); 

				if ( fh_chain == (fr_header *)NULL )
					fh_chain = fh_chain_tmp;
				else /* moove to end of chain - needed for multible read's and save pos of last record for possible delete*/
				{
					fh_chain_act = fh_chain;
					while( fh_chain_act->next != (fr_header *)NULL )
						fh_chain_act = fh_chain_act->next;
					fh_chain_eoc = fh_chain_act;
				}

				if ( fh_chain_act != (fr_header *)NULL )
					fh_chain_act->next = fh_chain_tmp;
				fh_chain_act = fh_chain_tmp;

				fh_chain_act->next = (fr_header *)NULL;
				fh_chain_act->hit = 0;
				fh_chain_act->act_operator = '*';
			
				ec = sscanf( fbuff, "%ld\t%ld\t%s\n",
				             &fh_chain_act->vf, &fh_chain_act->vt, fh_chain_act->name );
				if ( ec != 3 )
				{
					print_message ("FilterRuelRead: syntax error at line %d (#F.0.3)\n",linenum); 
					rf = 0;
					break;
				}

				location_flag = 0;
				action_flag   = 0;
				fh_chain_act->loc_chain = location_act = (fr_location *)NULL;
				fh_chain_act->act_chain = action_act   = (fr_action *)NULL;
				header_flag++;
				break;
			case '$' : /* search rule found */
				if ( header_flag == 0 || action_flag > 0 )
				{
					print_message ("FilterRuelRead: syntax error at line %d (#F.0.3)\n",linenum); 
					rf = 0;
					break;
				}
				if ( (location_tmp = (fr_location *) malloc( sizeof( fr_location ) )) == NULL )
					failtemp ("FilterRuelRead: Out of memory (#F.0.1)\n"); 

				if ( fh_chain_act->loc_chain == (fr_location *)NULL )
					fh_chain_act->loc_chain = location_tmp;

				if ( location_act != (fr_location *)NULL )
					location_act->next = location_tmp;
				location_act = location_tmp;

				location_tmp->next   = (fr_location *)NULL;
				location_tmp->loctag = 0;

				ec = sscanf( fbuff, "%s\t%s\t%c\n",
				             location_tmp->location, location_tmp->sstring, &location_tmp->operator );
				if ( ec != 3 )
				{
					print_message ("FilterRuelRead: syntax error at line %d (#F.0.3)\n",linenum); 
					rf = 0;
					break;
				}
				for(flcp = filter_locs; flcp->loc != (char *)0; flcp++)
					if (strncmp(flcp->loc,location_tmp->location,strlen(flcp->loc)) == 0 )
						location_tmp->loctag = flcp->tag;

				if ( location_tmp->loctag == 0 )
				{
					print_message ("FilterRuelRead: syntax error at line %d (%s) (#F.0.3)\n",linenum,location_tmp->location); 
					rf = 0;
					break;
				}
				switch ( location_tmp->operator )
				{
				case '-' :
				case '&' :
				case '|' :
				case '!' :
					break;
				default  :
					print_message ("FilterRuelRead: syntax error at line %d (invalid operator %c) (#F.0.3)\n",linenum,location_tmp->operator); 
					rf = 0;
					break;
				}

				action_flag = 0;
				location_flag++;
				break;
			case '%' : /* action found      */
				if ( header_flag == 0 || location_flag == 0 )
				{
					print_message ("FilterRuelRead: syntax error at line %d (#F.0.3)\n",linenum); 
					rf = 0;
					break;
				}
				rf = 1;
				if ( (action_tmp = (fr_action *) malloc( sizeof( fr_action ) )) == NULL )
					failtemp ("FilterRuelRead: Out of memory (#F.0.1)\n"); 

				if ( fh_chain_act->act_chain == (fr_action *)NULL )
					fh_chain_act->act_chain = action_tmp;

				if ( action_act != (fr_action *)NULL )
					action_act->next = action_tmp;
				action_act = action_tmp;

				action_tmp->next = (fr_action *)NULL;

				ec = sscanf( fbuff, "%c\t%s\n",
				             &action_tmp->action, action_tmp->target );
				if ( ec != 2 )
				{
					print_message ("FilterRuelRead: syntax error at line %d (#F.0.3)\n",linenum); 
					rf = 0;
					break;
				}
				switch ( action_tmp->action )
				{
				case 'N' : /* normal local delivery (target - ) */
				case 'T' : /* transfer/copy to target folder */
				case 'F' : /* forward to target e-mail adress */
				case 'R' : /* redirect/send to target e-mail adress */
				case 'B' : /* bounce to target e-mail adress */
				case 'D' : /* delete mail (target - ) */
				case '*' : /* stop filter processing, ignore all rules following this one (target - ) */
					break;
				default  :
					print_message ("FilterRuelRead: syntax error at line %d (invalid action %c) (#F.0.3)\n",linenum,action_tmp->action); 
					rf = 0;
					break;
				}
				action_flag++;
				break;
			default  : /* skip the rest */
				continue;
			}
		}
		fclose( ffp );
		if ( rf == 0 && fh_chain_eoc != (fr_header *)NULL ) /* syntax error in act filter set - remoove it. */
		{
			fh_chain_act = fh_chain_eoc;
			while( fh_chain_act != (fr_header *)NULL )
			{
				/* delete locations */
				while ( fh_chain_act->loc_chain != (fr_location *)NULL )
				{
					location_tmp = fh_chain_act->loc_chain;
					fh_chain_act->loc_chain = fh_chain_act->loc_chain->next;
					free( location_tmp );
				}
				/* delete actions */
				while ( fh_chain_act->act_chain != (fr_action *)NULL )
				{
					action_tmp = fh_chain_act->act_chain;
					fh_chain_act->act_chain = fh_chain_act->act_chain->next;
					free( action_tmp );
				}
				fh_chain_tmp = fh_chain_act;
				fh_chain_act = fh_chain_act->next;
				free( fh_chain_tmp );
			}

			fh_chain_eoc->next = (fr_header *)NULL;
		}
		run_filter = ( rf == 1 || run_filter == 1) ? 1 : 0;
	}
}

/*****************************************************************************
** apply filter checks if one line matches filter set */

void apply_filter( char *buffer )
{
	fr_header *fh_chain_act = fh_chain;
	fr_location *location_act;
	char *ptr = buffer;
	flc *flcp;
	int len = 0;
	long int now;



	if ( hit_body == 0 && *buffer == '\n' )
		hit_body = 1;

	ptr = buffer;

	if ( hit_body == 0 )
	{
		if ( *buffer != '\t' && *buffer != ' ' ) /* if not continued header recalulate postion_tag */
		{
			while( *ptr != ' ' && *ptr != '\t' )
				ptr++;

			len = ( *ptr == ':' ) ? ptr - buffer - 2 : ptr - buffer - 1;

			while( *ptr == ' ' && *ptr == '\t' )
				ptr++;

			strncpy( header_text, buffer, len );
			header_text[len] = '\0';

			position_tag = FLOC_UNDEF;

			for(flcp = filter_locs; flcp->loc != (char *)0; flcp++)
				if (strcmp(flcp->loc,header_text) == 0 )
				{
					position_tag = flcp->tag;
					break;
				}
		}
	} else
		position_tag = FLOC_B;

	while ( *buffer != '\n' && fh_chain_act != (fr_header *)NULL )
	{
		/* is the filter valid ?? */
		if ( fh_chain_act->vf != 0 && fh_chain_act->vt != 0 )
		{
			now = time( (time_t *) NULL );
			if ( now < fh_chain_act->vf || now > fh_chain_act->vt  )
			{
				fh_chain_act = fh_chain_act->next;
				continue;
			}
		}
		location_act = fh_chain_act->loc_chain;

		while ( location_act != (fr_location *)NULL )
		{
			if ( location_act->loctag == FLOC_AH && position_tag == FLOC_B )
			{
				location_act = location_act->next;
				continue;
			} else
			if (location_act->loctag == FLOC_AR && (position_tag != FLOC_T && position_tag != FLOC_C) )
			{
				location_act = location_act->next;
				continue;
			} else
			if ( location_act->loctag != position_tag && location_act->loctag != FLOC_X  &&
			     location_act->loctag != FLOC_AH      && location_act->loctag != FLOC_AR )
			{
				location_act = location_act->next;
				continue;
			}

			switch ( location_act->loctag )
			{
			case FLOC_T :
			case FLOC_F :
			case FLOC_S :
			case FLOC_C :
			case FLOC_R :
			case FLOC_AH :
				if ( strstr(ptr, location_act->sstring) != (char *) NULL)
					location_act->hit = 1;
				break;
			case FLOC_AR :
				if ( position_tag == FLOC_T || position_tag == FLOC_C )
					if ( strstr(ptr, location_act->sstring) != (char *) NULL)
						location_act->hit = 1;
				break;
			case FLOC_B :
				if ( strstr(buffer, location_act->sstring) != (char *) NULL)
					location_act->hit = 1;
				break;
			case FLOC_X :
				location_act->hit = 1;
				break;
			default :
				break;
			}

			switch ( fh_chain_act->act_operator )
			{
				case '*' : /* first location for this rule */
					fh_chain_act->hit = location_act->hit;
					break;
				case '&' : /* and */
					fh_chain_act->hit &= location_act->hit;
					break;
				case '|' : /* or */
					fh_chain_act->hit |= location_act->hit;
					break;
				case '^' : /* or */
					fh_chain_act->hit ^= location_act->hit;
					break;
				case '!' : /* not */
					fh_chain_act->hit &= (!location_act->hit);
					break;
				case '-' : /* last location for this rule not possible per def */
				default  : /* not possible per def */
					break;
			}
			
			fh_chain_act->act_operator = location_act->operator;
			location_act = location_act->next;
		}

		hit_filter |= fh_chain_act->hit;

		fh_chain_act = fh_chain_act->next;
	}
}



/*****************************************************************************
** copy mail accoring to filter hit */

void copy_mail( char *mailfilename, char *target_folder )
{
 struct stat statdata;
 char mailname[256];
 char hostname[128];
 time_t tm;
 int pid,i;
 int mailfile;
 size_t bytes;
 off_t msg_size;
 int inputfile;
#ifdef HARD_QUOTA
 FILE *fs;
 off_t cur_msg_bytes;
 off_t per_user_limit;


	if ( pw_data!=NULL && 
	     strncmp(pw_data->pw_shell, "NO",2)!=0 ) {
         int i;
		per_user_limit = atol(pw_data->pw_shell);
		for(i=0;pw_data->pw_shell[i]!=0;++i){
			if ( pw_data->pw_shell[i] == 'k' || 
			     pw_data->pw_shell[i] == 'K' ) {
				per_user_limit = per_user_limit * 1000;
				break;
			}
			if ( pw_data->pw_shell[i] == 'm' || 
			     pw_data->pw_shell[i] == 'M' ) {
				per_user_limit = per_user_limit * 1000000;
				break;
			}
		}
	} else {
		per_user_limit = HARD_QUOTA; 
	}
	cur_msg_bytes = check_quota();
#endif

	gethostname(hostname,sizeof(hostname));
	pid=getpid();
	for (i=0; (i < 3) && (i > -1); i++) {
		time (&tm);
		sprintf(tmp_file_f,"%s/tmp/%lu.%d.%s",target_folder,tm,pid,hostname);
		if (stat(tmp_file_f,&statdata) == -1) {
			if (errno == ENOENT) {
				i=-2;	/* Not -1! Breaks program */
			}
		}
		if (i > -1) {
			sleep(2);
		}
	}
	if (i > 0)
		failtemp ("Unable to stat maildir (#4.3.1)\n");
	
	if ((mailfile = creat(tmp_file_f,S_IREAD | S_IWRITE)) == -1) {
		failtemp ("Can't create tempfile (#4.3.4)\n");
	}
	if ((inputfile = open(mailfilename,S_IREAD)) == -1) {
		failtemp ("Can't read tempfile (#4.3.4b)\n");
	}
	msg_size = 0;
	bytes=read(inputfile,msgbuf,sizeof(msgbuf));
	while (bytes > 0) {
		msg_size += bytes;
		if (write(mailfile,msgbuf,bytes) != bytes) {
			delete_tmp();
			failtemp ("Failed to write to tmp/ (#4.3.3)\n");
		}
		bytes=read(inputfile,msgbuf,sizeof(msgbuf));
	}

	/* Name the new file with Maildir++ extension */
	sprintf(mailname,"%s/new/%lu.%d.%s,S=%d",target_folder,tm,pid,hostname,(int)msg_size);

	if (bytes < 0) {
		delete_tmp_f();
		failtemp("Error occoured reading message (#4.3.4)\n");
	}
	if (fsync(mailfile) == -1) {
		delete_tmp_f();
		failtemp("Unable to sync file (#4.3.5)\n");
	}
	if (close(mailfile) == -1) {
		delete_tmp_f();
		failtemp("Unable to close() tmp file (#4.3.6)\n");
	}

	if (link(tmp_file_f,mailname) == -1) {
		delete_tmp_f();
		failtemp("Unable to link tmp to new (#4.3.7)\n");
	}

	delete_tmp_f();

#ifdef HARD_QUOTA
	if ( pw_data != NULL && strncmp(pw_data->pw_shell, "NO",2)!=0 &&
	     msg_size > 1000 && cur_msg_bytes + msg_size > per_user_limit ) {

		sprintf(tmp_file, "%s/.over-quota.msg", curdir);
		if ( (fs=fopen(tmp_file, "r"))==NULL){
			if ( strstr(curdir, "/users/") == NULL ) {
				sprintf(tmp_file, "%s/domains/.over-quota.msg", VPOPMAILDIR);
				fs=fopen(tmp_file, "r");
			}
		}
		if ( fs == NULL ) {
			printf("User is over quota email returned\n");
		} else {
			while( fgets( tmp_file, 256, fs ) != NULL ) {
				fputs(tmp_file, stdout);
			}
			fclose(fs);
		}
		unlink(mailname);
		exit(100);

		/*bounce_it_back( mailname );*/
	}
#endif
}



/*****************************************************************************
** do the actions defined in the filter chains */

void do_filter_delivery( char *tmp_file )
{
	fr_header *fh_chain_act;
	fr_action *action_act;
	int stopit = 0;

	fh_chain_act = fh_chain;

	while ( fh_chain_act != (fr_header *)NULL )
	{
		if ( fh_chain_act->hit == 1 ) /* rule is hit apply actions */
		{
			action_act = fh_chain_act->act_chain;
			while ( action_act != (fr_action *)NULL )
			{
				switch ( action_act->action )
				{
				case 'N' : /* normal local delivery (target - ) */
					copy_mail( tmp_file, "." );
					printf( "delivered by filterrule\n" );
					break;
				case 'T' : /* transfer/copy to target folder */
					copy_mail( tmp_file, action_act->target );
					printf( "mooved by filterrule\n" );
					break;
				case 'F' : /* forward to target e-mail adress */
				case 'R' : /* redirect/send to target e-mail adress */
				case 'B' : /* bounce to target e-mail adress */
					if ( is_email_addr( action_act->target ) == 1 )
					{
						sprintf(tmp_buf, "/bin/cat %s | %s/bin/qmail-inject %s", 
							tmp_file, QMAILDIR, action_act->target ); 
						system(tmp_buf);
						printf( "forwarded by filterrule\n" );
					}
					break;
				case 'D' : /* delete mail (target - ) */
					printf( "deleted by filterrule\n" );
					break;
				case '*' : /* stop filter processing, ignore all rules following this one (target - ) */
					stopit = 1;
					break;
				default  : /* impossible per def */
					break;
				}
				if ( stopit == 1 )
					break;
				action_act = action_act->next;
			}
		}
		if ( stopit == 1 )
			break;
		fh_chain_act = fh_chain_act->next;
	}
}

/*****************************************************************************
** prit filter chain for debugging */

void print_filter_chain()
{
	fr_header *fh_chain_act;
	fr_location *location_act;
	fr_action *action_act;

	fh_chain_act = fh_chain;
	printf("---------------------------------------------------------------------------\n");
	printf("FilterChain:\n");
	printf("---------------------------------------------------------------------------\n");
	while ( fh_chain_act != (fr_header *)NULL )
	{
		printf( "RULE - vf:%ld\tvt:%ld\tname:%s -> %d\n", 
		fh_chain_act->vf,
		fh_chain_act->vt,
		fh_chain_act->name,
		fh_chain_act->hit );

		location_act = fh_chain_act->loc_chain;
		while ( location_act != (fr_location *)NULL )
		{
			printf( "LOCATION: loc:%s(%d)\tss:%s\top:%c -> %d\n",
			location_act->location,
			location_act->loctag,
			location_act->sstring,
			location_act->operator,
			location_act->hit );
			location_act = location_act->next;
		}

		action_act = fh_chain_act->act_chain;
		while ( action_act != (fr_action *)NULL )
		{
			printf( "ACTION: ac:%c\ttarget:%s\n",
			action_act->action,
			action_act->target );
			action_act = action_act->next;
		}

		fh_chain_act = fh_chain_act->next;
	}
	printf("---------------------------------------------------------------------------\n");
}
#endif

/*****************************************************************************
** Deliver mail to Maildir in directory 'deliverto' **
** Follows procedure outlined in maildir(5).        */
void deliver_mail(char *deliverto, struct passwd *pw_data )
{
 struct stat statdata;
 char mailname[256];
 char hostname[128];
 time_t tm;
 int pid,i;
 int mailfile;
 size_t bytes;
#ifdef HARD_QUOTA
 FILE *fs;
#endif

	getcwd(curdir,256);

#ifdef USE_DELIVER_FILTER
	run_filter = 0;

	if (chdir("..") == -1)
		failtemp ("Could not chdir to vpops root dir (#4.2.2a) %s\n");

	/* we should be in vpop's root directory by now so we can read sytem level filter rules */
	read_filter();

	if (chdir(curdir) == -1)
		failtemp ("Could not chdir to %s (#4.2.2b) %s\n", curdir);

	/* we should be in the domain's directory by now so we can read domain level filter rules */
	read_filter();
#endif

	if (chdir(deliverto) == -1) {
		if ( vmake_maildir(deliverto, pw_data->pw_uid, pw_data->pw_gid)== -1) {
			failtemp ("Could not make user dirs (#4.2.2) %s\n");
		}
	}

#ifdef USE_DELIVER_FILTER
	/* we shold be in the users directory so we can read user level filter rules */
	read_filter();

#endif

	if (chdir("Maildir") == -1) {
		getcwd(mailname,256);
		failtemp ("Can't change to Maildir (#4.2.1) %s\n", mailname); 
	}

#ifdef HARD_QUOTA
	if ( pw_data!=NULL && 
	     sstrncmp(pw_data->pw_shell, "NO",2)!=0 ) {
         int i;
		per_user_limit = atol(pw_data->pw_shell);
		for(i=0;pw_data->pw_shell[i]!=0;++i){
			if ( pw_data->pw_shell[i] == 'k' || 
			     pw_data->pw_shell[i] == 'K' ) {
				per_user_limit = per_user_limit * 1000;
				break;
			}
			if ( pw_data->pw_shell[i] == 'm' || 
			     pw_data->pw_shell[i] == 'M' ) {
				per_user_limit = per_user_limit * 1000000;
				break;
			}
		}
	} else {
		per_user_limit = HARD_QUOTA; 
	}
	cur_msg_bytes = check_quota();
#endif

	gethostname(hostname,sizeof(hostname));
	pid=getpid();
	for (i=0; (i < 3) && (i > -1); i++) {
		time (&tm);
		sprintf(tmp_file,"tmp/%lu.%d.%s",tm,pid,hostname);
		if (stat(tmp_file,&statdata) == -1) {
			if (errno == ENOENT) {
				i=-2;	/* Not -1! Breaks program */
			}
		}
		if (i > -1) {
			sleep(2);
		}
	}
	if (i > 0)
		failtemp ("Unable to stat maildir (#4.3.1)\n");
	
	if ((mailfile = creat(tmp_file,S_IREAD | S_IWRITE)) == -1) {
		failtemp ("Can't create tempfile (#4.3.4)\n");
	}

	sprintf(msgbuf, "%sDelivered-To: %s@%s\n", 
		getenv("RPLINE"), TheUser, TheDomain);

	msg_size = slen(msgbuf);
	if (write(mailfile, msgbuf, msg_size) != msg_size) {
		delete_tmp();
		failtemp ("Failed to write RP & DT (#4.3.2)\n");
	}

#ifdef USE_DELIVER_FILTER
	while (fgets(msgbuf,sizeof(msgbuf),stdin) != NULL) {
		/* pipe lines throu filter if any filter rules have been read */
		if ( run_filter == 1 )
			apply_filter( msgbuf );
		bytes = strlen(msgbuf);
		msg_size += bytes;
                if (write(mailfile,msgbuf,bytes) != bytes) {
                        delete_tmp();
                        failtemp ("Failed to write to tmp/ (#4.3.3)\n");
                }
	}
#else
	bytes=read(0,msgbuf,sizeof(msgbuf));
	while (bytes > 0) {
		msg_size += bytes;
		if (write(mailfile,msgbuf,bytes) != bytes) {
			delete_tmp();
			failtemp ("Failed to write to tmp/ (#4.3.3)\n");
		}
		bytes=read(0,msgbuf,sizeof(msgbuf));
	}
#endif

	/* Name the new file with Maildir++ extension */
	sprintf(mailname,"new/%lu.%d.%s,S=%d",tm,pid,hostname,(int)msg_size);

	if (bytes < 0) {
		delete_tmp();
		failtemp("Error occoured reading message (#4.3.4)\n");
	}
	if (close(mailfile) == -1) {
		delete_tmp();
		failtemp("Unable to close() tmp file (#4.3.6)\n");
	}
#ifdef USE_DELIVER_FILTER

	/* do filter delivery if any of the filter rules was hit */ 
	if ( hit_filter == 1 ) {
		do_filter_delivery( tmp_file );
	} else /* link to new */
	{
		if (link(tmp_file,mailname) == -1) {
			delete_tmp();
			failtemp("Unable to link tmp to new (#4.3.7)\n");
		}
	}
#else
	if (link(tmp_file,mailname) == -1) {
		delete_tmp();
		failtemp("Unable to link tmp to new (#4.3.7)\n");
	}
#endif
	delete_tmp();

#ifdef HARD_QUOTA
	if ( pw_data != NULL && sstrncmp(pw_data->pw_shell, "NO",2)!=0 &&
	     msg_size > 1000 && cur_msg_bytes + msg_size > per_user_limit ) {

		cur_msg_bytes = recalc_quota(".");
	    if ( cur_msg_bytes + msg_size > per_user_limit ) {
			sprintf(tmp_file, "%s/domains/.over-quota.msg",VPOPMAILDIR);
			fs=fopen(tmp_file, "r");

			if ( fs == NULL ) {
				printf("User is over quota email returned\n");
			} else {
				while( fgets( tmp_file, 256, fs ) != NULL ) {
					fputs(tmp_file, stdout);
				}
				fclose(fs);
			}
			unlink(mailname);

			/* exit 100 will cause the email to be bounced back to sender */
			printf("bounce\n");
			exit(100);

		}
	} else {
		update_quota(cur_msg_bytes + msg_size);
	}
#endif


}

/*****************************************************************************
** The main bit :) If it gets to the end of here, delivery was successful   */
int main(int argc, char *argv[])
{
 static struct passwd *pw_data;
 char *deliverto;
 char bounce[1024];
 char prefix[1024];
 char *tmpstr;

	if (argc > 3) {
		failtemp ("Syntax: %s [prefix [bounceable_mail]]\n",argv[0]);
	}

	*bounce = 0;
	*prefix = 0;
	*tmp_file = 0;
	if (argc > 1) {
		scopy(prefix,argv[1],sizeof(prefix));
		if (argc == 3) {
			scopy (bounce,argv[2],sizeof(prefix));
		}
	}

	scopy(TheUser, getenv("EXT"), MAX_BUFF);
	scopy(TheDomain, getenv("HOST"), MAX_BUFF);
	lowerit(TheUser);
	lowerit(TheDomain);
	vget_real_domain(TheDomain,MAX_BUFF);

	pw_data=pop_user_exist(TheUser,TheDomain,prefix,bounce);

	if (!pw_data) {
		if ( is_delete(bounce) ) {
			exit(0);
		}
		if ( !is_bounce(bounce) ) {
			printf ("POP user does not exist, but will deliver to %s\n",bounce);
		}
		deliverto = bounce;
	} else {
		if ( check_forward_deliver(pw_data->pw_dir) == 1 ) {
			exit(0);
		}
        deliverto = pw_data->pw_dir;
	}

	if ( is_email_addr(deliverto) == 1) {
		tmpstr = email_it(deliverto);
		unlink(tmpstr);
	} else if ( is_bounce(deliverto) == 1) {
 		FILE *fs;
		if ( (fs=fopen(".no-user.msg","r")) != NULL ) {
			while( fgets( bounce, 1024, fs ) != NULL ) {
				printf("%s", bounce);
			}
			fclose(fs);
		} else {
			printf(
		   "Sorry, no mailbox here by that name. vpopmail (#5.1.1)\n");
		}
		exit(100);
	} else { 
		deliver_mail(deliverto, pw_data);
	}
	exit(0);
}

int is_bounce(deliverto)
 char *deliverto;
{
	if ( sstrcmp( deliverto, BOUNCE_ALL ) == 0 ) {
		return(1);
	} 
	return(0);
}

int is_delete(deliverto)
 char *deliverto;
{
	if ( sstrcmp( deliverto, DELETE_ALL ) == 0 ) {
		return(1);
	} 
	return(0);
}



#ifdef HARD_QUOTA

/*
 * Assumes the current working directory is user/Maildir
 *
 * We go off to look at cur and tmp dirs
 * 
 * return size of files
 *
 */
off_t check_quota()
{
 off_t mail_size = 0;
 char tmpbuf[100];

 	if ((CurrentQuotaSizeFd=
		open(".current_size", O_CREAT|O_RDWR, S_IWUSR|S_IRUSR))==-1){
		return(mail_size);
	}
	read(CurrentQuotaSizeFd, tmpbuf, 100);
	mail_size = (off_t)atoi(tmpbuf);
	return(mail_size);
}

off_t recalc_quota(char *dir_name)
{
 off_t mail_size = 0;
 char tmpbuf[100];

	mail_size = count_dir(dir_name);
	if ( msg_size > 1000 && 
	     cur_msg_bytes + msg_size > per_user_limit ) mail_size -= msg_size;

	sprintf(tmpbuf, "%d\n", (int)mail_size);
	lseek(CurrentQuotaSizeFd, 0L, SEEK_SET);
	write(CurrentQuotaSizeFd, tmpbuf, slen(tmpbuf));
	close(CurrentQuotaSizeFd);
	return(mail_size);
}

void update_quota(off_t new_size)
{
 char tmpbuf[100];

	sprintf(tmpbuf, "%d\n", (int)new_size);
	lseek(CurrentQuotaSizeFd, 0L, SEEK_SET);
	write(CurrentQuotaSizeFd, tmpbuf, slen(tmpbuf));
	close(CurrentQuotaSizeFd);
}

off_t count_dir(dir_name)
 char *dir_name;
{
 DIR *mydir;
 struct dirent *mydirent;
 struct stat statbuf;
 off_t file_size = 0;
 char *tmpstr;

	if ( dir_name != NULL ) if (chdir(dir_name) == -1) return(0);
	if ( (mydir = opendir(".")) == NULL ) return(0);

	while( (mydirent=readdir(mydir)) != NULL ) {
		if ( sstrcmp( mydirent->d_name, "..") == 0 ) continue;
		if ( sstrcmp( mydirent->d_name, ".") == 0 ) continue;
		if ( (tmpstr=strstr(mydirent->d_name, ",S="))!=NULL) {
			tmpstr += 3;
			file_size += atoi(tmpstr);
		} else if (stat(mydirent->d_name,&statbuf)==0 && 
		           (statbuf.st_mode & S_IFDIR) ) {
			file_size +=  count_dir(mydirent->d_name);
		}
	}
	closedir(mydir);
	if ( dir_name != NULL && sstrcmp(dir_name, ".." )!=0 && 
	                         sstrcmp(dir_name, "." )!=0) {
		chdir("..");
	}
	return(file_size);
}
#endif

int is_email_addr(deliverto)
 char *deliverto;
{
 int i;

	for(i=0;deliverto[i]!=0;++i){
		if ( deliverto[i] == '@' ) return(1);
	}
	return(0);
}

int email_file(char *filename, char *address)
{
 char tmp_buf[256];

	sprintf(tmp_buf, "/bin/cat %s | %s/bin/qmail-inject %s", 
		filename, QMAILDIR, address ); 
	system(tmp_buf);
	return(0);
}

char *email_it(deliverto)
 char *deliverto;
{
 static char tmp_file[256];
 char hostname[128];
 time_t tm;
 int pid;
 int mailfile;
 size_t bytes;

	gethostname(hostname,sizeof(hostname));
	pid=getpid();
	time (&tm);
	sprintf(tmp_file,"/tmp/%lu.%d.%s",tm,pid,hostname);
	if ((mailfile = creat(tmp_file,S_IREAD | S_IWRITE)) == -1) {
		failtemp ("Can't create tempfile (#4.3.8)\n");
	}

	sprintf(msgbuf, "%sDelivered-To: %s@%s\n", 
		getenv("RPLINE"), TheUser, TheDomain);
	if (write(mailfile, msgbuf, slen(msgbuf)) != slen(msgbuf)) {
		delete_tmp();
		failtemp ("Failed to write RP & DT (#4.3.9)\n");
	}

	while((bytes=read(0,msgbuf,sizeof(msgbuf)))>0) {
		if (write(mailfile,msgbuf,bytes) != bytes) {
			delete_tmp();
			failtemp ("Failed to write to tmp/ (#4.3.10)\n");
		}
	}
	if (bytes < 0) {
		delete_tmp();
		failtemp("Error occoured reading message (#4.3.11)\n");
	}
	if (fsync(mailfile) == -1) {
		delete_tmp();
		failtemp("Unable to sync file (#4.3.12)\n");
	}
	if (close(mailfile) == -1) {
		delete_tmp();
		failtemp("Unable to close() tmp file (#4.3.13)\n");
	}

	sprintf(tmp_buf, "/bin/cat %s | %s/bin/qmail-inject %s", 
		tmp_file, QMAILDIR, deliverto ); 
	system(tmp_buf);

	return(tmp_file);
}

int check_forward_deliver(char *dir)
{
 static char email_address[100];
 char tmpbuf[500];
 FILE *fs;
 int i;
 int return_value = 0;
 char *tmpstr = NULL;

	sprintf(tmpbuf, "%s/.qmail", dir);
	fs = fopen(tmpbuf,"r");
	if ( fs == NULL ) {
		return(-1);
	}
    	sprintf(tmpbuf, "%s@%s", TheUser, TheDomain);

	while ( fgets(email_address, 100, fs ) != NULL ) {
		for(i=0;email_address[i]!=0;++i) {
			if (email_address[i] == '\n') email_address[i] = 0;
		}
       		if ( strcmp( email_address, tmpbuf) == 0 ) continue;

		if ( return_value == 0 ) {
			tmpstr = email_it(email_address);
		} else {
			email_file(tmpstr, email_address);
		}
		return_value = 1;
	}
	fclose(fs);
	if ( tmpstr != NULL ) {
		unlink(tmpstr);
	}
	return(return_value);
}

int bounce_it_back( char *filename )
{		
 FILE *fs;
 FILE *fs1;
 char tmpbuf1[300];
 int i,j;
 int first_line = 0;
 int subject_found = 0;
 int from_found = 0;

	if ( (fs = fopen(filename,"r+")) == NULL ) {
		unlink(filename);
		return(-1);
	}
	/* find return path or from variable */

	while ( fgets(tmp_buf, TMP_BUF_SIZE, fs ) != NULL ) {
		for(i=0;return_path_field[i] != 0; ++i){
			if ( tmp_buf[i] != return_path_field[i] ) break;
		}
		if ( return_path_field[i] == 0 ) {
			while(tmp_buf[i] == ' ' || tmp_buf[i] == '<') ++i;
			for(j=0;tmp_buf[i]!='>' && tmp_buf[i]!=0;++i,++j){
				tmp_file[j] = tmp_buf[i];
			}
			tmp_file[j] = 0;
			break;
		} else {
			for(i=0;from_field[i] != 0; ++i){
				if ( tmp_buf[i] != return_path_field[i] ) break;
			}
			if ( from_field[i] == 0 ) {
				while(tmp_buf[i]==' '|| tmp_buf[i] == '<') ++i;
				for(j=0;tmp_buf[i]!='>'&&tmp_buf[i]!=0;++i,++j){
					tmp_file[j] = tmp_buf[i];
				}
				tmp_file[j] = 0;
				break;
			}
		}
	}
	rewind(fs);

	sprintf(tmpbuf1,"%s.bak", filename);
	if ( (fs1 = fopen(tmpbuf1,"w+")) == NULL ) {
		unlink(filename);
		return(-1);
	}

	while ( fgets(tmp_buf, TMP_BUF_SIZE, fs ) != NULL ) {
		if ( subject_found == 0 ) {
			for(i=0;subject_field[i] != 0; ++i){
				if ( tmp_buf[i] != subject_field[i] ) break;
			}
			if ( subject_field[i] == 0 ) {
				subject_found = 1;
				fprintf( fs1, "Subject: User over quota RE: %s",
					&tmp_buf[i]);
				continue;
			}
		} 

		if ( from_found == 0 ) {
			for(i=0;from_field[i] != 0; ++i){
				if ( tmp_buf[i] != from_field[i] ) break;
			}
			if ( from_field[i] == 0 ) {
				from_found = 1;
				fprintf( fs1, "From: postmaster@%s\n",
					 getenv("HOST"));
				continue;
			}
		} 

		if ( first_line == 0 ) {
			for(i=0;tmp_buf[i]!=0;++i) 
				if ( !isspace(tmp_buf[i]) ) break;
			if ( tmp_buf[i] == 0 ) {
				first_line = 1;
				fputs(tmp_buf, fs1);
				fprintf(fs1, 
				"User is over quota, message bounced.\n\n");
				continue;
			}
		} 

		fputs(tmp_buf, fs1);

		/* only return 15 lines of the message */
		if (first_line > 0) {
			++first_line;
			if ( first_line > 15 ) break;
		}
	}
	fclose(fs);
	fclose(fs1);

	sprintf(tmp_buf, "/bin/cat %s | %s/bin/qmail-inject %s", 
		tmpbuf1, QMAILDIR, tmp_file ); 
	system(tmp_buf);

	unlink(filename);
	unlink(tmpbuf1);
	return(0);
}
