#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <time.h>
#include <linux/videodev.h>
#include <stdlib.h>
#include <string.h>      
#include <pthread.h>
#include <errno.h>

/* includes needed for openCV library */
#include <ipl.h>
#include "piclib.h"

#ifdef DEBUG_PICLIB
#include <asm/timex.h>
unsigned int tmp[2];
#define MSG(string, args...) printf("piclib : " string, ##args)
int debug = 1;
#else
#define MSG(string, args...)
int debug = 0;
#endif

static void *video_thread(void *arg);
static int xioctl(int fd,int cmd, void *arg);

typedef struct pic_data {
  pthread_mutex_t mutex;
  pthread_cond_t cond;
  int wont_pic;
  pthread_cond_t got_it;
  unsigned char *frame_ptr;
} pic_data_struct;


static pic_data_struct 
pic_data = { PTHREAD_MUTEX_INITIALIZER, 
	     PTHREAD_COND_INITIALIZER,
	     0,
	     PTHREAD_COND_INITIALIZER, 
	     NULL};

#define RUN 1
#define STOP 0
static struct {
  int fd;
  int run;
  pthread_t thread_ID;
  int height,width;  
  unsigned char *buffer;
  unsigned char *pic_ptr;
  struct video_capability vidcap;
  struct video_buffer vidfbuf;
  struct video_mbuf vidmbuf;
  struct video_picture vidpict;
  struct video_channel vidchan;
  struct video_mmap vidmmap;
} vid_data;

/*
 * basic funktions
 * open, close get_pic and free_pic 
 *----------------------------------*/

/*
 * open a video device
 * input  : a initialiced pic_struct and the name of the device
 * output : if succes 0 is returned
 *          if error 
 *              -EBUSY if open failed.
 *              error masasge from a ioctl call failed.
 *------------------------------------------------------------*/            

int open_video(IplImage *ipl_ptr,int EvOdd, char *dev_name) {
  int fd,res=0;
  
  vid_data.run = RUN;
  
  /*
   * open device 
   */
  vid_data.fd = open(dev_name,O_RDWR);
  MSG(" fd %d\n",vid_data.fd);
  fd = vid_data.fd;
  if(fd < 0) {
    vid_data.run = STOP;
    perror("open");
    MSG("error in open\n");
    return -EBUSY;
  }
  /* insert data for picture in vid_data */
  switch(EvOdd) {
  case EVEN:
    vid_data.height = 2*ipl_ptr->height;
    vid_data.width  = ipl_ptr->width;
    break;
  case ODD:
    vid_data.height = 2*ipl_ptr->height;
    vid_data.width  = ipl_ptr->width;
    break;
  case (EVEN|ODD):
    vid_data.height = ipl_ptr->height;
    vid_data.width  = ipl_ptr->width;
    break;
  default:
    close(fd);
    return -EINVAL;
    break;
  }
  
  /* get and set cap and frame structure */
  res = xioctl(fd,VIDIOCGCAP,&vid_data.vidcap);
  if(res==0) res = xioctl(fd,VIDIOCGFBUF,&vid_data.vidfbuf);  
  if(res==0) res = xioctl(fd,VIDIOCGMBUF,&vid_data.vidmbuf);

  /* set palette and depth. */ 
  if(res==0) res =xioctl(fd,VIDIOCGPICT,&vid_data.vidpict);
  vid_data.vidpict.palette = VIDEO_PALETTE_GREY; 
  vid_data.vidpict.depth = 8; 
  if(res==0) res = xioctl(fd,VIDIOCSPICT,&vid_data.vidpict);

  /* set channel and type */ 
  vid_data.vidchan.channel = 1;
  vid_data.vidchan.type    = VIDEO_TYPE_CAMERA;
  if(res==0) res = xioctl(fd,VIDIOCSCHAN,&vid_data.vidchan);
  if(res==0) res = xioctl(fd,VIDIOCGCHAN,&vid_data.vidchan);

  /* starting up thread */  
  if(res==0) res = pthread_create(&vid_data.thread_ID, NULL, video_thread, (void *) fd);
  return res;
}

/*
 * get a picture funktion
 * input  : a initialiced pic_struct
 * output : 0 if succes
 *          -ETIMEDOUT if request was timed out
 *          -EBUSY if device is busy. (not open).  
 *-----------------------------------------------*/
int get_pic(IplImage *iplImg,int EvOdd) {
  struct timespec timeout;
  int status,i;
  unsigned char *even_ptr,*odd_ptr,*pic_ptr;
  pic_struct *pic_struct_ptr,pic;
  timeout.tv_nsec = 0;
  pic_struct_ptr = &pic;
  pic_struct_ptr->height  = iplImg->height;
  pic_struct_ptr->width   = iplImg->width;
  pic_struct_ptr->pic_ptr = iplImg->imageData;
  
  if(vid_data.run == STOP) 
    return -EBUSY;
  
  /* lock mutex */
  pthread_mutex_lock(&pic_data.mutex); 
 
  /* init structure and ask for pic */  
  pic_data.wont_pic = 1;
  timeout.tv_sec = time (NULL) + 2;
  status = pthread_cond_timedwait(&pic_data.cond,&pic_data.mutex,&timeout);
  if(status != 0) {
    MSG("pthread_cond_timedwait timed out get_pic\n");
    return -ETIMEDOUT;
  }
  /* allocate mem and copy form adr pic_data.frame_ptr */
  even_ptr = pic_data.frame_ptr;
  odd_ptr  = pic_data.frame_ptr + vid_data.width;
  pic_ptr  = pic_struct_ptr->pic_ptr;
  for(i=0;i<vid_data.height;i+=2) {
    switch(EvOdd) {
    case EVEN:
      memcpy(pic_ptr, even_ptr, vid_data.width);
      even_ptr += 2*vid_data.width;
      pic_ptr += vid_data.width;
      break;
    case ODD:
      memcpy(pic_ptr, odd_ptr, vid_data.width);
      odd_ptr += 2*vid_data.width;
      pic_ptr += vid_data.width;
      break;
    case (EVEN|ODD):
      memcpy(pic_ptr, even_ptr, 2*vid_data.width);
      even_ptr += 2*vid_data.width;
      pic_ptr += 2*vid_data.width;
      break;
    }
  }
  status = pthread_cond_broadcast(&pic_data.got_it);
  pthread_mutex_unlock(&pic_data.mutex);
  return 0;
} 

/*
 * close video
 *----------------*/

void close_video(void) {
  int res;
  vid_data.run = STOP;
  res = pthread_join(vid_data.thread_ID,NULL); 
  close(vid_data.fd);
  munmap(vid_data.buffer,vid_data.vidmbuf.size);
  printf("closed video %d \n",res);
}

/*
 * thread funktion 
 * runnig in loob and grap picture to buffer
 * communicate with get_pic
 *------------------------------*/
void *video_thread(void *arg) {
  int fd =vid_data.fd;
  int res,status;
  unsigned char *frame[2];
#ifdef DEBUG_PICLIB
  double tmp[2]={0,0};
  double val;
#endif
  struct timespec timeout;
  timeout.tv_sec = time (NULL) + 2;
  timeout.tv_nsec = 0;
  
  /* 
   * init structs 
   */
  vid_data.vidmmap.frame = 0;
  vid_data.vidmmap.height = vid_data.height;
  vid_data.vidmmap.width = vid_data.width;
  vid_data.vidmmap.format = VIDEO_PALETTE_GREY;
  vid_data.buffer = (unsigned char *) mmap(0,vid_data.vidmbuf.size, 
				    PROT_READ|PROT_WRITE, MAP_SHARED, 
				    fd ,0);
  /* init frame */ 
  frame[0] = vid_data.buffer;
  frame[1] = vid_data.buffer + vid_data.vidmbuf.offsets[1]; 

  /* starting first picture */ 
  xioctl(fd,VIDIOCMCAPTURE, &vid_data.vidmmap);
  vid_data.vidmmap.frame++;
  vid_data.vidmmap.frame &= 0x1;
  while(vid_data.run == RUN) {
    /* starting other image */
    xioctl(fd,VIDIOCMCAPTURE, &vid_data.vidmmap);
    vid_data.vidmmap.frame++;
    vid_data.vidmmap.frame &= 0x1; 
    /* wait read picture */
    xioctl(fd,VIDIOCSYNC,&vid_data.vidmmap.frame);
    
    /* got a picture, and signal */ 
    res = pthread_mutex_lock(&pic_data.mutex); 
    if(res != 0) {
      vid_data.run = STOP;
      MSG("error in mutex");
    }
    pic_data.frame_ptr = frame[ vid_data.vidmmap.frame ];
    res = pthread_cond_broadcast(&pic_data.cond);
#ifdef DEBUG_PICLIB
      tmp[0] = get_cycles()/666.0;
      val = (tmp[0] > tmp[1]) ? (tmp[0] - tmp[1]) : ((tmp[1] - tmp[0]) + 1e6);
      tmp[1] = tmp[0];
      val /= 1e6;
      printf("BROADCAST frame %d time %f\n",vid_data.vidmmap.frame,val);
#endif
    if(res != 0) {
      vid_data.run = STOP;
      MSG("error in cond");
    } 
    if(pic_data.wont_pic==1) {
      timeout.tv_sec = time (NULL) + 2;
      status = pthread_cond_timedwait(&pic_data.got_it,&pic_data.mutex,&timeout);
#ifdef DEBUG_PICLIB
      if(status != 0) {
	MSG("pthread_cond_timedwait timed out in thread %d\n",status);
	switch(status) {
	case ETIMEDOUT:
	  MSG("time out\n");
	  break;
	case EINTR:
	  MSG("signal\n");
	  break;
	case EBUSY:
	  MSG("other thread\n");
	  break;
	}
      }
#endif
      pic_data.wont_pic = 0;
    }
    res = pthread_mutex_unlock(&pic_data.mutex);
  }
  return (void *) 0;
}


/* 
 * smart system calling for debug 
 *-------------------------------*/
static int xioctl(int fd,int cmd, void *arg) {
  int res;
  struct video_capability *a;
  struct video_buffer *b;
  struct video_mbuf *c;
  struct video_picture *d;
  struct video_channel *e;
  struct video_mmap *f;
  int *g;

  res = ioctl(fd,cmd,arg);

  if(res == 0 && !debug) {
    return res;
  }
  else {
    switch(cmd) {
    case VIDIOCGCAP:
      a = arg;
      if(res < 0) perror("VIDIOCGCAP");
      MSG(" name %s, type 0x%x, chan %d, maxwidth %d, maxheight %d, minwidth %d, minheight %d\n",
	     a->name,a->type, a->channels, a->maxwidth, a->maxheight, a->minwidth, a->minheight);
      /* VID_TYPE_TELETEXT 4  Does NOT teletext */
      /* VID_TYPE_CHROMAKEY  16  Does NOT Overlay by chromakey */  
      break;
    case VIDIOCGFBUF:
    case VIDIOCSFBUF:
      b = arg;
      if(res < 0) perror("VIDIOCGFBUF");
      MSG("base %p, height %d, width %d, depth %d, bytes line %d\n",
	     b->base,b->height, b->width, b->depth, b->bytesperline);
      break;
    case VIDIOCGMBUF:
      c = arg;
      if(res < 0) perror("VIDIOCGCAP");
      MSG("size %x, frames %d, offsets 1= %x 2= %x\n",
	     c->size,c->frames,c->offsets[0],c->offsets[1]);
      break;

    case VIDIOCGPICT:
    case VIDIOCSPICT:
      d = arg;
      if(res < 0) perror("VIDIOCGCAP");
      MSG("brightness %x, hue %x, colour %x, contrast %x, whiteness %x, depth %x, palette %x\n",
	     d->brightness, d->hue, d->colour, d->contrast, d->whiteness, d->depth, d->palette);
      break;
    case VIDIOCGCHAN:
    case VIDIOCSCHAN:
      e = arg;
      if(res < 0) perror("VIDIOCGCAP");
      MSG("channel %d, name %s, tuners %d, flags %x, type %x, norm %d\n",
	     e->channel, e->name, e->tuners, e->flags, e->type, e->norm);
      break;
    case VIDIOCMCAPTURE:
      f = arg;
      if(res < 0) perror("VIDIOCMCAPTURE");
      break;
    case VIDIOCSYNC:
      g = arg;
      if(res < 0) perror("VIDIOCSYNC");
      break;
    }
  }
  return res;
}

/*
 * other prictical funktions.
 * funktions added 1/6-01
 *------------------------------*/


/*
 * prints picture in pic_struct format
 * and save picture as <name>.pgm
 *--------------------------------------*/
int pic2file( IplImage *iplImg, char *name) {
  FILE *file_ptr;
  int
    height = iplImg->height,
    width  = iplImg->width;
  char *buf;
  const char extension[] = ".pgm";
  char *tmp_name; 
  
  tmp_name = malloc(strlen(name) + strlen(extension));
  tmp_name = strcpy(tmp_name,name);
  tmp_name = strcat(tmp_name,extension);

  buf = iplImg->imageData; 
  if(buf == NULL)
    return -1;
 
  if( (file_ptr = fopen(tmp_name,"wb")) == NULL)
    return -1;
  
  /* 
   * picture saved in pgm format. see man pgm.
   */
  fprintf(file_ptr,"P5\n%d %d\n255\n",width,height);
  fwrite(buf,1,width*height,file_ptr);
  fclose(file_ptr);
  free(tmp_name);
  return 0;
}           

