#include <errno.h>
#include <fcntl.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <wchar.h>

#include "deps.h"
#include "arabic.h"
#include "arabic.c"

static size_t read_file(char *filename, char *buf, size_t size)
{
	size_t nread = 0;
	int cur = 0;
	int fd = open(filename, O_RDONLY);
	while ((cur = read(fd, buf + nread, size - nread)) > 0) {
		nread += cur;
	}
	if (cur < 0) {
		perror("failed to read");
		exit(1);
	}
	return nread;
}

static size_t write_file(char *filename, char *buf, int size)
{
	size_t nwrote = 0;
	int cur = 0;
	int fd = open(filename, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);
	while (nwrote < size &&
	       (cur = write(fd, buf + nwrote, size - nwrote)) >= 0) {
		nwrote += cur;
	}
	if (cur < 0) {
		perror("failed to write");
		exit(1);
	}
	return nwrote;
}

static size_t convert_to_wide(wchar_t *to, const char *from, size_t size)
{
	size_t ret = mbsrtowcs(to, &from, size, NULL);
	if (ret == -1) {
		perror("failed to decode");
		exit(1);
	}
	return ret;
}

static size_t convert_from_wide(char *to, const wchar_t *from, size_t size)
{
	size_t ret = wcsrtombs(to, &from, size, NULL);
	if (ret == -1) {
		perror("failed to encode");
		exit(1);
	}
	return ret;
}

static wchar_t *next_char(wchar_t *s)
{
	while (*s && utf_iscomposing(*s))
		s++;
	return s;
}

static int shape(wchar_t *dst, int len, wchar_t *src)
{
	wchar_t *limit = src + wcslen(src);
	int prev_c1;
	wchar_t nulls[MB_MAXBYTES] = {0};
	wchar_t *next;
	wchar_t *prev = nulls;

	while (src < limit) {
		if (!ARABIC_CHAR(*src)) {
			*dst++ = *src++;
			prev = nulls;
			continue;
		}
		next = next_char(src + 1);
		prev_c1 = prev != nulls && prev + 1 < src ? *(prev + 1) : 0L;
		*dst++ = arabic_shape(*src, NULL, src + 1, *prev,
			 	     prev_c1, *next);
		prev = src++;
		while (!*src && src < limit)
			src++;
		while (src < next)
			*dst++ = *src++;
	}
	*dst = L'\0';
	return 0;
}

int main(int argc, char *argv[])
{
	char data[MAXFILELEN];
	wchar_t content[MAXFILELEN];
	wchar_t newcontent[MAXFILELEN];
	if (argc < 3) {
		printf("Usage: %s input output\n", argv[0]);
		return 0;
	}
	setlocale(LC_ALL, "");
	read_file(argv[1], data, MAXFILELEN);
	convert_to_wide(content, data, MAXFILELEN);
	shape(newcontent, MAXFILELEN, content);
	convert_from_wide(data, newcontent, MAXFILELEN);
	write_file(argv[2], data, strlen(data));
	return 0;
}
