#!/bin/bash
#
# Author: Olaf Dabrunz
# Year: 2015
#

PRG="${0##*/}"

cleanup_and_exit () {
    local rc=$?
    sudo sh -c 'policy="ondemand"; for gov in /sys/devices/system/cpu/cpu?/cpufreq/scaling_governor ; do echo "$policy" > "$gov" ; cat "$gov" ; done ; cat /sys/devices/system/cpu/cpu?/cpufreq/scaling_cur_freq'
    trap - EXIT SIGINT SIGTERM SIGABRT
    exit $rc
}

trap "cleanup_and_exit" EXIT SIGINT SIGTERM SIGABRT

sudo sh -c 'policy="performance"; for gov in /sys/devices/system/cpu/cpu?/cpufreq/scaling_governor ; do echo "$policy" > "$gov" ; cat "$gov" ; done ; cat /sys/devices/system/cpu/cpu?/cpufreq/scaling_cur_freq'

OUT="vim-main-test-results.out"
LOG="vim-main-test-results.log"
DOTEST="./test-extend-error-mode.sh"


_get_rc()       { echo $(( $? != 0 ? $? : 1 )) ; }
error()         { echo -e "$PRG: error: $*" >&2 ; }
die()           { rc=$(_get_rc) ; error "$1" ; exit ${2:-$rc} ; }


DO_WARMUP_TESTS=1
DO_STRING_VAL_TESTS=1
DO_NUM_VAL_TESTS=1
# -
DO_OVERLAP_TESTS=1
DO_NOCOPY_TESTS=1
DO_SELF_EXTEND_TESTS=1

START_DICT_SIZE=10000000
END_DICT_SIZE=10
#START_DICT_SIZE=1000
#END_DICT_SIZE=1000
#TESTCNT_DICTSIZE_PRODUCT=10000000
TESTCNT_DICTSIZE_PRODUCT=2000000
TESTNUM=11
IFENCE_FACT=1.1


#export VIM_TESTOPTS="testcnt=1000, d_size=10000, d2_size=10000, d2_start=1, d_val_expr='t'.i, d2_val_expr='t'.i, use_same_dict=0, do_copy=1, do_test=1"
export VIM_TESTOPTS="testnum=$TESTNUM, testcnt=200, ifence_fact=$IFENCE_FACT,d_size=10000, d2_size=10000, d2_start=1, d_val_expr='t'.i, d2_val_expr='t'.i, use_same_dict=0, do_copy=1, do_test=1"


print_headline()
{
    printf >>"$OUT" "\n        overlap between dicts:"
    for o in "${overlaps[@]}" ; do printf >>"$OUT" "%5d%%  " "$o" ; done
    printf >>"$OUT" "\n        ----------------------"
    for o in "${overlaps[@]}" ; do printf >>"$OUT" "%8s" "--------" ; done
    printf >>"$OUT" -- "----\n"
}

print_results()
{
    local v

    for v in "$@"
    do
        printf >>"$OUT" $([[ $v =~ ^-?[0-9] ]] && echo "%7.1f%%" || echo "%8s" ) "$v"
    done
}

test_overlaps()
{
    local o

    testcnt=$(( $TESTCNT_DICTSIZE_PRODUCT / $dsize ))
    test $testcnt -lt 1 && testcnt=1

    if [ $dsize -eq 1000 ]
    then
        # These tests run longer and tend to trigger CPU freq throttling.
        # Avoid this by making them run shorter.
#        let testcnt/=10
        :
    fi

    # For tests with at least 1000 items in the dicts, measure the time spent
    # in extend() separately in each test.
    #
    # This prevents larger variations in other parts of each test run from
    # exerting a large influence on the results, esp. since the copy() part for
    # large dictionaries can take much longer than the extend() part, and has a
    # correspondingly large variation that easily can dominate over the
    # relatively short extend() running time.
    measure_separately=$(( $dsize >= 1000 ? 1 : 0 ))

    times=()
    loads=()
    stores=()
    accesses=()

    for overlap in "${overlaps[@]}"
    do
        d2_start=$(bc -l <<<"$dsize - ($dsize / 100 * $overlap) + 1")
        d2_start=${d2_start%%.*}

        if [ $dsize -eq 10000000 -a $overlap -eq 0 ]
        then
            # this test crashes for me
            read time_oh load_oh store_oh access_oh <<<"crash crash crash crash"
        else
            {
                TEST_TITLE="$TEST_TITLE ($overlap%)" \
                $DOTEST "$@" testcnt=$testcnt d2_start=$d2_start \
                    measure_separately=$measure_separately \
                    || die "error executing $DOTEST"
            } |& tee -a "$LOG"
            stty sane
            read time_oh load_oh store_oh access_oh \
                < vim-test-results.out
        fi

        times=("${times[@]}" "$time_oh")
        loads=("${loads[@]}" "$load_oh")
        stores=("${stores[@]}" "$store_oh")
        accesses=("${accesses[@]}" "$access_oh")

        echo "Waiting 2 seconds, press CTRL-C to interrupt..."
        sleep 2 || die "interrupted"
    done

    items=$(printf "%-d items" $dsize)
    spcs=$(( 21 - ${#items} ))
    printf >>"$OUT" "        %s %*.*s" "$items" $spcs $spcs " "

    print_results "${times[@]}"

    printf >>"$OUT" "\n            L1 dcache-loads   "
    print_results "${loads[@]}"

    printf >>"$OUT" "\n            L1 dcache-stores  "
    print_results "${stores[@]}"

    printf >>"$OUT" "\n            L1 dcache-l+s     "
    print_results "${accesses[@]}"

    printf >>"$OUT" "\n\n"
}

main_test_loop()
{
    for (( dsize=$START_DICT_SIZE; dsize >= $END_DICT_SIZE; dsize /= 10 ))
    do
        test_overlaps d_size=$dsize d2_size=$dsize \
            || exit
    done
}

main_test_set()
{
    if [ $DO_OVERLAP_TESTS -eq 1 ]
    then
        export VIM_TESTOPTS="$VIM_TESTOPTS_BASE, use_same_dict=0, do_copy=1, do_test=1"
        overlaps=(100 90 50 10 0)
#        overlaps=(0)
        print_headline
        TEST_TITLE="$TEST_TITLE  overlap tests" main_test_loop
    fi


    if [ $DO_OVERLAP_TESTS -eq 1 -o $DO_SELF_EXTEND_TESTS -eq 1 ]
    then
        cat >>"$OUT" <<EOF


    No priming with copy() (not needed as overlap is 100%):
    -------------------------------------------------------
EOF
    fi


    if [ $DO_NOCOPY_TESTS -eq 1 ]
    then
        export VIM_TESTOPTS="$VIM_TESTOPTS_BASE, use_same_dict=0, do_copy=0, do_test=1"
        overlaps=(100)
        print_headline
        TEST_TITLE="$TEST_TITLE  no-copy tests" main_test_loop
    fi


    if [ $DO_SELF_EXTEND_TESTS -eq 1 ]
    then
        cat >>"$OUT" <<EOF

        extend dict with itself:
        ---------------------------------
EOF

        export VIM_TESTOPTS="$VIM_TESTOPTS_BASE, use_same_dict=1, do_copy=0, do_test=1"
        overlaps=(100)
#        print_headline
        echo >>"$OUT"
        TEST_TITLE="$TEST_TITLE  self-extend tests" main_test_loop
    fi
}


# dry-run once to clear memory, preventing later swapping as much as possible
test $DO_WARMUP_TESTS -eq 1 && TEST_TITLE="warmup tests" $DOTEST

cat >> "$LOG" <<EOF

New main test
-------------

EOF

cat >>"$OUT" <<EOF

Overhead of 'rollback' (all-or-nothing) error checking loop over 'stop'
(exit-on-error) in vim's extend() function:
------------------------------------------------------------------------

EOF

if [ $DO_STRING_VAL_TESTS -eq 1 ]
then
    printf >>"$OUT" "\nUsing strings (eval('\"t\".i')) as dict values:\n"
    VIM_TESTOPTS_BASE="testnum=$TESTNUM, ifence_fact=$IFENCE_FACT, d_val_expr='t'.i, d2_val_expr='t'.i"
    TEST_TITLE="strings as dict values:" main_test_set
fi

if [ $DO_NUM_VAL_TESTS -eq 1 ]
then
    printf >>"$OUT" "\nSame tests with numbers as dict values:\n"
    VIM_TESTOPTS_BASE="testnum=$TESTNUM, ifence_fact=$IFENCE_FACT, d_val_expr=i, d2_val_expr=i"
    TEST_TITLE="numbers as dict values:" main_test_set
fi


cat "$OUT"

