#!/bin/bash
#
# Author: Olaf Dabrunz
# Year: 2015
#

PRG="${0##*/}"

for f in src/vim $(type -p vim)
do
    if [ -x "$f" ]
    then
        VIM="$f"
        break
    fi
done

echo "$PRG: using vim: $VIM"

times=()
vars=()
loads=()
stores=()
accesses=()

VIM_TESTOPTS="$VIM_TESTOPTS $*"

_get_rc()       { echo $(( $? != 0 ? $? : 1 )) ; }
error()         { echo -e "$PRG: error: $*" >&2 ; }
die()           { rc=$(_get_rc) ; error "$1" ; exit ${2:-$rc} ; }

do_test()
{
    local load store avg var title_sfx="$1"
    shift

    TEST_TITLE="$TEST_TITLE: $title_sfx" \
    VIM_TESTOPTS="$VIM_TESTOPTS $*" \
    perf stat -e L1-dcache-loads,L1-dcache-load-misses,L1-dcache-stores \
        -o vim-test-perf.out \
        $VIM >& /dev/tty -u NONE -U NONE --noplugin \
        -S test-extend-error-mode.vim \
        || return

# -c 'breakadd file */test-extend-error-mode.vim' \

    {
        read load ; load="${load//,/}"
        read store ; store="${store//,/}"
    } < <(sed -ne '/L1-dcache-\(loads\|stores\)/{s/[[:space:]]*\([0-9,]*\).*$/\1/;p}' \
            < vim-test-perf.out)

    {
        read avg
        read var
    } < <(sed -ne '/^\(avg\|var\):/{s/^[^[:space:]]*[[:space:]]*//;p}' < vim-test.out)

    echo $load $store $avg $var
}

print_test_output()
{
    cat vim-test.out
    cat vim-test-perf.out
}

# measure overhead of everything but the actual test
results="$(do_test "test overhead" do_oh_test=1)" \
    || die "error executing src/vim"
print_test_output

read toh_load toh_store toh_avg toh_var <<<"$results"

for errmode in stop rollback
do
    results="$(do_test "$errmode" errmode=$errmode)" \
        || die "error executing src/vim"

    print_test_output

    read load store avg var <<<"$results"

    loads=("${loads[@]}" $(bc <<<"$load - $toh_load"))
    stores=("${stores[@]}" $(bc <<<"$store - $toh_store"))
    accesses=("${accesses[@]}" $(bc <<<"$load + $store - $toh_load - $toh_store"))

    # the test program already subtracted overhead from time measurements
#    times=("${times[@]}" $(bc <<<"$avg - $toh_avg"))
#    vars=("${vars[@]}" $(bc <<<"$var - $toh_var"))
    times=("${times[@]}" "$avg")
    vars=("${vars[@]}" "$var")

    echo "Waiting 1 second, press CTRL-C to interrupt..."
    sleep 1 || die "interrupted"
done

compute_overhead()
{
    bc -l <<<"(( $2 / $1 ) - 1) * 100"
}

time_oh=$(compute_overhead "${times[@]}")
load_oh=$(compute_overhead "${loads[@]}")
store_oh=$(compute_overhead "${stores[@]}")
access_oh=$(compute_overhead "${accesses[@]}")

printf > vim-test-results.out "%4.1f %4.1f %4.1f %4.1f\n" \
    $time_oh $load_oh $store_oh $access_oh

echo
printf "Overhead of 'rollback' over 'stop':\n"
printf "            slowdown: %4.1f%%\n" $time_oh
printf "     L1-dcache-loads: %4.1f%%\n" $load_oh
printf "    L1-dcache-stores: %4.1f%%\n" $store_oh
printf "    L1-dcache-access: %4.1f%%\n" $access_oh
echo

