exe 'source' expand("<sfile>:p:h").'/util.vim'

func SetUp()
  set diffopt&
  set diffopt+=context:1
  set foldcolumn=7
  let s:starting_bufnr = bufnr('%')
  new
  call append(1, ['one', 'two', 'three', 'four'])
  syntax enable
  syntax match Comment /two/
  /two/,/three/fold
  runtime plugin/tohtml.vim
endfun

func s:open_diff(close_buf)
  if a:close_buf
    bd!
  endif
  diffthis
  call append(0, repeat(['.'],5))
  call append(line('$'), repeat(['.'],5))
  %y
  vnew
  put
  1d
  %s/three/THREE
  put ='...and a half'
  diffthis
  wincmd p
  set foldcolumn=7
endfun

func Check_css_generated_rules(syn_group, assert_func)
  call function(a:assert_func)(search('\V[data-'.a:syn_group.'-content]::before { content: attr(data-'.a:syn_group.'-content); }', 'w'),
        \ "CSS ::before rule for " . a:syn_group . " content")
  call function(a:assert_func)(search('\V[data-'.a:syn_group.'-content]::before { padding-bottom: 1px; display: inline-block;\[^}]\+}', 'w'),
        \ "CSS ::before rule for " . a:syn_group . " padding")
  call function(a:assert_func)(search('\Vspan[data-'.a:syn_group.'-content]::before { cursor: default; }', 'w'),
        \ "CSS ::before rule for " . a:syn_group . " cursor")
  call function(a:assert_func)(search('\V.'.a:syn_group.' { user-select: none; }', 'w'),
        \ "CSS unselectable rule for " . a:syn_group)
endfun
fun Check_shared_css_input_rules(assert_func)
  call function(a:assert_func)(search('input { border: none; margin: 0; padding: 0; font-family:.*monospace; }', 'w'),
        \ "unselectable input styling")
  for w in range(1,20)
    call function(a:assert_func)(search('input\[size='''.w.'''] { width: '.w.'em; width: '.w.'ch; }', 'w'),
          \ "unselectable input sizing, width ".w)
  endfor
endfun
fun Check_css_input_rules(syn_group, assert_func)
  if a:syn_group ==# 'FoldColumn'
    call function(a:assert_func)(search('input.FoldColumn { cursor: pointer; }', 'w'),
          \ "unselectable FoldColumn cursor style")
    call function(a:assert_func)(search('input.FoldColumn\[value="       "] { cursor: default; }', 'w'),
          \ "unselectable FoldColumn fill cursor style")
  else
    call function(a:assert_func)(search('input.'.a:syn_group.' { cursor: default; }', 'w'),
          \ "unselectable ".a:syn_group." cursor style")
  endif
endfun

func Check_generated_fold_column(assert_func)
  call function(a:assert_func)(search('<a href=''#'' class=''\(toggle-open \)\?FoldColumn[^>]*data-FoldColumn-content=''+''></a>', 'w'),
        \ "uncopyable fold column with generated content: open")
  call function(a:assert_func)(search('<a href=''#'' class=''\(toggle-closed \)\?FoldColumn[^>]*data-FoldColumn-content=''|*- *''></a>', 'w'),
        \ "uncopyable fold column with generated content: close")
  call function(a:assert_func)(search('<a href=''#'' class=''\(toggle-\(open\|closed\) \)\?FoldColumn[^>]*data-FoldColumn-content=''|\+ *''></a>', 'w'),
        \ "uncopyable fold column with generated content: toggle")
  call function(a:assert_func)(search('<\(a\%( href=''#''\)\@=\|span\)\>[^>]* class=[''"]FoldColumn[^>]*data-FoldColumn-content=\([''"]\) \+\2></\1>', 'w'),
        \ "uncopyable fold column with generated content: empty")
  call Check_css_generated_rules('FoldColumn', a:assert_func)
endfun
func Check_combined_fold_column(assert_func)
  call function(a:assert_func)(search('<span class=[''"]\%([-_0-9a-zA-Z]\+ \)*FoldColumn[^>]*><input[^>]*><a href=''#'' class=''FoldColumn[^>]*data-FoldColumn-content=''+''></a>', 'w'),
        \ "uncopyable fold column combined method: open")
  call function(a:assert_func)(search('<span class=[''"]\%([-_0-9a-zA-Z]\+ \)*FoldColumn[^>]*><input[^>]*><a href=''#'' class=''FoldColumn[^>]*data-FoldColumn-content=''|*- *''></a>', 'w'),
        \ "uncopyable fold column combined method: close")
  call function(a:assert_func)(search('<span class=[''"]\%([-_0-9a-zA-Z]\+ \)*FoldColumn[^>]*><input[^>]*><a href=''#'' class=''FoldColumn[^>]*data-FoldColumn-content=''|\+ *''></a>', 'w'),
        \ "uncopyable fold column combined method: toggle")
  call function(a:assert_func)(search('<span class=[''"]\%([-_0-9a-zA-Z]\+ \)*FoldColumn[^>]*><input[^>]*><\(a\%( href=''#''\)\@=\|span\)\>[^>]* class=''FoldColumn[^>]*data-FoldColumn-content='' \+''></\1>', 'w'),
        \ "uncopyable fold column combined method: empty")
  " CSS
  call function(a:assert_func)(search('\Va[data-FoldColumn-content="       "] { cursor: default; }', 'w'),
        \ "uncopyable fold column combined method: column fill CSS")
endfun
func Check_input_fold_column(assert_func)
  call function(a:assert_func)(search('<span class=''\%([-_0-9a-zA-Z]\+ \)*FoldColumn[^>]*><input[^>]*readonly=''readonly''[^>]*return false;[^>]*value=''+''>', 'w'),
        \ "uncopyable fold column using <input> tags: open")
  call function(a:assert_func)(search('<span class=''\%([-_0-9a-zA-Z]\+ \)*FoldColumn[^>]*><input[^>]*readonly=''readonly''[^>]*return false;[^>]*value=''|*- *''>', 'w'),
        \ "uncopyable fold column using <input> tags: close")
  call function(a:assert_func)(search('<span class=''\%([-_0-9a-zA-Z]\+ \)*FoldColumn[^>]*><input[^>]*readonly=''readonly''[^>]*return false;[^>]*value=''|\+ *''>', 'w'),
        \ "uncopyable fold column using <input> tags: toggle")
  call function(a:assert_func)(search('<span class=''\%([-_0-9a-zA-Z]\+ \)*FoldColumn[^>]*><input[^>]*readonly=''readonly''[^>]*return false;[^>]*value='' \+''>', 'w'),
        \ "uncopyable fold column using <input> tags: empty")
  call Check_css_input_rules('FoldColumn', a:assert_func)
endfun

func Check_generated_line_nums(assert_func, check_folded)
  call function(a:assert_func)(search('<span id=" *\%(W\d\+\)\?L\(\d\+\)" class="LineNr" data-LineNr-content=" *\1 "', 'w'),
        \ "uncopyable line numbers with generated content: standard")
  if a:check_folded
    call function(a:assert_func)(search('\%(<span [^>]*class=''\%(closed\|open\)-fold''[^>]*>\|<\(span\|a\) [^>]*class=''[^'']*FoldColumn''[^>]*>\%(\%(</\1>\)\@!.\)*</\1>\)<span class="Folded" data-Folded-content=" *\d\+ "', 'w'),
          \ "uncopyable line numbers with generated content: folded")

    " CSS for folded line numbers
    call function(a:assert_func)(search('\V.FoldColumn + [data-Folded-content]::before, [data-Folded-content]:first-child::before { content: attr(data-Folded-content); }', 'w'),
          \ "CSS ::before rule for folded LineNr content")
    call function(a:assert_func)(search('\V.FoldColumn + [data-Folded-content]::before, [data-Folded-content]:first-child::before { padding-bottom: 1px; display: inline-block;\[^}]\+}', 'w'),
          \ "CSS ::before rule for folded LineNr padding")
    call function(a:assert_func)(search('\V.FoldColumn + span[data-Folded-content]::before, [data-Folded-content]:first-child::before { cursor: default; }', 'w'),
          \ "CSS ::before rule for folded LineNr cursor")
    call function(a:assert_func)(search('\V.FoldColumn + .Folded, .Folded:first-child { user-select: none; }', 'w'),
          \ "CSS unselectable rule for folded LineNr")
  endif
  call Check_css_generated_rules('LineNr', a:assert_func)
endfun
func Check_input_line_nums(assert_func, check_folded)
  call function(a:assert_func)(search('<span id=" *\%(W\d\+\)\?L\(\d\+\)"[^>]*><input[^>]*class="LineNr" value=" *\1"', 'w'),
      \ "uncopyable line numbers with <input> tags: standard")
  if a:check_folded
    call function(a:assert_func)(search('\%(<span [^>]*class=''\%(closed\|open\)-fold''[^>]*>\|<\(span\|a\) [^>]*class=''[^'']*FoldColumn''[^>]*>\%(\%(</\1>\)\@!.\)*</\1>\)\%(<span class="Folded"[^>]*>\)\?<input[^>]*class="Folded" value=" *\d\+"', 'w'),
          \ "uncopyable line numbers with <input> tags: folded")
  endif
  call Check_css_input_rules('LineNr', a:assert_func)
endfun

func Check_generated_foldtext(assert_func)
  call function(a:assert_func)(search('<span [^>]*class=\([''"]\)Folded\1[^>]\+data-Folded-content=\([''"]\)\%( *\d\+ *\2\)\@![^''"]\+\2', 'w'),
        \ "uncopyable fold text with generated content")
  call Check_css_generated_rules('Folded', a:assert_func)
endfun
func Check_input_foldtext(assert_func)
  call function(a:assert_func)(search('<input [^>]*class=\([''"]\)Folded\1[^>]\+value=\([''"]\)\%( *\d\+ *\2\)\@![^''"]\+\2', 'w'),
      \ "uncopyable fold text with <input> tags")
  call Check_css_input_rules('Folded', a:assert_func)
endfun

func Check_generated_diff_filler(assert_func)
  let filler_pattern='<span [^>]*class=\([''"]\)DiffDelete\1[^>]\+data-DiffDelete-content=\([''"]\)-\+\2'
  1
  call function(a:assert_func)(search(filler_pattern, 'Wc'),
        \ "uncopyable diff filler with generated content")
  " ensure max of 1 diff filler line (relies on diff setup to only have one deleted line)
  if a:assert_func==#'assert_true'
    call assert_false(search(filler_pattern, 'W'),
        \ "uncopyable diff filler with generated content: more than one occurance found")
  endif
  call Check_css_generated_rules('DiffDelete', a:assert_func)
endfun
func Check_input_diff_filler(assert_func)
  let filler_pattern='<input [^>]*class=\([''"]\)DiffDelete\1[^>]\+value=\([''"]\)-\+\2'
  1
  call function(a:assert_func)(search(filler_pattern, 'Wc'),
      \ "uncopyable diff filler with <input> tags")
  " ensure max of 1 diff filler line (relies on diff setup to only have one deleted line)
  if a:assert_func==#'assert_true'
    call assert_false(search(filler_pattern, 'W'),
        \ "uncopyable diff filler with <input> tags: more than one occurance found")
  endif
  call Check_css_input_rules('DiffDelete', a:assert_func)
endfun

func s:Check_fallback_supports_rule(assert_func)
  call function(a:assert_func)(search('@supports\s*(\s*content:\s*attr(data-[^)]*)\s*)\_s*{'),
        \ "@supports rule for 'fallback' method")
endfun

" test just the fold column with default values for implementation method for
" uncopyable regions (primary generated content, fallback to using inputs)
func Test_fold_column()
  let g:html_prevent_copy = 'f'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 1
  unlet! g:html_use_input_for_pc

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_fold_column("assert_true")
  call Check_combined_fold_column("assert_true")
  call Check_input_fold_column("assert_true")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 1)
  call Check_input_foldtext("assert_false")

  call s:open_diff(1)
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_fold_column("assert_true")
  call Check_combined_fold_column("assert_true")
  call Check_input_fold_column("assert_true")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 1)
  call Check_input_foldtext("assert_false")
endfun

" test just the fold column, explicitly using generated content for all browsers
" (fold column has some nested content so we need to check all three ways of
" doing it)
func Test_fold_column_generated_only()
  let g:html_prevent_copy = 'f'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 1
  let g:html_use_input_for_pc = "none"

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")
  call Check_generated_fold_column("assert_true")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 1)
  call Check_input_foldtext("assert_false")

  call s:open_diff(1)
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")
  call Check_generated_fold_column("assert_true")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 1)
  call Check_input_foldtext("assert_false")
endfun

" test just the fold column, explicitly using input tags for all browsers (fold
" column has some nested content so we need to check all three ways of doing it)
func Test_fold_column_input_only()
  let g:html_prevent_copy = 'f'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 1
  let g:html_use_input_for_pc = "all"

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_true")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 1)
  call Check_input_foldtext("assert_false")

  call s:open_diff(1)
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_true")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 1)
  call Check_input_foldtext("assert_false")
endfun

" test just the line numbers, explicitly using generated content for all
" browsers
func Test_line_numbers()
  let g:html_prevent_copy = 'n'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 1
  let g:html_use_input_for_pc = "none"

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")
  call Check_generated_line_nums("assert_true", 1)
  call Check_input_line_nums("assert_false", 1)
  
  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")
  call Check_input_foldtext("assert_false")

  call s:open_diff(1)
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")
  call Check_generated_line_nums("assert_true", 1)
  call Check_input_line_nums("assert_false", 1)
  
  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")
  call Check_input_foldtext("assert_false")
endfun

" test just the fold text, explicitly using primary generated content, with
" fallback to using inputs
func Test_fold_text()
  let g:html_prevent_copy = 't'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 1
  let g:html_use_input_for_pc = "fallback"

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_foldtext("assert_true")
  call Check_input_foldtext("assert_true")
  
  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")
  call Check_input_line_nums("assert_false", 1)

  call s:open_diff(1)
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_foldtext("assert_true")
  call Check_input_foldtext("assert_true")
  
  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")
  call Check_input_line_nums("assert_false", 1)
endfun

" test foldtext without line numbering doesn't include the folded line number styles
func Test_no_lnums_in_fallback()
  let g:html_prevent_copy = 'tn'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 0
  let g:html_use_input_for_pc = "fallback"

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_foldtext("assert_true")
  call Check_input_foldtext("assert_true")
  
  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")
  call Check_input_line_nums("assert_false", 1)

  call s:open_diff(1)
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_foldtext("assert_true")
  call Check_input_foldtext("assert_true")
  
  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")
  call Check_input_line_nums("assert_false", 1)
endfun
func Test_no_lnums_in_all()
  let g:html_prevent_copy = 'tn'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 0
  let g:html_use_input_for_pc = "all"

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_foldtext("assert_false")
  call Check_input_foldtext("assert_true")
  call Check_generated_line_nums("assert_false", 1)
  call Check_input_line_nums("assert_false", 1)

  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")

  call s:open_diff(1)
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_foldtext("assert_false")
  call Check_input_foldtext("assert_true")
  call Check_generated_line_nums("assert_false", 1)
  call Check_input_line_nums("assert_false", 1)
  
  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")
endfun
func Test_no_lnums_in_none()
  let g:html_prevent_copy = 'tn'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 0
  let g:html_use_input_for_pc = "none"

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")
  call Check_generated_foldtext("assert_true")
  call Check_input_foldtext("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_input_line_nums("assert_false", 1)

  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")

  call s:open_diff(1)
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")
  call Check_generated_foldtext("assert_true")
  call Check_input_foldtext("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_input_line_nums("assert_false", 1)

  call Check_generated_diff_filler("assert_false")
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_diff_filler("assert_false")
  call Check_input_fold_column("assert_false")
endfun

" test just diff filler uncopyable regions, explicitly using the old <input>
" method for all browsers
func Test_diff_filler()
  let g:html_prevent_copy = 'd'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 1
  let g:html_diff_one_file = 1
  let g:html_use_input_for_pc = "all"

  call s:open_diff(0)

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_diff_filler("assert_false")
  call Check_input_diff_filler("assert_true")
  
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_generated_foldtext("assert_false")
  call Check_input_fold_column("assert_false")
  call Check_input_line_nums("assert_false", 1)
  call Check_input_foldtext("assert_false")

  let g:html_diff_one_file = 0
  bd!
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")
  call Check_generated_diff_filler("assert_false")
  call Check_input_diff_filler("assert_true")
  
  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_generated_line_nums("assert_false", 1)
  call Check_generated_foldtext("assert_false")
  call Check_input_fold_column("assert_false")
  call Check_input_line_nums("assert_false", 1)
  call Check_input_foldtext("assert_false")
endfun

" test all uncopyable regions with default values for implementation method
" (primary generated content, fallback to using inputs)
func Test_all_regions()
  let g:html_prevent_copy = 'fntd'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 1
  let g:html_diff_one_file = 1
  let g:html_hover_unfold = 1 " why not
  unlet! g:html_use_input_for_pc

  call s:open_diff(0)

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_fold_column("assert_true")
  call Check_combined_fold_column("assert_true")
  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 1)
  call Check_generated_foldtext("assert_true")
  call Check_input_fold_column("assert_true")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 1)
  call Check_input_foldtext("assert_true")

  let g:html_diff_one_file = 0
  unlet g:html_hover_unfold
  " After processing, the folds will all be open in just one side of the diff,
  " which TOhtml will happily match. This doesn't really cause a problem for
  " this particular test but it looks weird.
  bd!
  %foldclose
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_fold_column("assert_true")
  call Check_combined_fold_column("assert_true")
  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 1)
  call Check_generated_foldtext("assert_true")
  call Check_input_fold_column("assert_true")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 1)
  call Check_input_foldtext("assert_true")
endfun

" Test all uncopyable regions with default values for implementation method
" (primary generated content, fallback to using inputs), including invalid input
" for the <input> method
func Test_invalid()
  let g:html_prevent_copy = 'fntd'
  let g:html_no_invalid = 0
  let g:html_dynamic_folds = 1
  let g:html_number_lines = 1
  let g:html_diff_one_file = 1
  unlet! g:html_use_input_for_pc

  call s:open_diff(0)

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_invalid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_fold_column("assert_true")
  call Check_combined_fold_column("assert_true")
  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 1)
  call Check_generated_foldtext("assert_true")
  call Check_input_fold_column("assert_true")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 1)
  call Check_input_foldtext("assert_true")

  let g:html_diff_one_file = 0
  bd!
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_invalid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_fold_column("assert_true")
  call Check_combined_fold_column("assert_true")
  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 1)
  call Check_generated_foldtext("assert_true")
  call Check_input_fold_column("assert_true")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 1)
  call Check_input_foldtext("assert_true")
endfun

" test that line numbers work fine without folding enabled
" (primary generated content, fallback to using inputs)
func Test_folding_disabled()
  let g:html_prevent_copy = 'fntd'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 0
  let g:html_number_lines = 1
  let g:html_diff_one_file = 1
  unlet! g:html_use_input_for_pc

  call s:open_diff(0)

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 0)
  call Check_generated_foldtext("assert_true")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 0)
  call Check_input_foldtext("assert_true")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")

  let g:html_diff_one_file = 0
  " After processing, the folds will all be open in just one side of the diff,
  " which TOhtml will happily match. This doesn't really cause a problem for
  " this particular test but it looks weird.
  bd!
  %foldclose
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 0)
  call Check_generated_foldtext("assert_true")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 0)
  call Check_input_foldtext("assert_true")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")
endfun

" test that line numbers work fine without folding enabled
" (force generated content)
func Test_folding_disabled_generated()
  let g:html_prevent_copy = 'fntd'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 0
  let g:html_number_lines = 1
  let g:html_diff_one_file = 1
  let g:html_use_input_for_pc = "none"

  call s:open_diff(0)

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")

  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 0)
  call Check_generated_foldtext("assert_true")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 0)
  call Check_input_foldtext("assert_false")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")

  let g:html_diff_one_file = 0
  " After processing, the folds will all be open in just one side of the diff,
  " which TOhtml will happily match. This doesn't really cause a problem for
  " this particular test but it looks weird.
  bd!
  %foldclose
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")

  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 0)
  call Check_generated_foldtext("assert_true")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 0)
  call Check_input_foldtext("assert_false")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")
endfun

" test that line numbers work fine without folding enabled
" (force using input method)
func Test_folding_disabled_inputs()
  let g:html_prevent_copy = 'fntd'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 0
  let g:html_number_lines = 1
  let g:html_diff_one_file = 1
  let g:html_use_input_for_pc = "all"

  call s:open_diff(0)

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 0)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 0)
  call Check_input_foldtext("assert_true")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")

  let g:html_diff_one_file = 0
  " After processing, the folds will all be open in just one side of the diff,
  " which TOhtml will happily match. This doesn't really cause a problem for
  " this particular test but it looks weird.
  bd!
  %foldclose
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 0)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 0)
  call Check_input_foldtext("assert_true")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")
endfun

" test that line numbers work fine without a fold column, but with hover unfold
" (primary generated content, fallback to using inputs)
func Test_lnum_no_foldcolumn()
  let g:html_prevent_copy = 'fntd'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_no_foldcolumn = 1
  let g:html_number_lines = 1
  let g:html_diff_one_file = 1
  unlet! g:html_use_input_for_pc

  call s:open_diff(0)

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 0)
  call Check_generated_foldtext("assert_true")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 0)
  call Check_input_foldtext("assert_true")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")

  let g:html_diff_one_file = 0
  " After processing, the folds will all be open in just one side of the diff,
  " which TOhtml will happily match. This doesn't really cause a problem for
  " this particular test but it looks weird.
  bd!
  %foldclose
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_true")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 0)
  call Check_generated_foldtext("assert_true")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 0)
  call Check_input_foldtext("assert_true")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")
endfun

" test that line numbers work fine without folding enabled
" (force generated content)
func Test_lnum_no_foldcolumn_generated()
  let g:html_prevent_copy = 'fntd'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_no_foldcolumn = 1
  let g:html_number_lines = 1
  let g:html_diff_one_file = 1
  let g:html_use_input_for_pc = "none"

  call s:open_diff(0)

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")

  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 0)
  call Check_generated_foldtext("assert_true")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 0)
  call Check_input_foldtext("assert_false")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")

  let g:html_diff_one_file = 0
  " After processing, the folds will all be open in just one side of the diff,
  " which TOhtml will happily match. This doesn't really cause a problem for
  " this particular test but it looks weird.
  bd!
  %foldclose
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_false")

  call Check_generated_diff_filler("assert_true")
  call Check_generated_line_nums("assert_true", 0)
  call Check_generated_foldtext("assert_true")
  call Check_input_diff_filler("assert_false")
  call Check_input_line_nums("assert_false", 0)
  call Check_input_foldtext("assert_false")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")
endfun

" test that line numbers work fine without folding enabled
" (force using input method)
func Test_lnum_no_foldcolumn_inputs()
  let g:html_prevent_copy = 'fntd'
  let g:html_no_invalid = 1
  let g:html_dynamic_folds = 1
  let g:html_no_foldcolumn = 1
  let g:html_number_lines = 1
  let g:html_diff_one_file = 1
  let g:html_use_input_for_pc = "all"

  call s:open_diff(0)

  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 0)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 0)
  call Check_input_foldtext("assert_true")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")

  let g:html_diff_one_file = 0
  " After processing, the folds will all be open in just one side of the diff,
  " which TOhtml will happily match. This doesn't really cause a problem for
  " this particular test but it looks weird.
  bd!
  %foldclose
  TOhtml
  call Tohtml_test_util__assert_cur_buffer_valid()

  call s:Check_fallback_supports_rule("assert_false")
  call Check_shared_css_input_rules("assert_true")

  call Check_generated_diff_filler("assert_false")
  call Check_generated_line_nums("assert_false", 0)
  call Check_generated_foldtext("assert_false")
  call Check_input_diff_filler("assert_true")
  call Check_input_line_nums("assert_true", 0)
  call Check_input_foldtext("assert_true")

  call Check_generated_fold_column("assert_false")
  call Check_combined_fold_column("assert_false")
  call Check_input_fold_column("assert_false")
endfun

func TearDown()
  diffoff!
  exec 'b!' s:starting_bufnr
  only!
endfun
