package com.webobjects.jdbcadaptor;

import java.io.*;

import com.webobjects.eoaccess.*;
import com.webobjects.foundation.*;
import com.webobjects.foundation.NSComparator.*;

import er.extensions.eof.*;
import er.extensions.foundation.*;
import er.extensions.jdbc.*;



/**
 * <p>Sub-class of <code>com.webobjects.jdbcadaptor.MicrosoftPlugIn</code> to customize its functionality.
 * Use this by including this line in your connnection dictionary:</p>
 *
 * <pre>
 * plugin=er.extensions.jdbcadaptor.ERXMicrosoftPlugIn;
 * </pre>
 * @see com.webobjects.jdbcadaptor.MicrosoftPlugIn
 * @author chill
 */
public class ERXMicrosoftPlugIn extends com.webobjects.jdbcadaptor.MicrosoftPlugIn {

    private static final String QUERY_STRING_USE_BUNDLED_JDBC_INFO = "useBundledJdbcInfo";

    /**
     * @param adaptor the JDBCAdaptor that this adaptor configures
     */
    public ERXMicrosoftPlugIn(JDBCAdaptor adaptor) {
        super(adaptor);
    }

    /**
     * <P>This method returns true if the connection URL for the
     * database has a special flag on it which indicates to the
     * system that the jdbcInfo which has been bundled into the
     * plugin is acceptable to use in place of actually going to
     * the database and getting it.
     * @return <code>true</code> if the URL contains <code>useBundledJdbcInfo=</code> with true or yes
     */
    protected boolean shouldUseBundledJdbcInfo() {
      boolean shouldUseBundledJdbcInfo = false;
      String url = connectionURL();
      if (url != null) {
          shouldUseBundledJdbcInfo = url.toLowerCase().matches(".*(\\?|\\?.*&)" + QUERY_STRING_USE_BUNDLED_JDBC_INFO.toLowerCase() + "=(true|yes)(;.*|\\&.*|$)");
      }

      return shouldUseBundledJdbcInfo;
    }

    /**
     * <P>This is usually extracted from the the database using
     * JDBC, but this is really inconvenient for users who are
     * trying to generate SQL at some.  A specific version of the
     * data has been written into the property list of the
     * framework and this can be used as a hard-coded equivalent.
     * </P>
     */
    public NSDictionary jdbcInfo() {
      // you can swap this code out to write the property list out in order
      // to get a fresh copy of the JDBCInfo.plist.
//      try {
//        String jdbcInfoS = NSPropertyListSerialization.stringFromPropertyList(super.jdbcInfo());
//        FileOutputStream fos = new FileOutputStream("/tmp/JDBCInfo.plist");
//        fos.write(jdbcInfoS.getBytes());
//        fos.close();
//      }
//      catch(Exception e) {
//        throw new IllegalStateException("problem writing JDBCInfo.plist",e);
//      }

      NSDictionary jdbcInfo;
      // have a look at the JDBC connection URL to see if the flag has been set to
      // specify that the hard-coded jdbcInfo information should be used.
      if (shouldUseBundledJdbcInfo())
        {
            NSLog.debug.appendln("Loading jdbcInfo from JDBCInfo.plist as opposed to using the JDBCPlugIn default implementation.");

            InputStream jdbcInfoStream = ERXMicrosoftPlugIn.class.getResourceAsStream("/ERXMicrosoftPlugInJDBCInfo.plist");
            if (jdbcInfoStream == null)
            {
                throw new IllegalStateException("Unable to find 'ERXMicrosoftPlugInJDBCInfo.plist' in this plugin jar.");
            }

            try
            {
                jdbcInfo = (NSDictionary) NSPropertyListSerialization.propertyListFromData(new NSData(jdbcInfoStream, 2048), "US-ASCII");
            }
            catch (IOException e)
            {
                throw new RuntimeException("Failed to load 'ERXMicrosoftPlugInJDBCInfo.plist' from this plugin jar.", e);
            }
        }
      else {
        jdbcInfo = super.jdbcInfo();
      }
      return jdbcInfo;
    }

    public EOSynchronizationFactory createSynchronizationFactory() {
        NSLog.debug.appendln("Called createSynchronizationFactory");

        return new ERXMicrosoftSynchronizationFactory(_adaptor);
    }



    /**
     * Sub-class of <code>com.webobjects.jdbcadaptor.MicrosoftPlugIn.MicrosoftSynchronizationFactory</code> to customize
     * SQL generation.
     *
     * @see com.webobjects.jdbcadaptor.MicrosoftPlugIn.MicrosoftSynchronizationFactory
     * @see com.webobjects.eoaccess.EOSynchronizationFactory
     */
    public static class ERXMicrosoftSynchronizationFactory extends EOSynchronizationFactory {


        /**
         * As hacks go, this one is pretty brutal.  In WO 5.4, the EOSynchronizationFactory was deprecated, and a
         * superclass, EOSchemaSynchronizationFactory, created.  EOSynchronizationFactory delegates many of its
         * former actions to an internal instance of EOSchemaSynchronizationFactory.  This mobius strip of mangled
         * logic makes customizing the behavior quite difficult, particularly if you are trying to maintain compatibility
         * with WO 5.3 an earlier versions.
         * <p>
         * The original problem behind this has was that as of 5.4, dropTableStatementsForEntityGroups no longer got
         * called (the version in EOSchemaSynchronizationFactory gets called, instead of the custom version in this class).
         * Rather than try and contort this code to get that called, the list of entities is sorted before any SQL
         * is generated.  Once 5.3 compatibility is no longer required, it should be possible to migrate this class
         * to the newer API and do a lot of cleanup.   In the meantime, there is this odious hack.
         * </p>
         *
         * @see com.webobjects.eoaccess.EOSynchronizationFactory#schemaCreationScriptForEntities(com.webobjects.foundation.NSArray, com.webobjects.foundation.NSDictionary)
         *
         * @param allEntities list of EOEntity to generate schema for
         * @param options schema generation options
         * @return SQL as a string
         */
        public String schemaCreationScriptForEntities(NSArray allEntities, NSDictionary options) {
            options = (options != null) ? options : NSDictionary.EmptyDictionary;

            // Convert the list of entities into entity groups so that we can sort them with the existing code
            NSArray groupedEntities = tableEntityGroupsForEntities(allEntities);

            // Sort the groups entities
            ERXEntityFKConstraintOrder constraintOrder = new ERXEntityFKConstraintOrder(ERXJDBCPlugInUtilities.modelGroupForEntityGroups(groupedEntities));
            NSComparator entityOrderingComparator = new ERXJDBCPlugInUtilities.EntityGroupDeleteOrderComparator(constraintOrder);
            NSArray sortedEntities = NSArray.EmptyArray;
            try {
                sortedEntities = groupedEntities.sortedArrayUsingComparator(entityOrderingComparator);
            }
            catch (ComparisonException e) {
                throw NSForwardException._runtimeExceptionForThrowable(e);
            }

            // Convert the grouped entities back into a normal list
            sortedEntities = ERXArrayUtilities.flatten(sortedEntities);

            StringBuffer result = new StringBuffer();
            NSArray statements = schemaCreationStatementsForEntities(sortedEntities, options);
            int i = 0;
            for(int count = statements.count(); i < count; i++)
                appendExpressionToScript((EOSQLExpression)statements.objectAtIndex(i), result);
            return new String(result);
        }


        /**
         * Uses ERXEntityFKConstraintOrder to order the entities before generating the DROP TABLE statements as MSSQL does not
         * support deferred constraints.  Hence the tables need to be dropped so as to avoid foreign key constraint violations.
         *
         * This method is not called due to API changes in 5.4.
         *
         * @see com.webobjects.eoaccess.EOSynchronizationFactory#dropTableStatementsForEntityGroups(com.webobjects.foundation.NSArray)
         *
         * @param entityGroups array of arrays each containing one EOEntity
         * @return SQL to drop the tables for these entities in an order that avoids foreign key constraint violations
         */
        public NSArray dropTableStatementsForEntityGroups(NSArray entityGroups) {

            // SQL generation from Entity Modeler does not use the EOModelGroup.defaultGroup
            ERXEntityFKConstraintOrder constraintOrder = new ERXEntityFKConstraintOrder(ERXJDBCPlugInUtilities.modelGroupForEntityGroups(entityGroups));
            NSComparator entityOrderingComparator = new ERXJDBCPlugInUtilities.EntityGroupDeleteOrderComparator(constraintOrder);
            try {
                return super.dropTableStatementsForEntityGroups(entityGroups.sortedArrayUsingComparator(entityOrderingComparator));
            }
            catch (ComparisonException e) {
                throw NSForwardException._runtimeExceptionForThrowable(e);
            }
        }

        public NSArray primaryKeySupportStatementsForEntityGroups(NSArray entityGroups) {
            String pkTable = ((JDBCAdaptor) adaptor()).plugIn().primaryKeyTableName();
            NSMutableArray statements = new NSMutableArray();
            statements.addObject(_expressionForString((new StringBuilder()).append("CREATE TABLE ").append(pkTable).append(" (NAME CHAR(40) PRIMARY KEY, PK INT)").toString()));
            return statements;
        }

        public NSArray dropPrimaryKeySupportStatementsForEntityGroups(NSArray entityGroups) {
            return new NSArray(_expressionForString((new StringBuilder()).append("DROP TABLE ").append(((JDBCAdaptor) adaptor()).plugIn().primaryKeyTableName()).toString()));
        }

        public NSArray _statementsToDeleteTableNamedOptions(String tableName, EOSchemaGeneration options) {
            return new NSArray(_expressionForString((new StringBuilder()).append("DROP TABLE ").append(tableName).toString()));
        }

        public NSArray dropTableStatementsForEntityGroup(NSArray entityGroup) {
            if (entityGroup == null) {
                return NSArray.EmptyArray;
            }

            com.webobjects.eoaccess.EOSQLExpression sqlExpr = _expressionForString((new StringBuilder()).append("DROP TABLE ").append(((EOEntity) entityGroup.objectAtIndex(0)).externalName()).toString());
            return new NSArray(sqlExpr);
        }

        public boolean supportsSchemaSynchronization() {
            return false;
        }

        public NSArray _statementsToDropPrimaryKeyConstraintsOnTableNamed(String tableName) {
            return NSArray.EmptyArray;
        }

        public NSArray statementsToRenameTableNamed(String tableName, String newName, EOSchemaGeneration options) {
            String aTableName = tableName;
            int lastDot = aTableName.lastIndexOf('.');
            if (lastDot != -1) {
                aTableName = aTableName.substring(lastDot + 1);
            }
            String aNewName = newName;
            lastDot = aNewName.lastIndexOf('.');
            if (lastDot != -1) {
                aNewName = aNewName.substring(lastDot + 1);
            }
            return new NSArray(_expressionForString((new StringBuilder()).append("execute sp_rename ").append(aTableName).append(", ").append(aNewName).toString()));
        }

        public ERXMicrosoftSynchronizationFactory(EOAdaptor adaptor) {
            super(adaptor);
            NSLog.debug.appendln("Created ERXMicrosoftSynchronizationFactory");

        }

    }

}
