/*
 * OFF2DEV - logical to device translation layer
 *
 * Copyright 2000 Patrik Stridvall
 */

#include <stdlib.h>

#include "gdi.h"

/**********************************************************************/

#define SWAP_INT(a, b)  do { int t = a; a = b; b = t; } while(0)

/**********************************************************************/

BOOL OFF2DEV_Arc(DC *dc, INT left, INT top, INT right, INT bottom,
		 INT xstart, INT ystart, INT xend, INT yend);
BOOL OFF2DEV_Chord(DC *dc, INT left, INT top, INT right, INT bottom,
		   INT xstart, INT ystart, INT xend, INT yend);
BOOL OFF2DEV_Ellipse(DC *dc, INT left, INT top, INT right, INT bottom);
BOOL OFF2DEV_ExtFloodFill(DC *dc, INT x, INT y, COLORREF color,
			 UINT fillType);
BOOL OFF2DEV_ExtTextOut(DC *dc, INT x, INT y, UINT flags,
			const RECT *lprect, LPCWSTR wstr, UINT count,
			const INT *lpDx);
COLORREF OFF2DEV_GetPixel(DC *dc, INT x, INT y);
BOOL OFF2DEV_LineTo(DC *dc, INT x, INT y);
BOOL OFF2DEV_MoveToEx(DC *dc, INT x, INT y, LPPOINT pt);
BOOL OFF2DEV_Pie(DC *dc, INT left, INT top, INT right, INT bottom,
		 INT xstart, INT ystart, INT xend, INT yend);
BOOL OFF2DEV_Polygon(DC *dc, const POINT *pt, INT count);
BOOL OFF2DEV_Polyline(DC *dc, const POINT *pt, INT count);
BOOL OFF2DEV_PolyPolygon(DC *dc, const POINT *pt,
			 const INT *counts, UINT polygons);
BOOL OFF2DEV_PolyPolyline(DC *dc, const POINT *pt,
			  const DWORD *counts, DWORD polylines);
BOOL OFF2DEV_Rectangle(DC *dc, INT left, INT top, INT right, INT bottom);
BOOL OFF2DEV_RoundRect(DC *dc, INT left, INT top, INT right,
		      INT bottom, INT ell_width, INT ell_height);
COLORREF OFF2DEV_SetPixel(DC *dc, INT x, INT y, COLORREF color);
BOOL OFF2DEV_StretchBlt(DC *dcDst, INT xDst, INT yDst,
			INT widthDst, INT heightDst,
			DC *dcSrc, INT xSrc, INT ySrc,
			INT widthSrc, INT heightSrc, DWORD rop);

/***********************************************************************
 *		OFF2DEV_DC_Funcs
 */
const DC_FUNCTIONS OFF2DEV_DC_Funcs = {
  NULL,                            /* pAbortDoc */
  NULL,                            /* pAbortPath */
  NULL,                            /* pAngleArc */
  OFF2DEV_Arc,                     /* pArc */
  NULL,                            /* pArcTo */
  NULL,                            /* pBeginPath */
  NULL,                            /* pBitBlt */
  NULL,                            /* pBitmapBits */
  NULL,                            /* pChoosePixelFormat */
  OFF2DEV_Chord,                   /* pChord */
  NULL,                            /* pCloseFigure */
  NULL,                            /* pCreateBitmap */
  NULL,                            /* pCreateDC */
  NULL,                            /* pCreateDIBSection */
  NULL,                            /* pCreateDIBSection16 */
  NULL,                            /* pDeleteDC */
  NULL,                            /* pDeleteObject */
  NULL,                            /* pDescribePixelFormat */
  NULL,                            /* pDeviceCapabilities */
  OFF2DEV_Ellipse,                 /* pEllipse */
  NULL,                            /* pEndDoc */
  NULL,                            /* pEndPage */
  NULL,                            /* pEndPath */
  NULL,                            /* pEnumDeviceFonts */
  NULL,                            /* pEscape */
  NULL,                            /* pExcludeClipRect */
  NULL,                            /* pExtDeviceMode */
  OFF2DEV_ExtFloodFill,            /* pExtFloodFill */
  OFF2DEV_ExtTextOut,              /* pExtTextOut */
  NULL,                            /* pFillPath */
  NULL,                            /* pFillRgn */
  NULL,                            /* pFlattenPath */
  NULL,                            /* pFrameRgn */
  NULL,                            /* pGetCharWidth */
  NULL,                            /* pGetDCOrgEx */
  OFF2DEV_GetPixel,                /* pGetPixel */
  NULL,                            /* pGetPixelFormat */
  NULL,                            /* pGetTextExtentPoint */
  NULL,                            /* pGetTextMetrics */
  NULL,                            /* pIntersectClipRect */
  NULL,                            /* pInvertRgn */
  OFF2DEV_LineTo,                  /* pLineTo */
  OFF2DEV_MoveToEx,                /* pMoveToEx */
  NULL,                            /* pOffsetClipRgn */
  NULL,                            /* pOffsetViewportOrg (optional) */
  NULL,                            /* pOffsetWindowOrg (optional) */
  NULL,                            /* pPaintRgn */
  NULL,                            /* pPatBlt */
  OFF2DEV_Pie,                     /* pPie */
  NULL,                            /* pPolyBezier */
  NULL,                            /* pPolyBezierTo */
  NULL,                            /* pPolyDraw */
  OFF2DEV_PolyPolygon,             /* pPolyPolygon */
  OFF2DEV_PolyPolyline,            /* pPolyPolyline */
  OFF2DEV_Polygon,                 /* pPolygon */
  OFF2DEV_Polyline,                /* pPolyline */
  NULL,                            /* pPolylineTo */
  NULL,                            /* pRealizePalette */
  OFF2DEV_Rectangle,               /* pRectangle */
  NULL,                            /* pRestoreDC */
  OFF2DEV_RoundRect,               /* pRoundRect */
  NULL,                            /* pSaveDC */
  NULL,                            /* pScaleViewportExt (optional) */
  NULL,                            /* pScaleWindowExt (optional) */
  NULL,                            /* pSelectClipPath */
  NULL,                            /* pSelectClipRgn */
  NULL,                            /* pSelectObject */
  NULL,                            /* pSelectPalette */
  NULL,                            /* pSetBkColor */
  NULL,                            /* pSetBkMode */
  NULL,                            /* pSetDeviceClipping */
  NULL,                            /* pSetDIBitsToDevice */
  NULL,                            /* pSetMapMode (optional) */
  NULL,                            /* pSetMapperFlags */
  NULL,                            /* pSetPixel */
  NULL,                            /* pSetPixelFormat */
  NULL,                            /* pSetPolyFillMode */
  NULL,                            /* pSetROP2 */
  NULL,                            /* pSetRelAbs */
  NULL,                            /* pSetStretchBltMode */
  NULL,                            /* pSetTextAlign */
  NULL,                            /* pSetTextCharacterExtra */
  NULL,                            /* pSetTextColor */
  NULL,                            /* pSetTextJustification */
  NULL,                            /* pSetViewportExt (optional) */
  NULL,                            /* pSetViewportOrg (optional) */
  NULL,                            /* pSetWindowExt (optional) */
  NULL,                            /* pSetWindowOrg (optional) */
  NULL,                            /* pStartDoc */
  NULL,                            /* pStartPage */
  OFF2DEV_StretchBlt,              /* pStretchBlt */
  NULL,                            /* pStretchDIBits */
  NULL,                            /* pStrokeAndFillPath */
  NULL,                            /* pStrokePath */
  NULL,                            /* pSwapBuffers */
  NULL                             /* pWidenPath */
};

/***********************************************************************
 *		OFF2DEV_Arc
 */
BOOL OFF2DEV_Arc(DC *dc, INT left, INT top, INT right, INT bottom,
		 INT xstart, INT ystart, INT xend, INT yend)
{
  int offsetX = dc->wndOrgX - dc->vportOrgX;
  int offsetY = dc->wndOrgY - dc->vportOrgY;

  left   += offsetX;
  top    += offsetY;
  right  += offsetX;
  bottom += offsetY;
  xstart += offsetX;
  ystart += offsetY;
  xend   += offsetX;
  yend   += offsetY;
  
  return dc->devfuncs->pArc(dc, left, top, right, bottom,
			    xstart, ystart, xend, yend);
}

/***********************************************************************
 *		OFF2DEV_Chord
 */
BOOL OFF2DEV_Chord(DC *dc, INT left, INT top, INT right, INT bottom,
		   INT xstart, INT ystart, INT xend, INT yend)
{
  int offsetX = dc->wndOrgX - dc->vportOrgX;
  int offsetY = dc->wndOrgY - dc->vportOrgY;

  left   += offsetX;
  top    += offsetY;
  right  += offsetX;
  bottom += offsetY;
  xstart += offsetX;
  ystart += offsetY;
  xend   += offsetX;
  yend   += offsetY;

  return dc->devfuncs->pChord(dc, left, top, right, bottom,
			      xstart, ystart, xend, yend);
}

/***********************************************************************
 *		OFF2DEV_Ellipse
 */
BOOL OFF2DEV_Ellipse(DC *dc, INT left, INT top, INT right, INT bottom)
{
  int offsetX = dc->wndOrgX - dc->vportOrgX;
  int offsetY = dc->wndOrgY - dc->vportOrgY;

  left   += offsetX;
  top    += offsetY;
  right  += offsetX;
  bottom += offsetY;

  return dc->devfuncs->pEllipse(dc, left, top, right, bottom);
}

/***********************************************************************
 *		OFF2DEV_ExtFloodFill
 */
BOOL OFF2DEV_ExtFloodFill(DC *dc, INT x, INT y, COLORREF color,
			  UINT fillType)
{
  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  x += offsetX;
  y += offsetY;

  return dc->devfuncs->pExtFloodFill(dc, x, y, color, fillType);
}

/***********************************************************************
 *		OFF2DEV_ExtTextOut
 */
BOOL OFF2DEV_ExtTextOut(DC *dc, INT x, INT y, UINT flags,
			const RECT *lprect, LPCWSTR wstr, UINT count,
			const INT *lpDx)
{
  RECT rect;

  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  x += offsetX;
  y += offsetY;

  if(lprect) {
    rect.left   = lprect->left + offsetX;
    rect.right  = lprect->right + offsetY;
    rect.top    = lprect->top + offsetX;
    rect.bottom = lprect->bottom + offsetY;
  }

  return dc->devfuncs->pExtTextOut(dc, x, y, flags, &rect, wstr, count, lpDx);
}

/***********************************************************************
 *		OFF2DEV_GetPixel
 */
COLORREF OFF2DEV_GetPixel(DC *dc, INT x, INT y)
{
  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  x += offsetX;
  y += offsetY;

  return dc->devfuncs->pGetPixel(dc, x, y);
}

/***********************************************************************
 *		OFF2DEV_LineTo
 */
BOOL OFF2DEV_LineTo(DC *dc, INT x, INT y)
{
  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  x += offsetX;
  y += offsetY;

  return dc->devfuncs->pLineTo(dc, x, y);
}

/***********************************************************************
 *		OFF2DEV_MoveToEx
 */
BOOL OFF2DEV_MoveToEx(DC *dc, INT x, INT y, LPPOINT pt)
{
  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  x += offsetX;
  y += offsetY;

  return dc->devfuncs->pMoveToEx(dc, x, y, pt);
}

/***********************************************************************
 *		OFF2DEV_Pie
 */
BOOL OFF2DEV_Pie(DC *dc, INT left, INT top, INT right, INT bottom,
		 INT xstart, INT ystart, INT xend, INT yend)
{
  int offsetX = dc->wndOrgX - dc->vportOrgX;
  int offsetY = dc->wndOrgY - dc->vportOrgY;

  left   += offsetX;
  top    += offsetY;
  right  += offsetX;
  bottom += offsetY;
  xstart += offsetX;
  ystart += offsetY;
  xend   += offsetX;
  yend   += offsetY;

  return dc->devfuncs->pPie(dc, left, top, right, bottom,
			    xstart, ystart, xend, yend);
}

/***********************************************************************
 *		OFF2DEV_Polygon
 */
BOOL OFF2DEV_Polygon(DC *dc, const POINT *pt, INT count)
{
  POINT *pt2;
  INT i;
  BOOL result;

  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  pt2 = malloc(count * sizeof(POINT));

  for (i = 0; i < count; i++) {
    pt2[i].x = pt[i].x + offsetX;
    pt2[i].y = pt[i].y + offsetY;
  }

  result = dc->devfuncs->pPolygon(dc, pt2, count);

  free(pt2);

  return result;
}

/***********************************************************************
 *		OFF2DEV_Polyline
 */

BOOL OFF2DEV_Polyline(DC *dc, const POINT *pt, INT count)
{
  POINT *pt2;
  INT i;
  BOOL result;

  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  pt2 = malloc(count * sizeof(POINT));

  for (i = 0; i < count; i++) {
    pt2[i].x = pt[i].x + offsetX;
    pt2[i].y = pt[i].y + offsetY;
  }

  result = dc->devfuncs->pPolyline(dc, pt2, count);

  free(pt2);

  return result;
}

/***********************************************************************
 *		OFF2DEV_PolyPolygon
 */
BOOL OFF2DEV_PolyPolygon(DC *dc, const POINT *pt, 
			 const INT *counts, UINT polygons)
{
  const POINT *p;
  POINT *pt2, *p2;
  INT i, j, total;
  BOOL result;

  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  total = 0;
  for(i = 0; i < polygons; i++) {
      total += counts[i];
  }

  pt2 = malloc(total * sizeof(POINT));

  p = pt;
  p2 = pt2;
  for(i = 0; i < polygons; i++) {
    for(j = 0; j < counts[i]; j++) {
      p2->x = p->x + offsetX;
      p2->y = p->y + offsetY;
      p++; p2++;
    }
  }

  result = dc->devfuncs->pPolyPolygon(dc, pt2, counts, polygons);

  free(pt2);

  return result;
}

/***********************************************************************
 *		OFF2DEV_PolyPolyline
 */
BOOL OFF2DEV_PolyPolyline(DC *dc, const POINT *pt,
			  const DWORD *counts, DWORD polylines)
{
  const POINT *p;
  POINT *pt2, *p2;
  INT i, j, total;
  BOOL result;

  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  total = 0;
  for(i = 0; i < polylines; i++) {
    total += counts[i];
  }

  pt2 = malloc(total * sizeof(POINT));

  p = pt;
  p2 = pt2;
  for(i = 0; i < polylines; i++) {
    for(j = 0; j < counts[i]; j++) {
      p2->x = p->x + offsetX;
      p2->y = p->y + offsetY;
      p++; p2++;
    }
  }

  result = dc->devfuncs->pPolyPolyline(dc, pt2, counts, polylines);
  
  free(pt2);

  return result;
}

/***********************************************************************
 *		OFF2DEV_Rectangle
 */
BOOL OFF2DEV_Rectangle(DC *dc, INT left, INT top, INT right, INT bottom)
{
  int offsetX = dc->wndOrgX - dc->vportOrgX;
  int offsetY = dc->wndOrgY - dc->vportOrgY;

  left   += offsetX;
  top    += offsetY;
  right  += offsetX;
  bottom += offsetY;

  return dc->devfuncs->pRectangle(dc, left, top, right, bottom);
}

/***********************************************************************
 *		OFF2DEV_RoundRect
 */
BOOL OFF2DEV_RoundRect(DC *dc, INT left, INT top, INT right,
		       INT bottom, INT ell_width, INT ell_height)
{
  int offsetX = dc->wndOrgX - dc->vportOrgX;
  int offsetY = dc->wndOrgY - dc->vportOrgY;

  left   += offsetX;
  top    += offsetY;
  right  += offsetX;
  bottom += offsetY;

  return dc->devfuncs->pRoundRect(dc, left, top, right, bottom, ell_width, ell_height);
}

/***********************************************************************
 *		OFF2DEV_SetPixel
 */
COLORREF OFF2DEV_SetPixel(DC *dc, INT x, INT y, COLORREF color)
{
  int offsetX = dc->w.DCOrgX + (dc->wndOrgX - dc->vportOrgX);
  int offsetY = dc->w.DCOrgY + (dc->wndOrgY - dc->vportOrgY);

  x += offsetX;
  y += offsetY;
  
  return dc->devfuncs->pSetPixel(dc, x, y, color);
}

/***********************************************************************
 *		OFF2DEV_StretchBlt
 */
BOOL OFF2DEV_StretchBlt(DC *dcDst, INT xDst, INT yDst,
			INT widthDst, INT heightDst,
			DC *dcSrc, INT xSrc, INT ySrc,
			INT widthSrc, INT heightSrc, DWORD rop)
{
  BOOL useDst, useSrc;

  useDst = (((rop >> 1) & 0x550000) != (rop & 0x550000));
  if(useDst) {
    int offsetX = dcDst->w.DCOrgX + (dcDst->wndOrgX - dcDst->vportOrgX);
    int offsetY = dcDst->w.DCOrgY + (dcDst->wndOrgY - dcDst->vportOrgY);

    /* Compensate for off-by-one shifting for negative widths and heights */
    /* FIXME: This must be done before this function is called */
    if(widthDst < 0) xDst++;
    if(heightDst < 0) yDst++;

    xDst += offsetX;
    yDst += offsetY;
  }

  useSrc = (((rop >> 2) & 0x330000) != (rop & 0x330000));
  if(useSrc) {
    int offsetX = dcSrc->w.DCOrgX + (dcSrc->wndOrgX - dcSrc->vportOrgX);
    int offsetY = dcSrc->w.DCOrgY + (dcSrc->wndOrgY - dcSrc->vportOrgY);

    /* Compensate for off-by-one shifting for negative widths and heights */
    /* FIXME: This must be done before this function is called */
    if(widthSrc < 0) xSrc++;
    if(heightSrc < 0) ySrc++;

    xSrc += offsetX;
    ySrc += offsetY;
  }

  return dcDst->devfuncs->pStretchBlt(dcDst, xDst, yDst, widthDst, heightDst,
				      dcSrc, xSrc, ySrc, widthSrc, heightSrc,
				      rop);
}
