/*
 * SCL2DEV - logical to device translation layer
 *
 * Copyright 2000 Patrik Stridvall
 */

#include <stdlib.h>

#include "gdi.h"

/**********************************************************************/

#define SWAP_INT(a, b)  do { int t = a; a = b; b = t; } while(0)

/**********************************************************************/

BOOL SCL2DEV_Arc(DC *dc, INT left, INT top, INT right, INT bottom,
		 INT xstart, INT ystart, INT xend, INT yend);
BOOL SCL2DEV_Chord(DC *dc, INT left, INT top, INT right, INT bottom,
		   INT xstart, INT ystart, INT xend, INT yend);
BOOL SCL2DEV_Ellipse(DC *dc, INT left, INT top, INT right, INT bottom);
BOOL SCL2DEV_ExtFloodFill(DC *dc, INT x, INT y, COLORREF color,
			 UINT fillType);
BOOL SCL2DEV_ExtTextOut(DC *dc, INT x, INT y, UINT flags,
			const RECT *lprect, LPCWSTR wstr, UINT count,
			const INT *lpDx);
COLORREF SCL2DEV_GetPixel(DC *dc, INT x, INT y);
BOOL SCL2DEV_LineTo(DC *dc, INT x, INT y);
BOOL SCL2DEV_MoveToEx(DC *dc, INT x, INT y, LPPOINT pt);
BOOL SCL2DEV_Pie(DC *dc, INT left, INT top, INT right, INT bottom,
		 INT xstart, INT ystart, INT xend, INT yend);
BOOL SCL2DEV_Polygon(DC *dc, const POINT *pt, INT count);
BOOL SCL2DEV_Polyline(DC *dc, const POINT *pt, INT count);
BOOL SCL2DEV_PolyPolygon(DC *dc, const POINT *pt,
			 const INT *counts, UINT polygons);
BOOL SCL2DEV_PolyPolyline(DC *dc, const POINT *pt,
			  const DWORD *counts, DWORD polylines);
BOOL SCL2DEV_Rectangle(DC *dc, INT left, INT top, INT right, INT bottom);
BOOL SCL2DEV_RoundRect(DC *dc, INT left, INT top, INT right,
		      INT bottom, INT ell_width, INT ell_height);
COLORREF SCL2DEV_SetPixel(DC *dc, INT x, INT y, COLORREF color);
BOOL SCL2DEV_StretchBlt(DC *dcDst, INT xDst, INT yDst,
			INT widthDst, INT heightDst,
			DC *dcSrc, INT xSrc, INT ySrc,
			INT widthSrc, INT heightSrc, DWORD rop);

/***********************************************************************
 *		SCL2DEV_DC_Funcs
 */
const DC_FUNCTIONS SCL2DEV_DC_Funcs = {
  NULL,                            /* pAbortDoc */
  NULL,                            /* pAbortPath */
  NULL,                            /* pAngleArc */
  SCL2DEV_Arc,                     /* pArc */
  NULL,                            /* pArcTo */
  NULL,                            /* pBeginPath */
  NULL,                            /* pBitBlt */
  NULL,                            /* pBitmapBits */
  NULL,                            /* pChoosePixelFormat */
  SCL2DEV_Chord,                   /* pChord */
  NULL,                            /* pCloseFigure */
  NULL,                            /* pCreateBitmap */
  NULL,                            /* pCreateDC */
  NULL,                            /* pCreateDIBSection */
  NULL,                            /* pCreateDIBSection16 */
  NULL,                            /* pDeleteDC */
  NULL,                            /* pDeleteObject */
  NULL,                            /* pDescribePixelFormat */
  NULL,                            /* pDeviceCapabilities */
  SCL2DEV_Ellipse,                 /* pEllipse */
  NULL,                            /* pEndDoc */
  NULL,                            /* pEndPage */
  NULL,                            /* pEndPath */
  NULL,                            /* pEnumDeviceFonts */
  NULL,                            /* pEscape */
  NULL,                            /* pExcludeClipRect */
  NULL,                            /* pExtDeviceMode */
  SCL2DEV_ExtFloodFill,            /* pExtFloodFill */
  SCL2DEV_ExtTextOut,              /* pExtTextOut */
  NULL,                            /* pFillPath */
  NULL,                            /* pFillRgn */
  NULL,                            /* pFlattenPath */
  NULL,                            /* pFrameRgn */
  NULL,                            /* pGetCharWidth */
  NULL,                            /* pGetDCOrgEx */
  SCL2DEV_GetPixel,                /* pGetPixel */
  NULL,                            /* pGetPixelFormat */
  NULL,                            /* pGetTextExtentPoint */
  NULL,                            /* pGetTextMetrics */
  NULL,                            /* pIntersectClipRect */
  NULL,                            /* pInvertRgn */
  SCL2DEV_LineTo,                  /* pLineTo */
  SCL2DEV_MoveToEx,                /* pMoveToEx */
  NULL,                            /* pOffsetClipRgn */
  NULL,                            /* pOffsetViewportOrg (optional) */
  NULL,                            /* pOffsetWindowOrg (optional) */
  NULL,                            /* pPaintRgn */
  NULL,                            /* pPatBlt */
  SCL2DEV_Pie,                     /* pPie */
  NULL,                            /* pPolyBezier */
  NULL,                            /* pPolyBezierTo */
  NULL,                            /* pPolyDraw */
  SCL2DEV_PolyPolygon,             /* pPolyPolygon */
  SCL2DEV_PolyPolyline,            /* pPolyPolyline */
  SCL2DEV_Polygon,                 /* pPolygon */
  SCL2DEV_Polyline,                /* pPolyline */
  NULL,                            /* pPolylineTo */
  NULL,                            /* pRealizePalette */
  SCL2DEV_Rectangle,               /* pRectangle */
  NULL,                            /* pRestoreDC */
  SCL2DEV_RoundRect,               /* pRoundRect */
  NULL,                            /* pSaveDC */
  NULL,                            /* pScaleViewportExt (optional) */
  NULL,                            /* pScaleWindowExt (optional) */
  NULL,                            /* pSelectClipPath */
  NULL,                            /* pSelectClipRgn */
  NULL,                            /* pSelectObject */
  NULL,                            /* pSelectPalette */
  NULL,                            /* pSetBkColor */
  NULL,                            /* pSetBkMode */
  NULL,                            /* pSetDeviceClipping */
  NULL,                            /* pSetDIBitsToDevice */
  NULL,                            /* pSetMapMode (optional) */
  NULL,                            /* pSetMapperFlags */
  NULL,                            /* pSetPixel */
  NULL,                            /* pSetPixelFormat */
  NULL,                            /* pSetPolyFillMode */
  NULL,                            /* pSetROP2 */
  NULL,                            /* pSetRelAbs */
  NULL,                            /* pSetStretchBltMode */
  NULL,                            /* pSetTextAlign */
  NULL,                            /* pSetTextCharacterExtra */
  NULL,                            /* pSetTextColor */
  NULL,                            /* pSetTextJustification */
  NULL,                            /* pSetViewportExt (optional) */
  NULL,                            /* pSetViewportOrg (optional) */
  NULL,                            /* pSetWindowExt (optional) */
  NULL,                            /* pSetWindowOrg (optional) */
  NULL,                            /* pStartDoc */
  NULL,                            /* pStartPage */
  SCL2DEV_StretchBlt,              /* pStretchBlt */
  NULL,                            /* pStretchDIBits */
  NULL,                            /* pStrokeAndFillPath */
  NULL,                            /* pStrokePath */
  NULL,                            /* pSwapBuffers */
  NULL                             /* pWidenPath */
};

/***********************************************************************
 *		SCL2DEV_Arc
 */
BOOL SCL2DEV_Arc(DC *dc, INT left, INT top, INT right, INT bottom,
		 INT xstart, INT ystart, INT xend, INT yend)
{
  left   = XLPTODP(dc, left);
  top    = YLPTODP(dc, top);
  right  = XLPTODP(dc, right);
  bottom = YLPTODP(dc, bottom);
  xstart = XLPTODP(dc, xstart);
  ystart = YLPTODP(dc, ystart);
  xend   = XLPTODP(dc, xend);
  yend   = YLPTODP(dc, yend);

  if(right < left) SWAP_INT(left, right);
  if(bottom < top) SWAP_INT(top, bottom);
  
  return dc->devfuncs->pArc(dc, left, top, right, bottom,
			    xstart, ystart, xend, yend);
}

/***********************************************************************
 *		SCL2DEV_Chord
 */
BOOL SCL2DEV_Chord(DC *dc, INT left, INT top, INT right, INT bottom,
		   INT xstart, INT ystart, INT xend, INT yend)
{
  left   = XLPTODP(dc, left);
  top    = YLPTODP(dc, top);
  right  = XLPTODP(dc, right);
  bottom = YLPTODP(dc, bottom);
  xstart = XLPTODP(dc, xstart);
  ystart = YLPTODP(dc, ystart);
  xend   = XLPTODP(dc, xend);
  yend   = YLPTODP(dc, yend);

  if(right < left) SWAP_INT(left, right);
  if(bottom < top) SWAP_INT(top, bottom);

  if((right - left == 1) || (bottom - top == 1))
    return TRUE;

  return dc->devfuncs->pChord(dc, left, top, right, bottom,
			      xstart, ystart, xend, yend);
}

/***********************************************************************
 *		SCL2DEV_Ellipse
 */
BOOL SCL2DEV_Ellipse(DC *dc, INT left, INT top, INT right, INT bottom)
{
  left   = XLPTODP(dc, left);
  top    = YLPTODP(dc, top);
  right  = XLPTODP(dc, right);
  bottom = YLPTODP(dc, bottom);

  if(right < left) SWAP_INT(left, right);
  if(bottom < top) SWAP_INT(top, bottom);

  return dc->devfuncs->pEllipse(dc, left, top, right, bottom);
}

/***********************************************************************
 *		SCL2DEV_ExtFloodFill
 */
BOOL SCL2DEV_ExtFloodFill(DC *dc, INT x, INT y, COLORREF color,
			  UINT fillType)
{
  x = dc->w.DCOrgX + XLPTODP(dc, x);
  y = dc->w.DCOrgY + YLPTODP(dc, y);

  return dc->devfuncs->pExtFloodFill(dc, x, y, color, fillType);
}

/***********************************************************************
 *		SCL2DEV_ExtTextOut
 */
BOOL SCL2DEV_ExtTextOut(DC *dc, INT x, INT y, UINT flags,
			const RECT *lprect, LPCWSTR wstr, UINT count,
			const INT *lpDx)
{
  BOOL result;
  RECT rect;
  INT *lpDx2;

  x = XLPTODP(dc, x);
  y = YLPTODP(dc, y);

  if(lprect) {
    rect.left   = XLPTODP(dc, lprect->left);
    rect.right  = XLPTODP(dc, lprect->right);
    rect.top    = YLPTODP(dc, lprect->top);
    rect.bottom = YLPTODP(dc, lprect->bottom);

    if(rect.right < rect.left) SWAP_INT(rect.left, rect.right);
    if(rect.bottom < rect.top) SWAP_INT(rect.top, rect.bottom);
  }

  if(lpDx) {
    int i;

    lpDx2 = malloc(count * sizeof(INT));
    for (i = 0; i < count; i++) {
      lpDx2[i] = XLSTODS(dc, lpDx[i]);
    }
  } else {
    lpDx2 = NULL;
  }

  result = dc->devfuncs->pExtTextOut(dc, x, y, flags, &rect, wstr, count, lpDx2);

  if(lpDx2) {
    free(lpDx2);
  }

  return result;
}

/***********************************************************************
 *		SCL2DEV_GetPixel
 */
COLORREF SCL2DEV_GetPixel(DC *dc, INT x, INT y)
{
  x = dc->w.DCOrgX + XLPTODP(dc, x);
  y = dc->w.DCOrgY + YLPTODP(dc, y);

  return dc->devfuncs->pGetPixel(dc, x, y);
}

/***********************************************************************
 *		SCL2DEV_LineTo
 */
BOOL SCL2DEV_LineTo(DC *dc, INT x, INT y)
{
  x = dc->w.DCOrgX + XLPTODP(dc, x);
  y = dc->w.DCOrgY + YLPTODP(dc, y);

  return dc->devfuncs->pLineTo(dc, x, y);
}

/***********************************************************************
 *		SCL2DEV_MoveToEx
 */
BOOL SCL2DEV_MoveToEx(DC *dc, INT x, INT y, LPPOINT pt)
{
  x = dc->w.DCOrgX + XLPTODP(dc, x);
  y = dc->w.DCOrgY + YLPTODP(dc, y);

  return dc->devfuncs->pMoveToEx(dc, x, y, pt);
}

/***********************************************************************
 *		SCL2DEV_Pie
 */
BOOL SCL2DEV_Pie(DC *dc, INT left, INT top, INT right, INT bottom,
		 INT xstart, INT ystart, INT xend, INT yend)
{
  left   = XLPTODP(dc, left);
  top    = YLPTODP(dc, top);
  right  = XLPTODP(dc, right);
  bottom = YLPTODP(dc, bottom);
  xstart = XLPTODP(dc, xstart);
  ystart = YLPTODP(dc, ystart);
  xend   = XLPTODP(dc, xend);
  yend   = YLPTODP(dc, yend);

  if(right < left) SWAP_INT(left, right);
  if(bottom < top) SWAP_INT(top, bottom);

  if((right - left == 1) || (bottom - top == 1))
     return TRUE;

  return dc->devfuncs->pPie(dc, left, top, right, bottom,
			    xstart, ystart, xend, yend);
}

/***********************************************************************
 *		SCL2DEV_Polygon
 */
BOOL SCL2DEV_Polygon(DC *dc, const POINT *pt, INT count)
{
  POINT *pt2;
  INT i;
  BOOL result;

  pt2 = malloc(count * sizeof(POINT));

  for (i = 0; i < count; i++) {
    pt2[i].x = dc->w.DCOrgX + XLPTODP(dc, pt[i].x);
    pt2[i].y = dc->w.DCOrgY + YLPTODP(dc, pt[i].y);
  }

  result = dc->devfuncs->pPolygon(dc, pt2, count);

  free(pt2);

  return result;
}

/***********************************************************************
 *		SCL2DEV_Polyline
 */

BOOL SCL2DEV_Polyline(DC *dc, const POINT *pt, INT count)
{
  POINT *pt2;
  INT i;
  BOOL result;

  pt2 = malloc(count * sizeof(POINT));

  for (i = 0; i < count; i++) {
    pt2[i].x = dc->w.DCOrgX + XLPTODP(dc, pt[i].x);
    pt2[i].y = dc->w.DCOrgY + YLPTODP(dc, pt[i].y);
  }

  result = dc->devfuncs->pPolyline(dc, pt2, count);

  free(pt2);

  return result;
}

/***********************************************************************
 *		SCL2DEV_PolyPolygon
 */
BOOL SCL2DEV_PolyPolygon(DC *dc, const POINT *pt, 
			 const INT *counts, UINT polygons)
{
  const POINT *p;
  POINT *pt2, *p2;
  INT i, j, total;
  BOOL result;

  total = 0;
  for(i = 0; i < polygons; i++) {
      total += counts[i];
  }

  pt2 = malloc(total * sizeof(POINT));

  p = pt;
  p2 = pt2;
  for(i = 0; i < polygons; i++) {
    for(j = 0; j < counts[i]; j++) {
      p2->x = dc->w.DCOrgX + XLPTODP(dc, p->x);
      p2->y = dc->w.DCOrgY + YLPTODP(dc, p->y);
      p++; p2++;
    }
  }

  result = dc->devfuncs->pPolyPolygon(dc, pt2, counts, polygons);

  free(pt2);

  return result;
}

/***********************************************************************
 *		SCL2DEV_PolyPolyline
 */
BOOL SCL2DEV_PolyPolyline(DC *dc, const POINT *pt,
			  const DWORD *counts, DWORD polylines)
{
  const POINT *p;
  POINT *pt2, *p2;
  INT i, j, total;
  BOOL result;

  total = 0;
  for(i = 0; i < polylines; i++) {
    total += counts[i];
  }

  pt2 = malloc(total * sizeof(POINT));

  p = pt;
  p2 = pt2;
  for(i = 0; i < polylines; i++) {
    for(j = 0; j < counts[i]; j++) {
      p2->x = dc->w.DCOrgX + XLPTODP(dc, p->x);
      p2->y = dc->w.DCOrgY + YLPTODP(dc, p->y);
      p++; p2++;
    }
  }

  result = dc->devfuncs->pPolyPolyline(dc, pt2, counts, polylines);
  
  free(pt2);

  return result;
}

/***********************************************************************
 *		SCL2DEV_Rectangle
 */
BOOL SCL2DEV_Rectangle(DC *dc, INT left, INT top, INT right, INT bottom)
{
  left   = XLPTODP(dc, left);
  top    = YLPTODP(dc, top);
  right  = XLPTODP(dc, right);
  bottom = YLPTODP(dc, bottom);

  if(right < left) SWAP_INT(left, right);
  if(bottom < top) SWAP_INT(top, bottom);

  return dc->devfuncs->pRectangle(dc, left, top, right, bottom);
}

/***********************************************************************
 *		SCL2DEV_RoundRect
 */
BOOL SCL2DEV_RoundRect(DC *dc, INT left, INT top, INT right,
		       INT bottom, INT ell_width, INT ell_height)
{
  left   = XLPTODP(dc, left);
  top    = YLPTODP(dc, top);
  right  = XLPTODP(dc, right);
  bottom = YLPTODP(dc, bottom);
  
  if(right < left) SWAP_INT(left, right);
  if(bottom < top) SWAP_INT(top, bottom);

  ell_width  = abs(ell_width * dc->vportExtX / dc->wndExtX);
  ell_height = abs(ell_height * dc->vportExtY / dc->wndExtY);

  return dc->devfuncs->pRoundRect(dc, left, top, right, bottom, ell_width, ell_height);
}

/***********************************************************************
 *		SCL2DEV_SetPixel
 */
COLORREF SCL2DEV_SetPixel(DC *dc, INT x, INT y, COLORREF color)
{
  x = dc->w.DCOrgX + XLPTODP(dc, x);
  y = dc->w.DCOrgY + YLPTODP(dc, y);
  
  return dc->devfuncs->pSetPixel(dc, x, y, color);
}

/***********************************************************************
 *		SCL2DEV_StretchBlt
 */
BOOL SCL2DEV_StretchBlt(DC *dcDst, INT xDst, INT yDst,
			INT widthDst, INT heightDst,
			DC *dcSrc, INT xSrc, INT ySrc,
			INT widthSrc, INT heightSrc, DWORD rop)
{
  BOOL useDst, useSrc;

  useDst = (((rop >> 1) & 0x550000) != (rop & 0x550000));
  if(useDst) {
    /* Compensate for off-by-one shifting for negative widths and heights */
    /* FIXME: This must be done before this function is called */
    if(widthDst < 0) xDst++;
    if(heightDst < 0) yDst++;

    xDst = dcDst->w.DCOrgX + XLPTODP(dcDst, xDst);
    yDst = dcDst->w.DCOrgY + YLPTODP(dcDst, yDst);

    widthDst  = MulDiv(widthDst, dcDst->vportExtX, dcDst->wndExtX);
    heightDst = MulDiv(heightDst, dcDst->vportExtY, dcDst->wndExtY);
  }

  useSrc = (((rop >> 2) & 0x330000) != (rop & 0x330000));
  if(useSrc) {
    /* Compensate for off-by-one shifting for negative widths and heights */
    /* FIXME: This must be done before this function is called */
    if(widthSrc < 0) xSrc++;
    if(heightSrc < 0) ySrc++;

    xSrc = dcSrc->w.DCOrgX + XLPTODP(dcSrc, xSrc);
    ySrc = dcSrc->w.DCOrgY + YLPTODP(dcSrc, ySrc);

    widthSrc  = widthSrc * dcSrc->vportExtX / dcSrc->wndExtX;
    heightSrc = heightSrc * dcSrc->vportExtY / dcSrc->wndExtY;
  }

  return dcDst->devfuncs->pStretchBlt(dcDst, xDst, yDst, widthDst, heightDst,
				      dcSrc, xSrc, ySrc, widthSrc, heightSrc,
				      rop);
}
