
#include <pcap.h>

#include <glib.h>
#include <direct.h>
#include <gtk/gtk.h>

#include "Menu.h"

//*****************************************************************************

#define MAXIMUM_PACKET_SIZE	1536

#define	CAPTURE_READ_TIMEOUT	250

//*****************************************************************************

typedef struct CaptureProperties{
	char EthDescription[256];
	GtkWidget * pEthDescriptionLabel;
	
	char IpAddr[HOST_ADDRESS_LEN];
	GtkWidget * pIpAddrLabel;
	char IpMask[HOST_ADDRESS_LEN];
	GtkWidget * pIpMaskLabel;

	GtkWidget * pPacketPerSecondLabel;
	GtkWidget * pPacketCountLabel;
} CaptureProperties_t;

typedef struct Acquisition{
	Status_e Stopped;
	Status_e InterfaceOpenned;
	Status_e Launched;
	Status_e HasBeenMade;

	HANDLE hLeaveThread;
	HANDLE pThread;
} Acquisition_t;

AppCnfCapture_t AppCnfCaptureSave = {0};
static AppCnfCapture_t AppCnfCaptureTemp = {0};

//*****************************************************************************

extern GtkWidget * pWindow;

//*****************************************************************************

static CaptureProperties_t Properties = {0};

static Acquisition_t Acquisition = {0};

//*****************************************************************************

static Status_e CaptureGetSelection(void);
static void OnEthernetSelectDevice(GtkWidget *widget, GdkEventButton *event);

static void EthReceive(HANDLE hSynchro);

static Status_e CapturePpsUpdate(void);

Status_e CaptureInterfaceGet(char * pDeviceName, char * pDesc, int DescSize, char * pIp, char * pMask);

//*****************************************************************************

void CaptureInit(void)
{
	Acquisition.Launched = FALSE;
	Acquisition.Stopped = TRUE;
	Acquisition.InterfaceOpenned = FALSE;
}

//*****************************************************************************

Status_e CaptureIsStarted(void)
{
	return Acquisition.Launched;
}

//*****************************************************************************

static CaptureStopInternal(void)
{
	Acquisition.Launched = FALSE;

	WaitForSingleObject(Acquisition.hLeaveThread, INFINITE);
	CloseHandle(Acquisition.hLeaveThread);
	Acquisition.hLeaveThread = NULL;

	CloseHandle(Acquisition.pThread);
	Acquisition.pThread = NULL;
}

//*****************************************************************************

Status_e CaptureStop(void)
{
	if(Acquisition.Launched == FALSE) return FALSE;

	CaptureStopInternal();

	return TRUE;
}

//*****************************************************************************

Status_e CaptureRecord(void)
{
	HANDLE hEthRcvSynchro;

	if(Acquisition.Stopped == FALSE)
	{
		CaptureStop();
	}

	hEthRcvSynchro = CreateSemaphore(NULL, 0, 1, NULL);
	if(hEthRcvSynchro == NULL)
	{
		return ERROR;
	}

	Acquisition.hLeaveThread = CreateSemaphore(NULL, 0, 1, NULL);
	if(Acquisition.hLeaveThread == NULL)
	{
		CloseHandle(hEthRcvSynchro);
		return ERROR;
	}

	Acquisition.Launched = TRUE;
	Acquisition.Stopped = FALSE;
	Acquisition.InterfaceOpenned = FALSE;

	/* Cration du thread utilis pour la connexion */
	Acquisition.pThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)EthReceive, hEthRcvSynchro, 0, NULL);
	if(Acquisition.pThread == NULL)
	{/* Impossible de crer le thread */
		CloseHandle(Acquisition.hLeaveThread);
		CloseHandle(hEthRcvSynchro);
		return ERROR;
	}

	/* Attendre le smaphore */
	WaitForSingleObject(hEthRcvSynchro, INFINITE);

	CloseHandle(hEthRcvSynchro);

	if(Acquisition.InterfaceOpenned == TRUE)
	{/* Indiquer que la capture est en cours */
		return TRUE;
	}
	else
	{/* Une erreur est survenue. L'indiquer  l'utilisateur */
		return ERROR;
	}
}

//*****************************************************************************

void CapturePropertiesChooseInterface(GtkWidget * pVBox, AppCnfCapture_t * p)
{
	gchar PCapErrorBuffer[PCAP_ERRBUF_SIZE];
	gchar * pString;
	GtkWidget * pCombo;
	gint DeviceFound = FALSE, DeviceId;
	pcap_if_t *alldevs;
	pcap_if_t *d;
	GtkWidget * pVBox2;
	GtkWidget *frame1;
	GtkWidget *alignment1;
	GtkWidget * pHBox2;
	GtkWidget * pLabel;

	FRAME_NEW(frame1, "Interface(s)");
	gtk_container_add (GTK_CONTAINER (pVBox), frame1);

	alignment1 = gtk_alignment_new (0.5, 0.5, 1, 1);
	gtk_container_add (GTK_CONTAINER (frame1), alignment1);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment1), 0, 0, 12, 0);

	pVBox2 = gtk_vbox_new(FALSE, 5);
	gtk_container_add (GTK_CONTAINER (alignment1), pVBox2);
	gtk_container_set_border_width(GTK_CONTAINER(pVBox2), 5);

	pHBox2 = gtk_hbox_new(FALSE, 5);
	gtk_container_add (GTK_CONTAINER (pVBox2), pHBox2);
	gtk_container_set_border_width(GTK_CONTAINER(pHBox2), 5);

	LABEL_NEW(pLabel, "Interface :");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	Properties.pEthDescriptionLabel = gtk_label_new(NULL);
	gtk_container_add (GTK_CONTAINER (pHBox2), Properties.pEthDescriptionLabel);

	/* Zone utilise par l'adresse IP et le masque */
	pHBox2 = gtk_hbox_new(FALSE, 5);
	gtk_container_add (GTK_CONTAINER (pVBox2), pHBox2);

	LABEL_NEW(pLabel, "IP address:");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	Properties.pIpAddrLabel = gtk_label_new(NULL);
	gtk_container_add (GTK_CONTAINER (pHBox2), Properties.pIpAddrLabel);

	LABEL_NEW(pLabel, "Mask:");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	Properties.pIpMaskLabel = gtk_label_new(NULL);
	gtk_container_add (GTK_CONTAINER (pHBox2), Properties.pIpMaskLabel);

	/* Zone utilise par le nombre de paquets/s */
	pHBox2 = gtk_hbox_new(FALSE, 5);
	gtk_container_add (GTK_CONTAINER (pVBox2), pHBox2);

	LABEL_NEW(pLabel, "Number of packets:");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	Properties.pPacketCountLabel = gtk_label_new(NULL);
	gtk_container_add (GTK_CONTAINER (pHBox2), Properties.pPacketCountLabel);

	LABEL_NEW(pLabel, "Packets/s:");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	Properties.pPacketPerSecondLabel = gtk_label_new(NULL);
	gtk_container_add (GTK_CONTAINER (pHBox2), Properties.pPacketPerSecondLabel);

	/* Cration de la combobox de slection des quipements */
	pCombo = gtk_combo_box_new_text();
	if(pcap_findalldevs(&alldevs, PCapErrorBuffer) == -1)
	{
		return;
	}
	for(DeviceId = 0, d=alldevs; d; d=d->next, DeviceId++)
	{/* Gnrer la liste des quipements */
		gtk_combo_box_append_text(GTK_COMBO_BOX(pCombo), d->name);
		if(strcmp(p->EthDeviceName, d->name) == 0)
		{
			gtk_combo_box_set_active(GTK_COMBO_BOX(pCombo), DeviceId);
			CaptureInterfaceGet(d->name, Properties.EthDescription, sizeof(Properties.EthDescription), Properties.IpAddr, Properties.IpMask);

			LABEL_SET(Properties.pEthDescriptionLabel, Properties.EthDescription);
			LABEL_SET(Properties.pIpAddrLabel, Properties.IpAddr);
			LABEL_SET(Properties.pIpMaskLabel, Properties.IpMask);

			DeviceFound = TRUE;

			CaptureStatCreate(AppCnfCaptureTemp.EthDeviceName);
		}
	}
	if(DeviceFound == FALSE)
	{/* Dfinir l'quipement par dfaut */
		gtk_combo_box_set_active(GTK_COMBO_BOX(pCombo), 0);
		pString = gtk_combo_box_get_active_text(GTK_COMBO_BOX(pCombo));
		if(pString != NULL) 
		{
			strcpy(p->EthDeviceName, pString);
			g_free(pString);
		}
	}
	pcap_freealldevs(alldevs);
	/* Ajouter la combo  la frame */
	gtk_box_pack_end (GTK_BOX (pVBox2), pCombo, TRUE, TRUE, 5);
	g_signal_connect(G_OBJECT(pCombo), "changed", G_CALLBACK(OnEthernetSelectDevice), NULL);
}

//*****************************************************************************

Status_e CaptureInterfaceGet(char * pDeviceName, char * pDesc, int DescSize, char * pIp, char * pMask)
{
	gchar PCapErrorBuffer[PCAP_ERRBUF_SIZE];
	pcap_if_t *alldevs;
	pcap_if_t *d;
	pcap_addr_t *a;
	Status_e DeviceFound = FALSE;

	if((pDeviceName == NULL)||(pDesc == NULL)||(pIp == NULL)||(pMask == NULL)) return ERROR;

	if(pcap_findalldevs(&alldevs, PCapErrorBuffer) == -1)
	{
		return ERROR;
	}

	for(d = alldevs; d; d=d->next)
	{/* Gnrer la liste des quipements */
		if(strcmp(pDeviceName, d->name) != 0) continue;
		
		DeviceFound = TRUE;
		if(d->description)
		{
			strncpy(pDesc, d->description, DescSize);
		}
		else
		{
			snprintf(pDesc, DescSize, "No description available");
		}
		*pIp = '\0';
		*pMask = '\0';
		for(a=d->addresses; a; a=a->next)
		{
			switch(a->addr->sa_family)
			{
				case AF_INET:
					if(a->addr)
					{
						char * p;
						p = inet_ntoa(((struct sockaddr_in *)a->addr)->sin_addr);
						strcpy(pIp, p);
					}
					if(a->netmask)
					{
						char * p;
						p = inet_ntoa(((struct sockaddr_in *)a->netmask)->sin_addr);
						strcpy(pMask, p);
					}
					break;
				case AF_INET6:
					break;
				default:
					break;
			}
		}
		break;
	}

	pcap_freealldevs(alldevs);
	return DeviceFound;
}

//*****************************************************************************

void CaptureProperties(void)
{
	gint Response;
	GtkWidget * pVBox;
	/*! Pointeur sur la boite de dialogue de slection de l'interface. */
	GtkWidget * pDialog;

	/* Rcupration de la configuration */
	memcpy(&AppCnfCaptureTemp, &AppCnfCaptureSave, sizeof(AppCnfCapture_t));

	pDialog = gtk_dialog_new_with_buttons ("Capture properties",
											GTK_WINDOW(pWindow),
											GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
											GTK_STOCK_OK,
											GTK_RESPONSE_OK,
											GTK_STOCK_CANCEL,
											GTK_RESPONSE_CANCEL,
											NULL);

	gtk_window_set_position(GTK_WINDOW(pDialog), GTK_WIN_POS_CENTER);

	gtk_dialog_set_default_response(GTK_DIALOG(pDialog), GTK_RESPONSE_OK);

	gtk_window_set_default_size(GTK_WINDOW(pDialog), 100, 100);

	pVBox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(pVBox), 5);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(pDialog)->vbox), pVBox, FALSE, FALSE, 0);

	/* Slection de l'interface */
	CapturePropertiesChooseInterface(pVBox, &AppCnfCaptureTemp);

	gtk_window_resize(GTK_WINDOW(pDialog), 10, 10);

	gtk_widget_show_all(pVBox);
	
	do
	{
		Response = gtk_dialog_run(GTK_DIALOG(pDialog));
		if(Response == GTK_RESPONSE_OK)
		{/* Mmoriser la configuration */
			if(CaptureGetSelection() == ERROR)
			{/* Configuration invalide */
				continue;
			}
			else
			{
				break;
			}
		}
		else break;
	}
	while(1);

	CaptureStatDestroy();

	gtk_widget_destroy(pDialog);
}

//*****************************************************************************

static void OnEthernetSelectDevice(GtkWidget *widget, GdkEventButton *event)
{
	gchar * pString;
	Status_e Status;
	
	pString = gtk_combo_box_get_active_text(GTK_COMBO_BOX(widget));
	if(pString == NULL) return;
	
	strcpy(AppCnfCaptureTemp.EthDeviceName, pString);
	g_free(pString);

	Status = CaptureInterfaceGet(AppCnfCaptureTemp.EthDeviceName,
								Properties.EthDescription, sizeof(Properties.EthDescription),
								Properties.IpAddr,
								Properties.IpMask);
	if(Status != TRUE)
	{
		return;
	}

	/* Changer la description de l'interface */
	LABEL_SET(Properties.pEthDescriptionLabel, Properties.EthDescription);
	LABEL_SET(Properties.pIpAddrLabel, Properties.IpAddr);
	LABEL_SET(Properties.pIpMaskLabel, Properties.IpMask);

	CaptureStatCreate(AppCnfCaptureTemp.EthDeviceName);
}

//*****************************************************************************

static Status_e CaptureGetSelection(void)
{
	strcpy(AppCnfCaptureSave.EthDeviceName, AppCnfCaptureTemp.EthDeviceName);

	return TRUE;
}

//*****************************************************************************

static void EthReceiveLeave(pcap_t * pFile)
{
	if(pFile != NULL)
	{/* Fermer l'interface pcap */
		pcap_close(pFile);
	}

	/* Se dconnecter */
	Acquisition.Stopped = TRUE;
	Acquisition.Launched = FALSE;

	/* Donner le smaphore */
	ReleaseSemaphore(Acquisition.hLeaveThread, 1, NULL);
}

//*****************************************************************************

static void EthReceive(HANDLE hSynchro)
{
	Status_e Status;
	struct pcap_pkthdr * pHeader;
	guchar * pData;
	pcap_t * pFile;
	gchar ErrorBuffer[PCAP_ERRBUF_SIZE];

	/* Ouverture de l'interface */
	pFile = pcap_open_live(AppCnfCaptureSave.EthDeviceName, 
										MAXIMUM_PACKET_SIZE, 
										PCAP_OPENFLAG_PROMISCUOUS,
										CAPTURE_READ_TIMEOUT, 
										ErrorBuffer);
	if(pFile == NULL)
	{/* Impossible d'ouvrir le priphrique */
		Acquisition.InterfaceOpenned = FALSE;
		/* Donner le smaphore */
		ReleaseSemaphore(hSynchro, 1, NULL);
		/* Terminer le thread */
		EthReceiveLeave(NULL);
		return;
	}
	else
	{/* Interface ouverte */
		Acquisition.InterfaceOpenned = TRUE;
		/* Donner le smaphore */
		ReleaseSemaphore(hSynchro, 1, NULL);
	}

	while(Acquisition.Launched == TRUE)
	{
		Status = pcap_next_ex(pFile, &pHeader, &pData);
		if(Status <= ERROR)
		{/* Erreur durant la lecture du paquet */
			goto EthReceiveError;
		}
		else if(Status == FALSE)
		{/* Timeout dpass, recommencer le test */
			continue;
		}

		/* Do something... */
	}

	/* Se dconnecter */
	EthReceiveLeave(pFile);

	return;

EthReceiveError:
	/* Se dconnecter */
	EthReceiveLeave(pFile);
}

//*****************************************************************************
