/*!
 * \file CaptureStat.c
 * \brief Fichier source de gestion des statistiques associes  une interface rseau.
 */

#include <pcap.h>

#include "main.h"
#include "Menu.h"

//*****************************************************************************

#define	CAPTURE_PPS_TIMEOUT	500

/*! Taille maximum d'une trame ethernet. */
#define CAPTURE_STAT_PACKET_SIZE	1536

//*****************************************************************************

typedef struct InfoProperties{
	GtkWidget * pPacketPerSecondLabel;
	GtkWidget * pPacketCountLabel;
	GtkWidget * pCapturePacketCount;
} InfoProperties_t;

//*****************************************************************************

typedef struct CaptureStat{
	Status_e Launched;
	Status_e Stopped;
	Status_e InterfaceOpenned;

	char InterfaceName[256];
	HANDLE hLeaveThread;
	HANDLE pThread;
}CaptureStat_t;

//*****************************************************************************

static gulong CapturePpsPacketCount = 0;
static gdouble CapturePpsPacketPerSecond = 0;

static gdouble InfoStartTime, InfoTimeStamp;
static gulong InfoPpsPacketCount = 0;
static gulong InfoCurrentCount = 0;
static gdouble InfoPpsPacketPerSecond = 0;

static InfoProperties_t Properties = {0};

static GtkWidget * pInfoWnd = NULL;

static CaptureStat_t CaptureStat = {0};

extern AppCnfCapture_t AppCnfCaptureSave;

extern GtkWidget * pWindow;

//*****************************************************************************

static void CapturePpsReceive(HANDLE hSynchro);
static Status_e CapturePpsUpdate(void);
static Status_e CaptureStatInfoPpsUpdate(void);
static void StopInternal(void);

void CaptureStatInfoDisable(void);
void CaptureStatInfoHide(void);

//*****************************************************************************

Status_e CaptureStatCreate(char * DeviceName)
{
	HANDLE hEthRcvSynchro;

	CaptureStatDestroy();

	strncpy(CaptureStat.InterfaceName, DeviceName, sizeof(CaptureStat.InterfaceName) - 1);

	hEthRcvSynchro = CreateSemaphore(NULL, 0, 1, NULL);
	if(hEthRcvSynchro == NULL)
	{
		return ERROR;
	}

	CaptureStat.hLeaveThread = CreateSemaphore(NULL, 0, 1, NULL);
	if(CaptureStat.hLeaveThread == NULL)
	{
		CloseHandle(hEthRcvSynchro);
		return ERROR;
	}

	CapturePpsPacketCount = 0;
	CapturePpsPacketPerSecond = 0;
	CaptureStat.Launched = TRUE;
	CaptureStat.Stopped = FALSE;
	CaptureStat.InterfaceOpenned = FALSE;

	/* Cration du thread utilis pour la connexion */
	CaptureStat.pThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)CapturePpsReceive, hEthRcvSynchro, 0, NULL);
	if(CaptureStat.pThread == NULL)
	{/* Impossible de crer le thread */
		CaptureStat.Launched = FALSE;
		CaptureStat.Stopped = TRUE;
		CloseHandle(CaptureStat.hLeaveThread);
		CaptureStat.hLeaveThread = NULL;
		CloseHandle(hEthRcvSynchro);
		return ERROR;
	}

	/* Attendre le smaphore */
	WaitForSingleObject(hEthRcvSynchro, INFINITE);

	CloseHandle(hEthRcvSynchro);

	if(CaptureStat.InterfaceOpenned == FALSE)
	{
		return ERROR;
	}

	return TRUE;
}

//*****************************************************************************

Status_e CaptureStatDestroy(void)
{
	StopInternal();	

	return TRUE;
}

//*****************************************************************************

Status_e CaptureStatGet(gulong * pPacketCount, gdouble * pPps)
{
	/* Mettre fin au timer */
	if(CaptureStat.Launched == FALSE) return FALSE;

	*pPacketCount = CapturePpsPacketCount;
	*pPps = CapturePpsPacketPerSecond;

	return TRUE;
}

//*****************************************************************************

Status_e CaptureStatInfoCreate(void)
{
	InfoPpsPacketCount = 0;
	InfoCurrentCount = 0;
	InfoPpsPacketPerSecond = 0;

	InfoStartTime = (gdouble)clock ();

	return TRUE;
}

//*****************************************************************************

void CaptureStatInfoDestroy(void)
{
	/* Suppression de la fentre */
	CaptureStatInfoHide();

	/* RAZ des variables */
	InfoPpsPacketCount = 0;
	InfoCurrentCount = 0;
	InfoPpsPacketPerSecond = 0;
	InfoStartTime = 0;
}

//*****************************************************************************

void CaptureStatInfoReceivePacket(void)
{
	InfoPpsPacketCount++;
	InfoCurrentCount++;
}

//*****************************************************************************

void CaptureStatInfoUpdate(void)
{
	gdouble Time;

	Time = (gdouble)clock();
	if((Time - InfoStartTime) >= (gdouble)CLK_TCK)
	{
		InfoTimeStamp = (Time - InfoStartTime) / (gdouble)CLK_TCK;
		InfoPpsPacketPerSecond = ((gdouble)InfoCurrentCount) / InfoTimeStamp;
		InfoCurrentCount = 0;
		InfoStartTime = Time;
	}
}

//*****************************************************************************

Status_e CaptureStatInfoShow(void)
{
//	GdkColor Color;
	GtkWidget * pVBox;
	GtkWidget * pVBox2;
	GtkWidget *frame1;
	GtkWidget *alignment1;
	GtkWidget * pHBox2;
	GtkWidget * pLabel;
	char IpAddr[HOST_ADDRESS_LEN];
	char IpMask[HOST_ADDRESS_LEN];
	char Description[64];

	if(pInfoWnd != NULL) return FALSE;

	/* Obtenir les paramtres de l'interface */
	if(CaptureInterfaceGet(AppCnfCaptureSave.EthDeviceName, Description, sizeof(Description), IpAddr, IpMask) != TRUE)
	{
		return FALSE;
	}

	/* Cration de la fentre */
	pInfoWnd = gtk_window_new(GTK_WINDOW_TOPLEVEL);

	gtk_window_set_transient_for(GTK_WINDOW(pInfoWnd), GTK_WINDOW(pWindow));
	gtk_window_set_title(GTK_WINDOW(pInfoWnd), "");
	gtk_window_set_destroy_with_parent(GTK_WINDOW(pInfoWnd), TRUE);
	gtk_window_set_default_size(GTK_WINDOW(pInfoWnd), 10, 10);
	gtk_window_set_type_hint(GTK_WINDOW(pInfoWnd), GDK_WINDOW_TYPE_HINT_SPLASHSCREEN);

	pVBox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(pInfoWnd), pVBox);

	/* Cration des messages */
	FRAME_NEW(frame1, "Interface(s)");
	gtk_container_add (GTK_CONTAINER (pVBox), frame1);

	alignment1 = gtk_alignment_new (0.5, 0.5, 1, 1);
	gtk_container_add (GTK_CONTAINER (frame1), alignment1);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment1), 0, 0, 5, 5);

	pVBox2 = gtk_vbox_new(FALSE, 5);
	gtk_container_add (GTK_CONTAINER (alignment1), pVBox2);
	gtk_container_set_border_width(GTK_CONTAINER(pVBox2), 5);

	pHBox2 = gtk_hbox_new(FALSE, 5);
	gtk_container_add (GTK_CONTAINER (pVBox2), pHBox2);
	gtk_container_set_border_width(GTK_CONTAINER(pHBox2), 5);

	LABEL_NEW(pLabel, "Interface :");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	pLabel = gtk_label_new(Description);
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	/* Zone utilise par l'adresse IP et le masque */
	pHBox2 = gtk_hbox_new(FALSE, 5);
	gtk_container_add (GTK_CONTAINER (pVBox2), pHBox2);

	LABEL_NEW(pLabel, "IP address:");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	pLabel = gtk_label_new(IpAddr);
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	LABEL_NEW(pLabel, "Mask:");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	pLabel = gtk_label_new(IpMask);
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	/* Zone utilise par le nombre de paquets/s */
	pHBox2 = gtk_hbox_new(FALSE, 5);
	gtk_container_add (GTK_CONTAINER (pVBox2), pHBox2);

	LABEL_NEW(pLabel, "Number of packets:");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	Properties.pPacketCountLabel = gtk_label_new(NULL);
	gtk_container_add (GTK_CONTAINER (pHBox2), Properties.pPacketCountLabel);

	LABEL_NEW(pLabel, "Packets/s:");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	Properties.pPacketPerSecondLabel = gtk_label_new(NULL);
	gtk_container_add (GTK_CONTAINER (pHBox2), Properties.pPacketPerSecondLabel);

	/* Zone utilise pour afficher le nombre des paquets mmoriss */
	frame1 = gtk_frame_new("Capture");
	gtk_container_add (GTK_CONTAINER (pVBox), frame1);

	alignment1 = gtk_alignment_new (0.5, 0.5, 1, 1);
	gtk_container_add (GTK_CONTAINER (frame1), alignment1);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment1), 0, 0, 5, 5);

	pHBox2 = gtk_hbox_new(FALSE, 5);
	gtk_container_add (GTK_CONTAINER (alignment1), pHBox2);
	gtk_container_set_border_width(GTK_CONTAINER(pHBox2), 5);

	pLabel = gtk_label_new("Packet count : ");
	gtk_container_add (GTK_CONTAINER (pHBox2), pLabel);

	Properties.pCapturePacketCount = gtk_label_new(NULL);
	gtk_container_add (GTK_CONTAINER (pHBox2), Properties.pCapturePacketCount);

	gtk_widget_show_all(pInfoWnd);

	g_timeout_add_full(G_PRIORITY_DEFAULT_IDLE,1000, (GSourceFunc)CaptureStatInfoPpsUpdate, NULL, NULL);

	return TRUE;
}

//*****************************************************************************

void CaptureStatInfoHide(void)
{
	if(pInfoWnd == NULL) return;
	gtk_widget_destroy(pInfoWnd);
	pInfoWnd = NULL;
}

//*****************************************************************************

static void StopInternal(void)
{
	if(CaptureStat.hLeaveThread == NULL) return;

	/* Stopper l'acquisition en cours */
	CaptureStat.Launched = FALSE;
	/* Attendre que le thread se termine puis supprimer le smaphore */
	WaitForSingleObject(CaptureStat.hLeaveThread, INFINITE);
	CloseHandle(CaptureStat.hLeaveThread);
	CaptureStat.hLeaveThread = NULL;

	CloseHandle(CaptureStat.pThread);
	CaptureStat.pThread = NULL;
}

//*****************************************************************************

static void PpsReceiveLeave(pcap_t * pFile)
{
	if(pFile != NULL)
	{/* Fermer l'interface pcap */
		pcap_close(pFile);
	}

	/* Se dconnecter */
	CaptureStat.Stopped = TRUE;
	CaptureStat.Launched = FALSE;

	/* Donner le smaphore */
	ReleaseSemaphore(CaptureStat.hLeaveThread, 1, NULL);
}

//*****************************************************************************

static void CapturePpsReceive(HANDLE hSynchro)
{
	Status_e Status;
	struct pcap_pkthdr * pHeader;
	guchar * pData;
	gulong CurrentCount = 0;
	gdouble StartTime, CurrentTime, TimeStamp;
	pcap_t * pFile;
	gchar CapturePpsErrorBuffer[PCAP_ERRBUF_SIZE];

	StartTime = (gdouble)clock ();

	/* Ouverture de l'interface */
	pFile = pcap_open_live(CaptureStat.InterfaceName, 
										CAPTURE_STAT_PACKET_SIZE, 
										PCAP_OPENFLAG_PROMISCUOUS,
										CAPTURE_PPS_TIMEOUT, 
										CapturePpsErrorBuffer);
	if(pFile == NULL)
	{/* Impossible d'ouvrir le priphrique */
		CaptureStat.InterfaceOpenned = FALSE;
		/* Donner le smaphore */
		ReleaseSemaphore(hSynchro, 1, NULL);
		/* Terminer le thread */
		PpsReceiveLeave(NULL);
		return;
	}
	else
	{
		CaptureStat.InterfaceOpenned = TRUE;
		/* Donner le smaphore */
		ReleaseSemaphore(hSynchro, 1, NULL);
	}

	/* Attente de la cible */
	while(CaptureStat.Launched == TRUE)
	{
		Status = pcap_next_ex(pFile, &pHeader, &pData);
		if(Status <= ERROR)
		{/* Erreur durant la lecture du paquet */
			goto CapturePpsReceiveError;
		}

		CurrentTime = (gdouble)clock();
		if((CurrentTime - StartTime) >= (gdouble)CLK_TCK)
		{
			TimeStamp = (CurrentTime - StartTime) / (gdouble)CLK_TCK;
			CapturePpsPacketPerSecond = ((gdouble)CurrentCount) / TimeStamp;
			CurrentCount = 0;
			StartTime = CurrentTime;
		}

		if(Status == FALSE)
		{/* Timeout dpass, recommencer le test */
			continue;
		}
		CapturePpsPacketCount++;
		CurrentCount++;
	}

	PpsReceiveLeave(pFile);

	return;

CapturePpsReceiveError:
	PpsReceiveLeave(pFile);
}

//*****************************************************************************

static Status_e CaptureStatInfoPpsUpdate(void)
{
	char Tab[64];

	/* Mettre fin au timer */
	if(pInfoWnd == NULL) return FALSE;

	CaptureStatInfoUpdate();

	snprintf(Tab, sizeof(Tab), "%ld", InfoPpsPacketCount);
	LABEL_SET(Properties.pPacketCountLabel, Tab);
	snprintf(Tab, sizeof(Tab), "%0.2f", InfoPpsPacketPerSecond);
	LABEL_SET(Properties.pPacketPerSecondLabel, Tab);

	snprintf(Tab, sizeof(Tab), "%ld", 0);
	LABEL_SET(Properties.pCapturePacketCount, Tab);
	
	return TRUE;
}

//*****************************************************************************
