/* ZeroMQ test server
 *
 */

#include <boost/thread.hpp>
#include <boost/thread/condition.hpp>
#include <boost/thread/lock_guard.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/function.hpp>

#include <zmq.h>

#include <iostream>
#include <deque>

// ==========================================================================
struct Message
{
	std::deque<zmq_msg_t>   payload;
};

class ActiveQueue
{
protected:
    typedef std::deque<Message*>                             message_queue_t;
    typedef boost::function<void (Message&)>                      consumer_t;

public:
	ActiveQueue (consumer_t consumer) : m_consumer(consumer) {;}

	bool  put (Message& message)
    {
    	boost::unique_lock<boost::mutex> guard(m_mutex);
        if (m_queue.empty())
        {
    		m_cond.notify_all();
        }
    	m_queue.push_back(&message);
		return true;
    }

	Message*  get ()
    {
    	boost::unique_lock<boost::mutex> guard(m_mutex);
        while (m_queue.empty())
        {
    		m_cond.wait(guard);
        }
        Message* retval = m_queue.front();
        m_queue.pop_front();
        return retval;
    }

    bool start (unsigned int threads)
    {
		for (unsigned int i = 0; i < threads; i++)
		{
			m_threadGroup.create_thread(boost::bind(&ActiveQueue::svc, this));
		}
		return true;
    }

protected:
    void svc ()
    {
		while (true)
		{
			m_consumer(*get());
		}
    }
    consumer_t                                                    m_consumer;
    boost::thread_group                                        m_threadGroup;

    boost::mutex                                                     m_mutex;
	boost::condition_variable                                         m_cond;
	message_queue_t                                                  m_queue;
};

// ##########################################################################
static void*                                             zmq_server_context = NULL;
static void*                                              zmq_server_socket = NULL;
static boost::mutex                                          zmq_send_mutex;


static bool      create_zmq ();
static Message* receive_zmq ();
static void        send_zmq (Message& messsage);

int main ()
{
	if (create_zmq())
	{
		ActiveQueue  queue(send_zmq);
		if (queue.start(2U))
		{
			while (true)
			{
				Message*    message = receive_zmq();
				if (message == NULL)
				{
					break;
				}
				queue.put(*message);
			}

		}
	}
	if (zmq_server_context != NULL)
	{
		zmq_ctx_destroy(zmq_server_context);
		zmq_server_context = NULL;
	}
	return 0;
}



static bool create_zmq ()
{
   	if ((zmq_server_context = zmq_ctx_new()) == NULL)
	{
   		return false;
	}
	if (zmq_ctx_set(zmq_server_context, ZMQ_IO_THREADS, 1) != 0)
	{
		return false;
	}
	if ((zmq_server_socket = zmq_socket(zmq_server_context, ZMQ_ROUTER)) == NULL)
	{
		return false;
	}
	int       linger_period = 100;
	zmq_setsockopt(zmq_server_socket, ZMQ_LINGER, &linger_period, sizeof(linger_period));

	if (zmq_bind(zmq_server_socket, "tcp://*:23510") != 0)
	{
		return false;
	}
	return true;
}

static Message* receive_zmq ()
{
	std::auto_ptr<Message>  retval(new Message);
	zmq_msg_t              zmq_msg;

	zmq_msg_init(&zmq_msg);
	while (true)
	{
		// Flags are ignored by a router.
		int rc = zmq_msg_recv(&zmq_msg, zmq_server_socket, 0);
		if (rc == -1)
		{
			return NULL;
		}
		bool more = zmq_msg_more(&zmq_msg);

		retval->payload.push_back(zmq_msg_t());
		zmq_msg_init(&retval->payload.back());
		if (zmq_msg_move(&retval->payload.back(), &zmq_msg) == -1)
		{
			return NULL;
		}
		if (! more)
		{
			return retval.release();
		}
	}
}


static void send_zmq (Message& messsage)
{
	boost::unique_lock<boost::mutex> guard(zmq_send_mutex);
	while (! messsage.payload.empty())
	{
		int         more = (messsage.payload.size() == 1) ? 0 : ZMQ_SNDMORE;
		if (zmq_msg_send(&messsage.payload.front(), zmq_server_socket, more) == -1)
		{
			break;
		}
		messsage.payload.pop_front();
	}
	delete &messsage;
}
